package com.liveperson.messaging.wm.impl

import android.content.ContentResolver
import android.net.Uri
import com.liveperson.messaging.utils.getOrDefault
import com.liveperson.messaging.utils.readBoolean
import com.liveperson.messaging.utils.readFloat
import com.liveperson.messaging.utils.readInt
import com.liveperson.messaging.utils.readLong
import com.liveperson.messaging.utils.readString
import com.liveperson.messaging.wm.core.ValueReader
import com.liveperson.messaging.wm.core.exceptions.throwTypeErrorOf
import org.json.JSONArray

/**
 * Default implementation of ValueReader that requests
 * values from multi-process shared preferences using ContentResolver API.
 *
 * @param contentResolver ContentResolver associated with ContentProvider
 * that uses multi-process shared preferences to store values.
 * @param uriGenerator generator for welcome message provider uri
 * to receive values
 *
 * @see ValueReader for documentation of implemented methods.
 * @throws ClassCastException if requested type is not matched with stored type
 */
internal class ValueReaderImpl constructor(
    private val contentResolver: ContentResolver,
    private val uriGenerator: (key: String, type: String) -> Uri
) : ValueReader {
    override fun <T : Any?> readValue(key: String, default: T): T {
        val requestedType = default?.receiveInternalType() ?: COMMON_TYPE
        val cursor = contentResolver.query(uriGenerator(key, requestedType), null, null, null, null)
        return cursor.getOrDefault(default) { currentCursor ->
            val value: Any = when (default) {
                is Boolean -> currentCursor.readBoolean(defaultValue = false)
                is Int -> currentCursor.readInt(defaultValue = 0)
                is Long -> currentCursor.readLong(defaultValue = 0)
                is Float -> currentCursor.readFloat(defaultValue = 0f)
                is String -> currentCursor.readString(defaultValue = "")
                is Set<*> -> runCatching { JSONArray(currentCursor.readString(defaultValue = "[]")) }
                    .getOrDefault(JSONArray())
                    .run { (0 until length()).map { index -> get(index) } }
                    .map { it.toString() }
                    .toSet()
                else -> throwTypeErrorOf(default)
            }
            value as T
        }
    }

    override fun getAllValues(): MutableMap<String, *> = mutableMapOf<String, Any>()

    override fun containsValue(key: String): Boolean {
        val cursor = contentResolver.query(uriGenerator(key, COMMON_TYPE), null, null, null, null)
        return cursor?.use { it.moveToFirst() && it.count > 0 } ?: false
    }

}