package com.liveperson.messaging.commands.pusher;

import android.text.TextUtils;

import androidx.annotation.Nullable;

import com.liveperson.infra.Command;
import com.liveperson.infra.ICallback;
import com.liveperson.infra.InternetConnectionService;
import com.liveperson.infra.PushUnregisterType;
import com.liveperson.infra.auth.LPAuthenticationType;
import com.liveperson.infra.errors.ErrorCode;
import com.liveperson.infra.log.LPLog;
import com.liveperson.infra.managers.PreferenceManager;
import com.liveperson.infra.network.http.requests.UnregisterPushRequest;
import com.liveperson.infra.preferences.PushMessagePreferences;
import com.liveperson.messaging.Messaging;
import com.liveperson.messaging.model.AmsAccount;
import com.liveperson.messaging.model.SynchronizedInternetConnectionCallback;

import java.util.List;
import java.util.Objects;

import static com.liveperson.infra.managers.PreferenceManager.APP_ID_PREFERENCE_KEY;
import static com.liveperson.infra.managers.PreferenceManager.CONSUMER_UNREGISTER_PUSHER_PREFERENCE_KEY;
import static com.liveperson.infra.managers.PreferenceManager.IS_PUSHER_REGISTERED_PREFERENCE_KEY;

/**
 * Command to unregister from push notifications for a given brandId.
 * <p>
 * Created by nirni on 1/27/16.
 */
public class UnregisterPusherCommand implements Command {

    private static final String TAG = "UnregisterPusherCommand";

    private static final String PUSHER_UNREGISTER_URL = "https://%s/api/account/%s/device/unregister?v=2.0";
    private static final String UN_AUTH_PUSHER_UNREGISTER_URL = "https://%s/api/account/%s/device/unregister";
    private static final int MAX_RETRY_NUMBER = 3;
    private final Messaging mMessagingController;

    private String mBrandId;
    private String mAppId;
    private String pusherURL;
    private String consumerId;
    @Nullable
    private ICallback<Void, Exception> listener;
    private boolean isFullLogout;
    private PushUnregisterType unregisterType;
    private int logoutRetry;
    private PusherHelper pusherHelper;

    private ICallback<Void, Exception> requestCallback = new ICallback<Void, Exception>() {
        @Override
        public void onSuccess(Void value) {
            PreferenceManager.getInstance().setBooleanValue(IS_PUSHER_REGISTERED_PREFERENCE_KEY, mBrandId, false);
            PreferenceManager.getInstance().remove(CONSUMER_UNREGISTER_PUSHER_PREFERENCE_KEY, mBrandId);
            if (listener != null) {
                listener.onSuccess(value);
            }
        }

        @Override
        public void onError(Exception exception) {
            if (isFullLogout && logoutRetry < MAX_RETRY_NUMBER) {
                logoutRetry++;
                LPLog.INSTANCE.w(TAG, "Unregister pusher retry #: " + logoutRetry, exception);
                runUnregisterPush();
            } else {
                LPLog.INSTANCE.e(TAG, ErrorCode.ERR_00000148, "Failed to unregister pusher", exception);
                if (listener != null) {
                    listener.onError(exception);
                }
            }
        }
    };

    public UnregisterPusherCommand(Messaging messagingController, String brandId, String consumerId) {
        this(messagingController, brandId, "", null, null, false);
        this.consumerId = consumerId;
    }

    public UnregisterPusherCommand(Messaging messagingController, String brandId, String appId, PushUnregisterType type, @Nullable ICallback<Void, Exception> listener, boolean isFullLogout) {
        mMessagingController = messagingController;
        mBrandId = brandId;
        mAppId = appId;
        this.listener = listener;
        this.isFullLogout = isFullLogout;
        unregisterType = type;
        logoutRetry = 0;
    }

    @Override
    public void execute() {
        pusherHelper = new PusherHelper(mBrandId, mMessagingController);
        if (!PreferenceManager.getInstance().getBooleanValue(IS_PUSHER_REGISTERED_PREFERENCE_KEY, mBrandId, false)
                || TextUtils.isEmpty(PushMessagePreferences.INSTANCE.getPushPlatform())
                || (unregisterType == PushUnregisterType.NONE && Objects.equals(pusherHelper.getAuthType(), LPAuthenticationType.AUTH.name()))) {
            notifySuccess();
            return;
        }

        LPLog.INSTANCE.i(TAG, "execute unregister for brandId " + mBrandId);

        // Create pusher unregister URL
        String pusherDomain = pusherHelper.getPusherDomain();
        if (TextUtils.isEmpty(pusherDomain)) {
            LPLog.INSTANCE.w(TAG, "pusherDomain does not exists. Quit unregister push");
            notifySuccess();
            return;
        }

        if (TextUtils.isEmpty(mAppId) && !TextUtils.isEmpty(consumerId)) {
            mAppId = PreferenceManager.getInstance().getStringValue(APP_ID_PREFERENCE_KEY, mBrandId, "");
            PreferenceManager.getInstance().setStringValue(CONSUMER_UNREGISTER_PUSHER_PREFERENCE_KEY, mBrandId, consumerId);
        } else {
            PreferenceManager.getInstance().setStringValue(APP_ID_PREFERENCE_KEY, mBrandId, mAppId);
        }

        // Use older version of unregister api to remove UnAuth user from pusher database on sign out.
        AmsAccount account = mMessagingController.mAccountsController.getAccount(mBrandId);
        if (isFullLogout && account != null && !account.isAuthenticated()) {
            pusherURL = String.format(UN_AUTH_PUSHER_UNREGISTER_URL, pusherDomain, mBrandId);
        } else {
            pusherURL = String.format(PUSHER_UNREGISTER_URL, pusherDomain, mBrandId);
        }

        if (TextUtils.isEmpty(consumerId)) {
            // Get consumerId from memory
            consumerId = pusherHelper.getConsumerId();
        }

        // If consumerId is not provided, get it for active user
        if (TextUtils.isEmpty(consumerId)) {
            LPLog.INSTANCE.w(TAG, "onResult: Cannot get consumerId. Quit unregister push");
            if (listener != null) {
                if (isFullLogout) {
                    listener.onSuccess(null); //Don't block logout.
                } else  {
                    listener.onError(new IllegalStateException("Failed to register pusher. Missing consumer id."));
                }
            }
        } else {
            runUnregisterPush();
        }
    }

    private void notifySuccess() {
        if (listener != null) {
            listener.onSuccess(null);
        }
    }

    private void runUnregisterPush() {
        final List<String> certificates = pusherHelper.getCertificatePinningKeys();
        String platform = PushMessagePreferences.INSTANCE.getPushPlatform();
        if (isFullLogout) { // Run immediately when logout.
            LPLog.INSTANCE.d(TAG, "run: Unregister push immediately");
            if (InternetConnectionService.isNetworkAvailable()) {
                // If available execute immediately
                new UnregisterPushRequest(pusherURL, consumerId, mAppId, platform, pusherHelper.getAuthType(), certificates)
                        .setUnregisterType(unregisterType).setCallback(requestCallback).execute();
            } else {
                if (listener != null) {
                    listener.onError(new Exception("No network available"));
                }
            }
        } else {
            new SynchronizedInternetConnectionCallback(() -> {
	            LPLog.INSTANCE.d(TAG, "run: Unregister push for consumerId: " + LPLog.INSTANCE.mask(consumerId));
                new UnregisterPushRequest(pusherURL, consumerId, mAppId, platform, pusherHelper.getAuthType(), certificates)
                        .setUnregisterType(unregisterType).setCallback(requestCallback).execute();
            }).execute();
        }
    }
}
