package com.liveperson.messaging.network.socket.requests

import com.liveperson.api.request.BaseAMSSocketRequest
import com.liveperson.api.request.ReqBody.StringResp
import com.liveperson.api.request.UpdateConversationField
import com.liveperson.infra.ICallback
import com.liveperson.infra.errors.ErrorCode
import com.liveperson.infra.log.LPLog
import com.liveperson.infra.network.socket.BaseResponseHandler
import org.json.JSONObject

/**
 * Class that represents body of the request to update conversation's active Co-Browse
 * dialog when main dialog was resolved by consumer without ending of Co-browse session.
 * This clas is a decorator of [UpdateConversationField] request.
 * @param socketUrl url of current active socket
 * @param cobrowseDialogId id of the active Co-Browse call
 * @param conversationId id of previously active conversation
 * @param coBrowseMetadata metadata to update active cobrowse call.
 * @param callback request callback.
 */
class CloseCobrowseDialogRequest(
    socketUrl: String,
    private val cobrowseDialogId: String,
    private val conversationId: String,
    private val coBrowseMetadata: JSONObject,
    private var callback: ICallback<Int, Exception>? = null
) : BaseAMSSocketRequest<StringResp, CloseDialogRequest>(socketUrl) {

    companion object {
        private const val TAG = "CloseCobrowseDialogRequest"
    }

    override fun getData(): String {
        val request = UpdateConversationField(
            cobrowseDialogId,
            conversationId,
            coBrowseMetadata
        )
        return request.toJsonString(requestId);
    }

    override fun getRequestName(): String = TAG

    override fun getResponseHandler(): BaseResponseHandler<StringResp, CloseDialogRequest> {
        return object : BaseResponseHandler<StringResp, CloseDialogRequest>() {

            override fun parse(jsonObject: JSONObject?): StringResp {
                return StringResp(jsonObject)
            }

            override fun getAPIResponseType(): String? {
                return null
            }

            override fun handle(data: StringResp): Boolean {

                //send update only when notification will arrive.
                if (data.code in 200..299) {
                    LPLog.d(TAG, "Succeeded, close dialog response code: " + data.code)
                    callback?.onSuccess(data.code)
                } else {
                    LPLog.e(TAG, ErrorCode.ERR_000000C2, "Error, close dialog response code: " + data.code)
                    callback?.onError(Exception(data.code.toString()))
                }
                return true
            }
        }

    }
}