package com.liveperson.messaging.offline

import android.content.ContentValues
import androidx.annotation.WorkerThread
import com.liveperson.api.response.model.MultiDialog
import com.liveperson.api.response.types.DialogState
import com.liveperson.api.response.types.DialogType
import com.liveperson.api.response.types.TTRType
import com.liveperson.infra.database.BaseDBRepository
import com.liveperson.infra.database.tables.DialogsTable
import com.liveperson.infra.log.LPLog.d
import com.liveperson.messaging.cache.DialogsCacheProvider
import com.liveperson.messaging.model.Dialog
import com.liveperson.messaging.offline.api.OfflineDialogRepository

/**
 * Repository used for storing and retrieving a dialog in offline mode.
 * The dialog will be stored in the in-memory cache and in the database.
 */
internal class OfflineDialogRepositoryImpl(
    private val dialogsCacheProvider: DialogsCacheProvider
) : BaseDBRepository(DialogsTable.TABLE_NAME), OfflineDialogRepository {

    companion object {
        private const val TAG = "OfflineDialogRepositoryImpl"
    }

    override val cachedActiveDialog: Dialog?
        get() = dialogsCacheProvider.getActiveDialog()

    override fun getOrCreateOfflineDialog(
        targetId: String,
        brandId: String,
        conversationId: String,
        requestId: Long
    ): Dialog {
        val offlineDialog = dialogsCacheProvider.getDialogById(Dialog.OFFLINE_DIALOG_ID)
            ?.takeIf { it.targetId == targetId }
            ?.takeIf { it.brandId == brandId }
            ?: queryOfflineDialog(targetId, brandId, conversationId)
            ?: createAndSaveOfflineDialog(targetId, brandId, conversationId, requestId)
        dialogsCacheProvider.cacheDialog(offlineDialog)
        return offlineDialog
    }

    override fun clearOfflineDialog(targetId: String) {
        val whereClause = DialogsTable.Key.TARGET_ID + " = ?" + " AND " + DialogsTable.Key.DIALOG_ID + " = ?"
        val whereArgs = arrayOf(targetId, Dialog.OFFLINE_DIALOG_ID)
        db.removeAll(whereClause, whereArgs)
        dialogsCacheProvider.removeDialogFromCache(targetId, Dialog.OFFLINE_DIALOG_ID)
        d(TAG, "Finished removing offline dialog")
    }

    override fun queryRealActiveDialog(brandId: String): Dialog? {
        val query = (DialogsTable.Key.BRAND_ID + " = ?"
                + " AND " + DialogsTable.Key.STATE + " = ?"
                + " AND " + DialogsTable.Key.DIALOG_ID + " != ?"
                + " AND " + DialogsTable.Key.DIALOG_ID + " != ?")
        val arguments = arrayOf(
            brandId,
            DialogState.OPEN.ordinal.toString(),
            Dialog.TEMP_DIALOG_ID,
            Dialog.OFFLINE_DIALOG_ID
        )

        val result = runCatching {
            db.query(null, query, arguments, null, null, null, "1")
                ?.use { it.takeIf { it.moveToFirst() }?.let { Dialog(it) } }
        }
        return result.getOrNull()
    }

    /**
     * Method used for getting the offline dialog from the database
     * @param targetId the target id
     * @param brandId the brand id
     * @param conversationId the conversation id
     * @return the stored offline dialog if found, otherwise null
     */
    @WorkerThread
    private fun queryOfflineDialog(
        targetId: String,
        brandId: String,
        conversationId: String
    ): Dialog? {
        val query = (DialogsTable.Key.CONVERSATION_ID + " = ?"
                + " AND " + DialogsTable.Key.TARGET_ID + " = ?"
                + " AND " + DialogsTable.Key.BRAND_ID + " = ?"
                + " AND " + DialogsTable.Key.STATE + " = ?")

        val arguments = arrayOf(
            conversationId,
            targetId,
            brandId, DialogState.OFFLINE.ordinal.toString()
        )

        val result = runCatching {
            db.query(null, query, arguments, null, null, null, "1")
                ?.use { it.takeIf { it.moveToFirst() }?.let { Dialog(it) } }
        }.onFailure {
            d(TAG, "Failed to query for offline dialog", it)
        }
        return result.getOrNull()
    }

    /**
     * Method used for creating an offline dialog and storing it in the database
     * @param targetId the target id
     * @param brandId the brand id
     * @param conversationId the conversation id
     * @param requestId the request id
     * @return the newly created offline dialog
     */
    @WorkerThread
    private fun createAndSaveOfflineDialog(
        targetId: String,
        brandId: String,
        conversationId: String,
        requestId: Long
    ): Dialog {
        val dialog = Dialog(targetId, brandId)
        dialog.conversationId = conversationId
        dialog.dialogId = Dialog.OFFLINE_DIALOG_ID
        dialog.channelType = MultiDialog.ChannelType.MESSAGING
        dialog.dialogType = DialogType.MAIN
        dialog.state = DialogState.OFFLINE
        dialog.conversationTTRType = TTRType.NORMAL
        dialog.unreadMessages = -1
        dialog.requestId = requestId
        dialog.startTimestamp = System.currentTimeMillis()
        insertOfflineDialog(dialog)
        d(TAG, "Created offline dialog with id: " + dialog.dialogId)
        return dialog
    }

    /**
     * Method used for inserting a dialog in the database
     * @param dialog the dialog to be stored
     */
    @WorkerThread
    private fun insertOfflineDialog(dialog: Dialog) {
        val insertValues = ContentValues()
        insertValues.put(DialogsTable.Key.DIALOG_ID, dialog.dialogId)
        insertValues.put(DialogsTable.Key.BRAND_ID, dialog.brandId)
        insertValues.put(DialogsTable.Key.TARGET_ID, dialog.targetId)
        insertValues.put(DialogsTable.Key.CONVERSATION_ID, dialog.conversationId)
        insertValues.put(DialogsTable.Key.STATE, dialog.state.ordinal)
        insertValues.put(DialogsTable.Key.TTR_TYPE, dialog.conversationTTRType.ordinal)
        insertValues.put(DialogsTable.Key.DIALOG_TYPE, dialog.dialogType.name)
        insertValues.put(DialogsTable.Key.CHANNEL_TYPE, dialog.channelType.name)
        insertValues.put(DialogsTable.Key.UNREAD_MESSAGES_COUNT, -1)
        insertValues.put(DialogsTable.Key.START_TIMESTAMP, dialog.startTimestamp)
        insertValues.put(DialogsTable.Key.REQUEST_ID, dialog.requestId)
        db.insert(insertValues)
    }
}
