package com.liveperson.messaging.wm

import android.content.Context
import android.content.SharedPreferences
import android.net.Uri
import com.liveperson.infra.model.MessageOption
import com.liveperson.infra.model.deserializeStyleFromJson
import com.liveperson.infra.model.serializeToJson
import com.liveperson.messaging.wm.core.MultiProcessStore
import com.liveperson.messaging.wm.impl.MultiProcessEditorHelperImpl
import com.liveperson.messaging.wm.impl.ValueReaderImpl
import com.liveperson.messaging.wm.impl.contentUriOf
import org.json.JSONArray
import org.json.JSONObject

private const val KEY_DISPLAY_TEXT = "action.text"
private const val KEY_VALUE = "action.value"
private const val KEY_METADATA = "action.metadata"
private const val KEY_STYLE = "action.style"

/**
 * Extension function on context which serves as
 * factory method to create a MultiProcessStore instance.
 */
internal fun Context.getWelcomeMessagePreferences(): SharedPreferences {
    val uriGenerator: (String, String) -> Uri = { key, type ->
        contentUriOf(key, type)
    }
    return MultiProcessStore(
        MultiProcessEditorHelperImpl(contentResolver, uriGenerator),
        ValueReaderImpl(contentResolver, uriGenerator)
    )
}

/**
 * Extension function over MessageOption class
 * used to serialize message option to json object.
 *
 * @see serialize to understand how this method is used.
 */
private fun MessageOption.toJson(): JSONObject {
    return JSONObject().apply {
        put(KEY_DISPLAY_TEXT, displayText)
        put(KEY_VALUE, value)
        put(KEY_STYLE, style?.serializeToJson())
        put(KEY_METADATA, metadata)
    }
}

/**
 * Extension function over JSONObject class used
 * to deserialize json object content to message option.
 *
 * @see toMessageOptions to understand how this method is used.
 */
private fun JSONObject.optionFromJson(): MessageOption {
    return MessageOption(
        optString(KEY_DISPLAY_TEXT),
        optString(KEY_VALUE),
        optJSONArray(KEY_METADATA) ?: JSONArray(),
        optJSONObject(KEY_STYLE)?.deserializeStyleFromJson()
    )

}

/**
 * Extension function over list of message options used to serialize it
 * to json array.
 */
internal fun List<MessageOption>.serialize(): JSONArray {
    return map { it.toJson() }.fold(JSONArray()) { right, left -> right.apply { put(left) } }
}

/**
 * Extension function over json array used to deserialize it
 * to list of message options.
 */
internal fun JSONArray.toMessageOptions(): List<MessageOption> {
    return (0 until length()).asSequence()
        .map { getJSONObject(it) }
        .map { it.optionFromJson() }
        .toList()
}