package com.liveperson.messaging.background.filesharing;

import com.liveperson.api.response.BaseGenerateURLResponse;
import com.liveperson.api.response.IOnUrlReady;
import com.liveperson.api.response.model.QueryParams;
import com.liveperson.infra.ICallback;
import com.liveperson.infra.database.DataBaseExecutor;
import com.liveperson.infra.database.tables.FilesTable;
import com.liveperson.infra.log.LPLog;
import com.liveperson.infra.network.socket.SocketManager;
import com.liveperson.messaging.MessagingFactory;
import com.liveperson.messaging.background.UploadFileTaskCallback;
import com.liveperson.messaging.commands.SendFileCommand;
import com.liveperson.messaging.network.http.UploadFileRequest;
import com.liveperson.messaging.network.socket.requests.GetUrlForUploadRequest;

public abstract class BaseUploadTask {

	private static final String TAG = "BaseUploadTask";

	protected SendFileCommand sendMessageCommand;
	protected long mFileRowId = -1;
	private long mMessageRowId = -1;
	private FilesTable.LoadStatus mStatus = FilesTable.LoadStatus.NOT_STARTED;
	private String mRelativePath;
	protected byte[] mOriginalFileByteArray = null;
	private UploadFileTaskCallback mUploadFileTaskCallback;
	private int mUploadTimeout = 30000; // Default value

	protected abstract UploadFileTaskBundle getUploadTaskBundle();

	protected abstract byte[] getFileByteArray();

	public abstract int getTaskId();

	public abstract String getThumbnailBase64();

	public BaseUploadTask(Integer uploadTimeout) {
		if(uploadTimeout != null) {
			mUploadTimeout = uploadTimeout;
		}
	}

	protected void setSendMessageCommandCallback() {
		sendMessageCommand.setCallback(new SendFileCommand.SendFileCommandListener() {

			/**
			 * Callback after calling sendMessageCommand.addMessageToDB();
			 * @param messageRowId
			 * @param fileRowId - image file rowId
			 */
			@Override
			public void onFileAddedToDB(long messageRowId, long fileRowId, boolean isOfflineMessage) {
				mFileRowId = fileRowId;
				mMessageRowId = messageRowId;
				if (mUploadFileTaskCallback != null) {
					mUploadFileTaskCallback.onFileAddedToDB(isOfflineMessage);
				}
			}

			/**
			 * after connecting to server and getting updated- we update the message in DB.
			 * this callback will run after updating the message in DB.
			 */
			@Override
			public void onMessageUpdatedInDB() {
				//after updating message (with new conversation id) we can upload the image to swift server.
				//starting by generate url from UMS.
				generateUrl();
			}

		});

	}

	private void generateUrl() {
		setStatus(FilesTable.LoadStatus.REQUESTING_URL);

		LPLog.INSTANCE.d(TAG + Thread.currentThread(), "generateUrlRunnable");
		SocketManager.getInstance().send(new GetUrlForUploadRequest(MessagingFactory.getInstance().getController(),
				getUploadTaskBundle().getBrandId(), getFileByteArray().length, getUploadTaskBundle().getFileTypeExtension(),
				new IOnUrlReady() {
					@Override
					public void onUrlReady(final BaseGenerateURLResponse body) {
						LPLog.INSTANCE.d(TAG + Thread.currentThread(), "URL ready!!" + body.relativePath);
						//updating file relativePath on DB\
						mRelativePath = body.relativePath;
						MessagingFactory.getInstance().getController().amsFiles.updateRelativePath(mFileRowId, mRelativePath);

						if (isUploadFailed()) {
							return;
						}
						uploadToSwift(body.queryParams);
					}

					@Override
					public void onUrlError(String errorMessage) {
						if (isUploadFailed()) {
							return;
						}
						onUploadFailed(new Exception("failed to generate url." + errorMessage));
					}
				}));
	}


	public void setCallBack(UploadFileTaskCallback uploadFileTaskCallback) {
		mUploadFileTaskCallback = uploadFileTaskCallback;
	}

	private void uploadToSwift(QueryParams queryParams) {
		LPLog.INSTANCE.d(TAG + Thread.currentThread(), "uploading to swift..");

		setStatus(FilesTable.LoadStatus.UPLOADING);

		UploadFileRequest mUploadFileRequest = new UploadFileRequest(getUploadTaskBundle().getSwiftDomain(), mRelativePath, queryParams, getUploadTaskBundle().getRestParams().mCertificates, new ICallback<Object, Throwable>() {
			@Override
			public void onSuccess(Object value) {
				if (isUploadFailed()) {
					return;
				}
				LPLog.INSTANCE.d(TAG + Thread.currentThread(), "uploading to swift succeeded!");
				setStatus(FilesTable.LoadStatus.COMPLETED);
				if (mUploadFileTaskCallback != null) {
					mUploadFileTaskCallback.onUploadFinishedSuccessfully(BaseUploadTask.this);
				}
				//sendPublishFile will be called by the manager after old waiting image-messages
			}

			@Override
			public void onError(Throwable exception) {
				if (isUploadFailed()) {
					return;
				}
				onUploadFailed(new Exception("failed to upload to swift " + exception.getMessage()));

			}
		}, getFileByteArray(), mUploadTimeout);
		mUploadFileRequest.execute();
	}

	/**
	 * Adding message and file to db, when completed- will run the callback's method- onFileAddedToDB.
	 */
	public void startUpload() {
		if (sendMessageCommand != null) DataBaseExecutor.execute(() -> {
			sendMessageCommand.addMessageToDB();
		});
	}

	public void onConnectionAvailable() {
		LPLog.INSTANCE.d(TAG + Thread.currentThread(), "onConnectionAvailable");
		if (mStatus == FilesTable.LoadStatus.NOT_STARTED){
			setStatus(FilesTable.LoadStatus.PROCESSING);
			sendMessageCommand.updateMessageConversationId();
		}
	}

	public void onConnectionUnavailable() {
		onUploadFailed(new Exception("Failed to upload. connection unavailable"));
	}

	protected void onUploadFailed(Throwable exception) {
		LPLog.INSTANCE.d(TAG , "onUploadFailed. ", exception);
		if (sendMessageCommand != null){
			sendMessageCommand.failMessage();
		}
		setStatus(FilesTable.LoadStatus.FAILED);

		if (mUploadFileTaskCallback != null) {
			mUploadFileTaskCallback.onUploadFailed(this, exception);
		}

	}

	private void notifyChange() {
		MessagingFactory.getInstance().getController().amsMessages.updateFileMessageByRowId(mMessageRowId, mFileRowId).execute();
	}

	public void sendPublishFile(boolean sendViaRest) {
		LPLog.INSTANCE.d(TAG + Thread.currentThread(), "sending PublishImage request..");
		sendMessageCommand.setFileDetails(mRelativePath, getUploadTaskBundle().getFileTypeExtension(), getThumbnailBase64());
		sendMessageCommand.setSendViaRest(sendViaRest, getUploadTaskBundle().getRestParams());
		sendMessageCommand.execute();
	}

	public void setStatus(FilesTable.LoadStatus status) {
		this.mStatus = status;
		LPLog.INSTANCE.d(TAG + Thread.currentThread(), "set file status: "+ status +" mFileRowId = " + mFileRowId);
		if (mFileRowId != -1){
			MessagingFactory.getInstance()
					.getController()
					.amsFiles
					.updateStatus(mFileRowId, mStatus, data -> {
						notifyChange();
					});
		} else {
			notifyChange();
		}
	}

	public long getMessageRowId() {
		return mMessageRowId;
	}

	public boolean isUploadCompleted() {
		return mStatus == FilesTable.LoadStatus.COMPLETED;
	}
	private boolean isUploadFailed() {
		return mStatus == FilesTable.LoadStatus.FAILED;
	}

	public String getEventId() {
		return (sendMessageCommand != null ? sendMessageCommand.getEventId() : null );

	}

}
