package com.liveperson.messaging.commands.tasks

import android.os.Bundle
import com.liveperson.api.request.SubscribeMessagingEvents
import com.liveperson.infra.database.DataBaseExecutor
import com.liveperson.infra.log.LPLog
import com.liveperson.infra.network.socket.ResponseCallback
import com.liveperson.infra.network.socket.SocketManager
import com.liveperson.infra.utils.LocalBroadcast
import com.liveperson.messaging.Messaging
import com.liveperson.messaging.network.socket.requests.SubscribeMessagingEventsRequest

/**
 * Created by shiranr on 1/30/18.
 */

class MessagingEventSubscriptionManager {

    companion object {
        @JvmStatic
        val SUCCESS = "TASK_SUCCESS"
        @JvmStatic
        val ERROR_MESSAGE = "TASK_ERROR_MESSAGE"
        @JvmStatic
        val MESSAGE_EVENT_COMPLETED = "MESSAGE_EVENT_COMPLETED"

        fun onError(conversationId: String) {
            val bundle = Bundle()
            bundle.putBoolean(MessagingEventSubscriptionManager.SUCCESS, false)
            LocalBroadcast.sendBroadcast(MessagingEventSubscriptionManager.MESSAGE_EVENT_COMPLETED + conversationId, bundle)
        }
    }

    private var subscribedDialogs : HashMap<String, Boolean> = HashMap()
    private var subscribedDialogUIUpdate : HashMap<String, Boolean> = HashMap()

    fun addSubscription(
        controller : Messaging,
        brandID : String,
        conversationId : String,
        dialogId: String,
        fromSeq: Int = 0,
        updateUI: Boolean
    ) {
        @Suppress("NAME_SHADOWING") // we effectively want to replace the original value.
        var fromSeq = fromSeq
        if (fromSeq == -1) {
            fromSeq = 0
        }

        val url: String = controller.mAccountsController.getConnectionUrl(brandID).guard { LPLog.w("SubscriptionManager", "The brand (wih ID: $brandID) has no connection URL!"); return }

        if (!subscribedDialogs.contains(dialogId)) {
            LPLog.w("SubscriptionManager", "Adding subscription for dialogId: $dialogId from seq: $fromSeq")
            subscribedDialogs[dialogId] = true
            subscribedDialogUIUpdate[dialogId] = updateUI

            /**
             * Offline messages should be sent to UMS only after
             * subscription to messaging events.
             */

            val request = SubscribeMessagingEventsRequest(url, conversationId, dialogId, fromSeq)

            request.addCallback(OfflineModeSubscriptionRequestCallback(request, controller, dialogId))
            SocketManager.getInstance().send(request)

        } else {
            LPLog.w("SubscriptionManager", "Already added subscription for dialogId: $dialogId Subscription already exists!")
            //send intent its completed - only if we got notification already
            /*if (firstNotificationAfterSubscribe[conversationId] == false){
                LPMobileLog.w("SubscriptionManager", " $conversationId Subscription already exists! and already got event.")
                onReceivedEvent(conversationId)
            }*/
        }

    }

    fun onReceivedEvent(dialogId: String): Boolean {
        //send intent its completed
        var oldValue = subscribedDialogs[dialogId]

        if (oldValue == null){
            //in case firstNotificationAfterSubscribe was cleared
            oldValue = false
        }

        if (oldValue == true) {
            LPLog.w("SubscriptionManager", "onReceivedEvent! sending intent notification received for dialogId: $dialogId")
            val bundle = Bundle()
            bundle.putBoolean(SUCCESS, true)
            LocalBroadcast.sendBroadcast(MESSAGE_EVENT_COMPLETED + dialogId, bundle)
            subscribedDialogs[dialogId] = false
        }
        return oldValue
    }

    fun shouldUpdateUI(conversationId: String): Boolean {
        var oldValue = subscribedDialogUIUpdate[conversationId]
        subscribedDialogUIUpdate[conversationId] = true

        if (oldValue == null){
            oldValue = false
        }

        return oldValue
    }

    fun clearAllSubscriptions() {
        subscribedDialogs.clear()
    }

}

// Extension for adding Swift's guard keyword, from: https://android.jlelse.eu/a-few-ways-to-implement-a-swift-like-guard-in-kotlin-ffd94027864e
inline fun <T> T.guard(block: T.() -> Unit): T {
    if (this == null) block(); this.let { return it }
}

/**
 * Custom callback to handle subscriptions to messaging events
 * for request dialog.
 * This callback is used to send offline messages only after subscription
 * to reorder them according to actual timestamp received from server
 * or to process required offline messages based on lp_offline_messages_for_pcs_behavior
 * flag's value.
 */
private class OfflineModeSubscriptionRequestCallback(
    private val request: SubscribeMessagingEventsRequest,
    private val controller: Messaging,
    private val requestedDialogId: String
): ResponseCallback<SubscribeMessagingEvents.Response> {
    override fun onSuccess(data: SubscribeMessagingEvents.Response?) {
        DataBaseExecutor.execute {
            val currentActiveDialog = controller.amsDialogs.getActiveDialog()
            if (currentActiveDialog?.dialogId == requestedDialogId) {
                val offlineManager = controller.offlineManager
                offlineManager.isSubscribedToActiveDialog = true
                offlineManager.processOfflineMessages(currentActiveDialog)
            }
        }

        request.removeCallback(this)
    }

    override fun onError() {
        request.removeCallback(this)
    }

}