package com.liveperson.messaging.network.http;

import android.net.Uri;
import android.text.TextUtils;

import com.liveperson.infra.Command;
import com.liveperson.infra.ICallback;
import com.liveperson.infra.log.LPLog;
import com.liveperson.infra.network.http.HttpHandler;
import com.liveperson.infra.network.http.request.HttpGetRequest;
import com.liveperson.infra.otel.LPTraceType;
import com.liveperson.infra.utils.SdkConst;
import com.liveperson.messaging.Messaging;
import com.liveperson.messaging.controller.connection.ConnectionParamsCache;
import com.liveperson.messaging.model.AmsAccount;

import org.json.JSONException;
import org.json.JSONObject;

import static com.liveperson.infra.errors.ErrorCode.ERR_000000CF;

public class IncaGetFileUrlRequest implements Command {

    private static final String TAG = "IncaGetFileUrlRequest";

    private static final String INCA_GET_FILE_URL = "https://%s/messaging_history/api/account/%s/conversations/consumer/file-sharing?conversationId=%s&fileId=%s";
    private static final int REQUEST_TIMEOUT = 30000;

    private final String brandID;
    private final String conversationServerId;
    private final String fileId;
    private final Messaging controller;
    private final ICallback<Uri, Exception> mCallback;

    public IncaGetFileUrlRequest(Messaging controller, String brandId, String conversationId, String fileId, ICallback<Uri, Exception> callback) {
        this.controller = controller;
        this.brandID = brandId;
        this.conversationServerId = conversationId;
        this.fileId = fileId;
        this.mCallback = callback;
    }

    @Override
    public void execute() {
        String incaDomain = controller.mAccountsController.getServiceUrl(brandID, ConnectionParamsCache.CSDS_INCA_KEY);

        String historyUrl = String.format(INCA_GET_FILE_URL, incaDomain, brandID, conversationServerId, fileId);
        Uri uri = Uri.parse(historyUrl).buildUpon()
                .appendQueryParameter("source", SdkConst.SOURCE_NAME)
                .build();
        LPLog.INSTANCE.d(TAG, "Getting inca messages url " + uri.toString());
        AmsAccount account = controller.mAccountsController.getAccount(brandID);
        String token = account == null ? "" : account.getToken();
        HttpGetRequest httpGetRequest = new HttpGetRequest(uri.toString(),
                LPTraceType.GENERATE_FILE_SHARING_DOWNLOAD_URL_REQ);
        httpGetRequest.addHeader("Authorization", "Bearer " + token);
        httpGetRequest.setCertificatePinningKeys(controller.mAccountsController.getCertificatePinningKeys(brandID));
        httpGetRequest.setTimeout(REQUEST_TIMEOUT);
        httpGetRequest.setCallback(new ICallback<String, Exception>() {
            @Override
            public void onSuccess(String res) {
                LPLog.INSTANCE.d(TAG, "onSuccess with INCA file response details " + res);
                if (TextUtils.isEmpty(res)) {
                    return;
                }
                try {
                    JSONObject jsonObject = new JSONObject(res);
                    String urlString = jsonObject.optString("urlForDownload");

                    Uri fileUri = Uri.parse(urlString);

                    mCallback.onSuccess(fileUri);

                } catch (JSONException exception) {
                    mCallback.onError(exception);
                }

            }

            @Override
            public void onError(Exception exception) {
                LPLog.INSTANCE.e(TAG, ERR_000000CF, "Exception", exception);
                mCallback.onError(exception);
            }
        });

        HttpHandler.execute(httpGetRequest);

    }
}
