package com.liveperson.messaging.background;

import android.content.Context;
import android.os.Bundle;
import android.text.TextUtils;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.liveperson.api.response.model.ContentType;
import com.liveperson.api.response.model.DialogData;
import com.liveperson.api.response.model.Result;
import com.liveperson.api.response.types.CloseReason;
import com.liveperson.api.response.types.DialogType;
import com.liveperson.infra.ICallback;
import com.liveperson.infra.Infra;
import com.liveperson.infra.LPConversationsHistoryStateToDisplay;
import com.liveperson.infra.log.LPLog;
import com.liveperson.infra.managers.PreferenceManager;
import com.liveperson.infra.network.http.HttpHandler;
import com.liveperson.infra.network.http.body.HttpRequestBody;
import com.liveperson.infra.network.http.body.LPJSONObjectBody;
import com.liveperson.infra.network.http.request.HttpPostRequest;
import com.liveperson.infra.otel.LPTraceType;
import com.liveperson.infra.utils.ClockUtils;
import com.liveperson.infra.utils.EncryptionVersion;
import com.liveperson.infra.utils.LocalBroadcast;
import com.liveperson.messaging.Messaging;
import com.liveperson.messaging.MessagingFactory;
import com.liveperson.messaging.model.AmsConversations;
import com.liveperson.messaging.model.AmsMessages;
import com.liveperson.messaging.model.CoBrowseMetadata;
import com.liveperson.messaging.model.Dialog;
import com.liveperson.messaging.model.FullMessageRow;
import com.liveperson.messaging.model.MessagingChatMessage;
import com.liveperson.messaging.model.MessagingUserProfile;

import org.json.JSONException;
import org.json.JSONObject;

import java.util.ArrayList;
import java.util.Objects;

import static com.liveperson.infra.errors.ErrorCode.*;
import static com.liveperson.messaging.model.CoBrowseMetadata.COLLABORATION_CALL_ENDED;
import static com.liveperson.messaging.model.CoBrowseMetadata.COLLABORATION_CALL_ENDED_BY_CONSUMER;

public class CoBrowseManager {

	private static final String TAG = "CoBrowseManager";
	private static final String KEY_IS_COBROWSE_ACTIVE = "cobrowse.is_active";

	public static String MODE_VIDEO_CALL = "VIDEO_CALL";
	public static String MODE_VOICE_CALL = "VOICE_CALL";

	public static String STATUS_INVITED = "INVITED";
	public static String STATUS_CLOSED = "CLOSED";
	public static String STATUS_ACCEPTED = "ACCEPTED";

	public static String NOTIFICATION_KEY_JOINED = "joined";
	public static String NOTIFICATION_KEY_ACCEPTED = "accepted";

	public static String NOTIFICATION_KEY_INVITED = "invited";
	public static String NOTIFICATION_KEY_CONSUMER_LEFT = "consumer_left";

	public static String KEY_COBROWSE_METADATA = "cobrowse_metadata";

	public static final CoBrowseManager instance = new CoBrowseManager();

	private PreferenceManager mPreferenceManager;

	private CoBrowseManager() {
		mPreferenceManager = PreferenceManager.getInstance();
	}

	public void createMessage(Messaging controller, String dialogId, long timestamp, String mainDialogId, String originatorId) {
		MessagingChatMessage inviteMessage = new MessagingChatMessage(
				originatorId,
				dialogId,
				ClockUtils.addDiffToTimestamp(timestamp),
				mainDialogId,
				dialogId,
				MessagingChatMessage.MessageType.COBROWSE,
				MessagingChatMessage.MessageState.RECEIVED,
				EncryptionVersion.NONE
		);
		LPConversationsHistoryStateToDisplay state = controller.getConversationViewParams()
				.getHistoryConversationsStateToDisplay();
		controller.amsMessages.addMessage(inviteMessage, state != LPConversationsHistoryStateToDisplay.CLOSE)
				.setPostQueryOnBackground((id) -> {
					if (id != null && id > 0) {
						LPLog.INSTANCE.d(TAG, "CoBrowse INVITATION intercepted!! creating a message for dialogId: " + dialogId);
					}
				})
				.execute();
	}

	public void createTranscendentMessage(
			Messaging controller,
			String mainDialogId,
			String originatorId,
			String text,
			DialogData cobrowseDialog
	) {
		long timestamp = ClockUtils.addDiffToTimestamp(cobrowseDialog.metaDataLastUpdateTs);
		MessagingChatMessage inviteMessage = new MessagingChatMessage(originatorId,
				text,
				timestamp,
				mainDialogId,
				cobrowseDialog.dialogId + cobrowseDialog.metaData.sessionState,
				MessagingChatMessage.MessageType.CONTROLLER_SYSTEM,
				MessagingChatMessage.MessageState.RECEIVED,
				AmsMessages.TRANSCENDENT_MESSAGE_SEQUENCE_NUMBER,
				ContentType.text_plain.getText(),
				EncryptionVersion.NONE
		);
		LPConversationsHistoryStateToDisplay state = controller
				.getConversationViewParams()
				.getHistoryConversationsStateToDisplay();
		boolean updateUi = state != LPConversationsHistoryStateToDisplay.CLOSE;
		controller.amsMessages.addMessage(inviteMessage, updateUi).execute();
	}

	public ArrayList<FullMessageRow> getMessages(Messaging controller, String dialogId, String filterDialogId, Long filterTimestamp, String filterString) {
		ArrayList<FullMessageRow> searchedMessageList = controller.amsMessages.loadMessagesOnDbThread(AmsMessages.MessagesSortedBy.DialogId, dialogId, -1, -1, -1);
		ArrayList<FullMessageRow> filtersMessageList = new ArrayList<>();

        if (searchedMessageList != null) {
            LPLog.INSTANCE.d(TAG, "CoBrowse messages retrieved: " + searchedMessageList.size() + "   for mainDialogId: " + dialogId);
            if (filterDialogId != null) {
                for (FullMessageRow msg : searchedMessageList) {
                    if (msg.getMessagingChatMessage().getMessage().equals(filterDialogId)) {
                        filtersMessageList.add(msg);
                    }
                }
            } else if (filterTimestamp != null && filterString != null) {
                for (FullMessageRow msg : searchedMessageList) {
                    if (msg.getMessagingChatMessage().getTimeStamp() == filterTimestamp && msg.getMessagingChatMessage().getMessage().equals(filterString)) {
                        filtersMessageList.add(msg);
                    }
                }
            } else {
                filtersMessageList.addAll(searchedMessageList);
            }
        }

		return filtersMessageList;
	}

	public CoBrowseMetadata saveMetadata(String brandId, String dialogId, CoBrowseMetadata metadata) {
		PreferenceManager.getInstance().setStringValue(KEY_COBROWSE_METADATA + dialogId, brandId, metadata.getJsonObject().toString());

		return metadata;
	}

	@Nullable
	public CoBrowseMetadata getMetadata(String brandId, String dialogId) {
		try {
			String metadataJson = PreferenceManager.getInstance().getStringValue(KEY_COBROWSE_METADATA + dialogId, brandId, null);

			if(metadataJson != null) {
				LPLog.INSTANCE.d(TAG, "CoBrowse manager metadataJson "+metadataJson);

				JSONObject metadata = new JSONObject(metadataJson);
				return new CoBrowseMetadata(metadata);
			}

			return null;
		} catch (JSONException e) {
			e.printStackTrace();
			return null;
		}
	}

	public void saveHistoricalCobrowseDialog(
			Messaging controller,
			Dialog cobrowseDialog,
			DialogData mainDialog,
			boolean updateUi
	) {
		if (!areTranscendedMessagesEnabled()) {
			LPLog.INSTANCE.d(TAG, "saveHistoricalCobrowseDialog: transcendent messages are disable, skipping saving");
			return;
		}

		if (cobrowseDialog.isOpen()) {
			LPLog.INSTANCE.d(TAG, "saveHistoricalCobrowseDialog: cobrowse dialog is open, skipping saving");
			return;
		}

		if (mainDialog.isOpen) {
			LPLog.INSTANCE.d(TAG, "saveHistoricalCobrowseDialog: main dialog is open, skipping saving");
			return;
		}

		final String messageKey;

		if (cobrowseDialog.getCloseReason() == CloseReason.CONSUMER) {
			messageKey = COLLABORATION_CALL_ENDED_BY_CONSUMER;
		} else {
			messageKey = COLLABORATION_CALL_ENDED;
		}

		String messageText = getTranscendentMessageContent(messageKey);

		MessagingChatMessage endCallMessage = createEndCallMessage(cobrowseDialog, mainDialog, messageText);

		controller.amsMessages.addMessage(endCallMessage, updateUi).execute();
	}

	private boolean areTranscendedMessagesEnabled() {
		PreferenceManager manager = PreferenceManager.getInstance();
		return manager.getBooleanValue(
				PreferenceManager.ENABLE_TRANSCENDENT_MESSAGES_PREFERENCE_KEY,
				PreferenceManager.APP_LEVEL_PREFERENCES,
				true
		);
	}

	private String getStringResourceByName(String resourceName) {
		final Context context = Infra.instance.getApplicationContext();
		try {
			String packageName = context.getPackageName();
			int resId = context.getResources().getIdentifier(resourceName, "string", packageName);
			return context.getString(resId);
		} catch (Exception exception) {
			LPLog.INSTANCE.d(TAG, "Failed to string resource: " + resourceName, exception);
			return "";
		}
	}

	/**
	 * Method used to receive a string resources associated with
	 * message key(cobrowse dialog's event)
	 * @param messageKey cobrowse dialog's event key
	 * @return String value of associated resource.
	 */
	public String getTranscendentMessageContent(String messageKey) {
		return getStringResourceByName("cobrowse_" + messageKey);
	}

	@NonNull
	private MessagingChatMessage createEndCallMessage(Dialog cobrowseDialog, DialogData mainDialog, String messageText) {
		long cobrowseEndTime = ClockUtils.addDiffToTimestamp(cobrowseDialog.getEndTimestamp());
		long mainDialogEndTime = ClockUtils.addDiffToTimestamp(mainDialog.endTs);

		long messageTimestamp;
		if (cobrowseEndTime > mainDialogEndTime) {
			messageTimestamp = mainDialogEndTime - 1;
		} else {
			messageTimestamp = cobrowseEndTime;
		}

		return new MessagingChatMessage(
				cobrowseDialog.getAssignedAgentId(),
				messageText,
				messageTimestamp,
				mainDialog.dialogId,
				cobrowseDialog.getDialogId(),
				MessagingChatMessage.MessageType.CONTROLLER_SYSTEM,
				MessagingChatMessage.MessageState.RECEIVED,
				AmsMessages.TRANSCENDENT_MESSAGE_SEQUENCE_NUMBER,
				ContentType.text_plain.getText(),
				EncryptionVersion.NONE
		);
	}

	public void cobrowseMessageReceived(
			String brandId,
			Messaging controller,
			DialogData dialog,
			Result result,
			String assignedAgentId
	) {
		DialogData mainDialog = null;

		for(DialogData dialogData : result.conversationDetails.dialogs) {
			if(dialogData.dialogType == DialogType.MAIN) {
				mainDialog = dialogData;
				break;
			}
		}

		CoBrowseMetadata metadata = dialog.metaData;
		if (metadata == null) {
			LPLog.INSTANCE.e(TAG, ERR_00000159,"metadata is missing");
			return;
		}

		CoBrowseMetadata currentMetadata = CoBrowseManager.instance.getMetadata(result.conversationDetails.brandId, dialog.dialogId);
		if(currentMetadata != null && metadata.callLink == null) {
			//because the call link is lost with the ACCEPTED cobrowse update
			metadata.callLink = currentMetadata.callLink;
		}

		if (mainDialog == null) {
			LPLog.INSTANCE.e(TAG, ERR_00000159,"main dialog is missed");
			return;
		}

		if (!mainDialog.isOpen) {
			LPLog.INSTANCE.e(TAG, ERR_00000159,"main dialog is closed");
			if (dialog.isOpen && Objects.equals(controller.amsDialogs.getActiveCoBrowseDialogId(), dialog.dialogId)) {
				if (mainDialog.closeReason == CloseReason.CONSUMER) {
					controller.closeCobrowseDialog(
							brandId,
							dialog.dialogId,
							dialog.conversationId,
							metadata.isVideoCall(),
							null
					);
				}
				sendCobrowseMetadataUpdates(brandId, mainDialog.dialogId, assignedAgentId, metadata, false, true);
			}
			return;
		}


		saveMetadata(result.conversationDetails.brandId, dialog.dialogId, metadata);

		ArrayList<FullMessageRow> messages = CoBrowseManager.instance.getMessages(controller, mainDialog.dialogId, metadata.dialogId, null, null);

		if(metadata.isJoinable()) {
			LPLog.INSTANCE.d(TAG, "CoBrowse INVITATION intercepted!! callLink: "+ metadata.callLink);
			LPLog.INSTANCE.d(TAG, "CoBrowse found "+messages.size()+" messages on mainDialogId "+mainDialog.dialogId+" with coBrowseDialog "+metadata.dialogId);
			createMessage(controller, dialog.dialogId, dialog.creationTs, mainDialog.dialogId, assignedAgentId);

			controller.amsDialogs.setActiveCoBrowseDialogId(dialog.dialogId);
		}

		boolean isAutoMessagesEnabled = controller.mAccountsController.isAutoMessagesEnabled(brandId);
		if (metadata.notificationKey != null && (isAutoMessagesEnabled || areTranscendedMessagesEnabled())) {
			final String str = getTranscendentMessageContent(metadata.getAutomaticMessageKey());
			String agentId = result.conversationDetails.participants.ASSIGNED_AGENT[0];
			DialogData finalMainDialog = mainDialog;
			controller.amsUsers.getUserById(agentId)
					.setPostQueryOnUI((MessagingUserProfile data) -> {
						String nickname = data == null
								? getStringResourceByName("lp_accessibility_agent")
								: data.getNickname();
						String msgFormatted = String.format(str, nickname);
						createTranscendentMessage(
								controller,
								finalMainDialog.dialogId,
								agentId,
								msgFormatted,
								dialog
						);
                        sendCobrowseMetadataUpdates(brandId, finalMainDialog.dialogId, assignedAgentId, metadata, messages.isEmpty(), false);
					}).execute();
		} else {
			sendCobrowseMetadataUpdates(brandId, mainDialog.dialogId, assignedAgentId, metadata, messages.isEmpty(), false);
		}

		LPLog.INSTANCE.d(TAG, "CoBrowse intercepted!! " + metadata.sessionState
				+ "  dialog.metaDataLastUpdateTs: " + dialog.metaDataLastUpdateTs
				+ "  dialogId: " + metadata.dialogId
				+ "  getAutomaticMessageKey: " + metadata.getAutomaticMessageKey()
		);
	}

	/**
	 * Method used notify components about cobrowse call
	 * metadata changes.
	 * @param brandId active brand id.
	 * @param mainDialogId id of main dialog.
	 * @param assignedAgentId id of latest assigned agent.
	 * @param metadata cobrowse dialog metadata to notify.
	 * @param hasNewMessages flag to check whether new transcended messages arrived.
	 * @param isMainDialogResolved boolean flag to check whether main dialog is resolved and Co-Browse
	 *                             invitation should be hidden.
	 */
	private void sendCobrowseMetadataUpdates(
			String brandId,
			String mainDialogId,
			String assignedAgentId,
			CoBrowseMetadata metadata,
			boolean hasNewMessages,
			boolean isMainDialogResolved
	) {
		Bundle bundle = new Bundle();
		bundle.putString("dialogId", metadata.dialogId);
		bundle.putString("brandId", brandId);
		bundle.putString("agentId", assignedAgentId);
		bundle.putString("sessionState", metadata.sessionState);
		bundle.putString("automaticMessageKey", metadata.getAutomaticMessageKey());
		bundle.putBoolean("newMessages", hasNewMessages);
		bundle.putString("mainDialogId", mainDialogId);
		bundle.putBoolean("isMainDialogResolved", isMainDialogResolved);
		LocalBroadcast.sendBroadcast(AmsConversations.BROADCAST_COBROWSE_RECEIVED, bundle);
	}

	public void cancelCobrowse(String brandId, String url, String action) {
		try {
			String token = MessagingFactory.getInstance().getController().mAccountsController.getToken(brandId);

			JSONObject requestJson = new JSONObject();
			requestJson.put("action", action);

			HttpPostRequest httpPostRequest = new HttpPostRequest(url,
					LPTraceType.HTTP_REQUEST);
			HttpRequestBody jsonBody = new LPJSONObjectBody(requestJson);
			httpPostRequest.setBody(jsonBody);
			httpPostRequest.addHeader("authorization", "bearer " + token);

			httpPostRequest.setCallback(new ICallback<String, Exception>() {

				@Override
				public void onSuccess(String value) {
                    if (TextUtils.isEmpty(value)) {
                        LPLog.INSTANCE.e(TAG, ERR_00000158, "Received empty response from server for cancel cobrowse");
                    }
                }

				@Override
				public void onError(Exception exception) {
					if (exception != null) {
						LPLog.INSTANCE.e(TAG, ERR_00000158, "Failed to send cancel CoBrowse request.", exception);
					}
				}
			});
			HttpHandler.execute(httpPostRequest);
		} catch (JSONException e) {
			e.printStackTrace();
		}
	}

	public void setIsCobrowseActive(String brandId, boolean isActive) {
		mPreferenceManager.setBooleanValue(KEY_IS_COBROWSE_ACTIVE, brandId, isActive);
	}

	public boolean isCobrowseActive(String brandId) {
		return mPreferenceManager.getBooleanValue(KEY_IS_COBROWSE_ACTIVE, brandId, false);
	}

}
