package com.liveperson.messaging.commands;

import androidx.annotation.NonNull;

import com.liveperson.infra.utils.EncryptionVersion;
import com.liveperson.infra.utils.MaskedMessage;
import com.liveperson.messaging.Messaging;
import com.liveperson.messaging.model.Form;
import com.liveperson.messaging.model.MessagingChatMessage;
import com.liveperson.messaging.network.socket.requests.SendFormSubmissionMessageRequest;


/**
 * A command for sending a message notifying UMS that a Secure Form has been submitted.
 */
public class SendFormSubmissionMessageCommand extends SendMessageCommand {

	private String mSubmissionId;
	private String mInvitationId;

	/**
	 * Creates a send command for submitted-form messages with a specific pre-existing event ID
	 *
	 * @param form The Form that this message is a submission for
	 * @param message The Message being sent that includes the invitationId and submissionId
	 * @param controller The Messaging instance that will be used to send this command.
	 */
	public SendFormSubmissionMessageCommand(Form form, MaskedMessage message, Messaging controller) {
		super(controller, form.getSiteId(), form.getSiteId(), message);
		mBrandId = form.getSiteId();
		mTargetId = form.getSiteId();
		mMessage = message;
		mSubmissionId = form.getSubmissionId();
		mInvitationId = form.getInvitationId();
	}

	@NonNull
	@Override
	protected MessagingChatMessage createNewChatMessage(String dialogId, MaskedMessage message, MessagingChatMessage.MessageState state) {
		return new MessagingChatMessage(mConsumerId,
				message.getDbMessage(),
				System.currentTimeMillis(),
				dialogId,
				mEventId,
				MessagingChatMessage.MessageType.CONSUMER_FORM,
				state,
				EncryptionVersion.NONE
		);
	}

	@NonNull
	@Override
	protected SendFormSubmissionMessageRequest createMessageRequest(Messaging mController, String mEventId, String mTargetId, String mBrandId, String dialogId, String conversationId) {
		return new SendFormSubmissionMessageRequest(mController, mEventId, mTargetId, mBrandId, dialogId, conversationId, mInvitationId, mSubmissionId);
	}
}