package com.liveperson.api.response.model;

import android.text.TextUtils;

import com.liveperson.infra.log.LPLog;

import java.util.ArrayList;
import java.util.HashSet;

import static com.liveperson.infra.errors.ErrorCode.ERR_0000005A;
import static com.liveperson.infra.errors.ErrorCode.ERR_0000005B;

/**
 * Created by oferd on 8/10/15.
 */
public class Participants {

    private static final String TAG = "Participants";

    public String[] CONSUMER = new String[]{};
    public String[] ASSIGNED_AGENT = new String[]{};
    public String[] MANAGER = new String[]{};
    public String[] READER = new String[]{};
    public String[] CONTROLLER = new String[]{};
    public String[] AGENTS = new String[]{};
    public String[] ALL_AGENTS = new String[]{};

    public enum ParticipantRole {
        CONSUMER("CONSUMER"), ASSIGNED_AGENT("ASSIGNED_AGENT"), AGENT("AGENT"), MANAGER("MANAGER"), READER("READER"), CONTROLLER("CONTROLLER");
        private static final String TAG = "ParticipantRole";
        String participantType;
        ParticipantRole(String type){
            participantType = type;
        }

        public static ParticipantRole getParticipantRole(String role) {
            for (ParticipantRole pr: values()){
                if (pr.participantType.equalsIgnoreCase(role)){
                    return pr;
                }
            }

            LPLog.INSTANCE.e(TAG, ERR_0000005A, "Unknown 'ParticipantRole' named: " + role);
            return null;
        }

        public static ParticipantRole parse(String participantRoleString) {
            ParticipantRole result = null;
            try {
                result = valueOf(participantRoleString);
            } catch (IllegalArgumentException e) {
                ParticipantRole[] values = values();
                for (ParticipantRole value : values) {
                    if (value.name().equalsIgnoreCase(participantRoleString)) {
                        result = value;
                        break;
                    }
                }
            }

            return result;
        }
    }

    public Participants(){}

    public void add(ArrayList<String> id, ParticipantRole role) {
        add(id.toArray(new String[]{}), role);
    }

    public void add(String[] id, ParticipantRole role) {
        switch (role) {
            case READER:
                READER = id;
                mergeAgents(role);
                break;
            case MANAGER:
                MANAGER = id;
                mergeAgents(role);
                break;
            case CONSUMER:
                mergeConsumers(id);
                break;
            case CONTROLLER:
                mergeControllers(id);
                break;
            case ASSIGNED_AGENT:
                ASSIGNED_AGENT = id;
                mergeAgents(role);
                break;
            case AGENT:
                AGENTS = id;
                mergeAgents(role);
                break;
            default:
	            LPLog.INSTANCE.e(TAG, ERR_0000005B, "Found new participant role: " + role + ". Ignore it!");
        }
    }

    /**
     * Merge new agent (MANAGER/ASSIGNED_AGENT/AGENT/READER) into ALL_AGENTS array
     * @param currentRole The new agent role we want to merge into ALL_AGENTS array
     */
    private void mergeAgents(ParticipantRole currentRole) {

        String[] currentAgents = new String[]{};
        if (currentRole == ParticipantRole.MANAGER) {
            currentAgents = MANAGER;
        } else if (currentRole == ParticipantRole.ASSIGNED_AGENT) {
            currentAgents = ASSIGNED_AGENT;
        } else if (currentRole == ParticipantRole.AGENT) {
            currentAgents = AGENTS;
        } else if (currentRole == ParticipantRole.READER) {
            currentAgents = READER;
        }

        String[] previousAgents = ALL_AGENTS; // Copy previous list of agents into temporary array

        ALL_AGENTS = new String[currentAgents.length + previousAgents.length];
        System.arraycopy(currentAgents, 0, ALL_AGENTS, 0, currentAgents.length);
        System.arraycopy(previousAgents, 0, ALL_AGENTS, currentAgents.length, previousAgents.length);
    }

    /**
     * Merge new controller into CONTROLLER array
     * @param controller New controller ids we want to merge into CONTROLLER array
     */
    private void mergeControllers(String[] controller) {
        String[] previousControllers = CONTROLLER;
        CONTROLLER = new String[controller.length + previousControllers.length];
        System.arraycopy(controller, 0, CONTROLLER, 0, controller.length);
        System.arraycopy(previousControllers, 0, CONTROLLER, controller.length, previousControllers.length);
    }

    /**
     * Merge new consumer into CONSUMER array
     * @param consumer New consumer ids we want to merge into CONSUMER array
     */
    private void mergeConsumers(String[] consumer) {
        String[] previousConsumers = CONSUMER;
        CONSUMER = new String[consumer.length + previousConsumers.length];
        System.arraycopy(consumer, 0, CONSUMER, 0, consumer.length);
        System.arraycopy(previousConsumers, 0, CONSUMER, consumer.length, previousConsumers.length);
    }

    /**
     * Extracts and returns all the participants IDs from all the IDs arrays
     * @return An array of strings that contains only IDsm without roles/
     */
    public String[] extractAllParticipantsIds() {
        HashSet<String> result = new HashSet<>();
        ArrayList<String[]> idsHolders = new ArrayList<>();
        idsHolders.add(CONSUMER);
        idsHolders.add(ASSIGNED_AGENT);
        idsHolders.add(AGENTS);
        idsHolders.add(MANAGER);
        idsHolders.add(READER);
        idsHolders.add(CONTROLLER);
        idsHolders.add(ALL_AGENTS);

        for (String[] idsHolder : idsHolders) {
            for (String id : idsHolder) {
                if (!TextUtils.isEmpty(id)) {
                    result.add(id);
                }
            }
        }

        return result.toArray(new String[0]);
    }
}
