package com.liveperson.messaging.commands;

import androidx.annotation.NonNull;

import com.liveperson.api.response.model.ContentType;
import com.liveperson.api.response.types.DialogState;
import com.liveperson.infra.log.LPLog;
import com.liveperson.infra.utils.EncryptionVersion;
import com.liveperson.infra.utils.MaskedMessage;
import com.liveperson.infra.utils.UniqueID;
import com.liveperson.messaging.Messaging;
import com.liveperson.messaging.model.AmsMessages;
import com.liveperson.messaging.model.Dialog;
import com.liveperson.messaging.model.MessagingChatMessage;
import com.liveperson.messaging.network.socket.requests.SendMessageRequest;

import org.json.JSONException;
import org.json.JSONObject;

import static com.liveperson.infra.errors.ErrorCode.ERR_000000D4;


/**
 * Created by Maayan on 23/11/2016
 * A command for sending message that contain at least one URL
 */
public class SendMessageWithURLCommand extends SendMessageCommand {

    private static final String TAG = "SendMessageWithURLCommand";
    private String mDescription;
    private String mTitle;
    private String mImageUrl;
    private String mURLToParse;
    private String mSiteName;

    /**
     * Creates a command for sending messages
     *
     * @param controller
     * @param message
     */
    public SendMessageWithURLCommand(Messaging controller, String targetId, String brandId, MaskedMessage message, String urlToParse, String title, String description, String imageURL, String siteName) {
        super(controller, targetId, brandId, message);
        mBrandId = brandId;
        mTargetId = targetId;
        mMessage = message;
        mDescription = description;
        mTitle = title;
        mURLToParse = urlToParse;
        mImageUrl = imageURL;
        mSiteName = siteName;
    }


    /**
     * generateJSON - from the original msg for later paring the message with the url
     *
     * @param msg - the original msg from the consumer
     * @return JSONObject in the format: ... with all the relevant data for later paring the message with the url
     */
    private JSONObject generateJSON(String msg) {
        try {
            JSONObject json = new JSONObject();
            JSONObject messageWithURL = new JSONObject();
            messageWithURL.put("title", mTitle);
            messageWithURL.put("description", mDescription);
            messageWithURL.put("image_url", mImageUrl);
            messageWithURL.put("original_message", msg);
            messageWithURL.put("original_url_to_parse", mURLToParse);
            messageWithURL.put("site_name_url_to_parse", mSiteName);
            json.put("message_with_url", messageWithURL);
            return json;
        } catch (JSONException e) {
            LPLog.INSTANCE.e(TAG, ERR_000000D4, "generateJSON exception", e);
        }
        return null;
    }

    @Override
    protected void sendMessageIfDialogIsOpen() {
        Dialog dialog = mController.amsDialogs.getActiveDialog();
        if (dialog != null) {
            SendMessageRequest sendMessageRequest;
            LPLog.INSTANCE.d(TAG, "sendMessageIfDialogIsOpen: " + dialog.getState());
            if (dialog.getState() == DialogState.OPEN) {
                sendMessageRequest = createMessageRequest(mController, mEventId, mTargetId, mBrandId, dialog.getDialogId(), dialog.getConversationId());
                sendMessage(dialog.getDialogId(), sendMessageRequest);
            } else if (dialog.getState() == DialogState.PENDING) {
                sendMessageRequest = createMessageRequest(mController, mEventId, mTargetId, mBrandId, null, null); // TODO: Perry: "why not 'dialog.getDialogId() + dialog.getConversationId()' instead of null ?"
                dialog.getPendingData().addToPendingRequests(sendMessageRequest);
            }
        }
    }

    @NonNull
    @Override
    protected MessagingChatMessage createNewChatMessage(
            String dialogId,
            MaskedMessage message,
            MessagingChatMessage.MessageState state
    ) {
        MessagingChatMessage.MessageType messageType = message.isMasked()
                ? MessagingChatMessage.MessageType.CONSUMER_URL_MASKED
                : MessagingChatMessage.MessageType.CONSUMER_URL;
        return new MessagingChatMessage(
                mConsumerId,
                message.getDbMessage(), System.currentTimeMillis(),
                dialogId,
                mEventId,
                messageType,
                state,
                EncryptionVersion.NONE
        );
    }

    protected void addMessageToDBAndSend(
            String dialogId,
            MaskedMessage message,
            MessagingChatMessage.MessageState messageState
    ) {

        String finalMsg = generateJSON(message.getDbMessage()).toString();
        message.setDbMessage(finalMsg);

        mEventId = UniqueID.createUniqueMessageEventId();
        LPLog.INSTANCE.i(TAG, "addMessageWithURLToDBAndSend: mEventId = " + mEventId + " dialog id = " + message);

        // create new chat message and send it
        final MessagingChatMessage chatMessage = createNewChatMessage(dialogId, message, messageState);
        LPLog.INSTANCE.i(TAG, "addMessageWithURLToDBAndSend: chatMessage.MessageType = " + chatMessage.getMessageType() + " MessageState() = " + chatMessage.getMessageState());

        mController.amsMessages.addMessage(chatMessage, true)
                .setPreQueryOnBackground(() -> attachMessageToOfflineModeIfNeeded(chatMessage))
                .setPostQueryOnBackground(data -> {
	                LPLog.INSTANCE.i(TAG, "Send message, time: " + chatMessage.getTimeStamp());
                    if (chatMessage.getMessageState() != MessagingChatMessage.MessageState.OFFLINE) {
                        sendMessageIfDialogIsOpen();
                    }
                }).execute();

        // If messages are different it means the message was masked. We add a system message for the user
        if (mMessage.isMasked()) {
            String maskedEventId = mEventId + AmsMessages.MASKED_MESSAGE_EVENT_ID_POSTFIX;

            MessagingChatMessage warning = new MessagingChatMessage(
                    chatMessage.getOriginatorId(),
                    message.getMaskedSystemMessage(),
                    chatMessage.getTimeStamp() + 1,
                    chatMessage.getDialogId(), maskedEventId,
                    MessagingChatMessage.MessageType.SYSTEM_MASKED,
                    MessagingChatMessage.MessageState.RECEIVED,
                    AmsMessages.MASKED_CC_MSG_SEQUENCE_NUMBER,
                    ContentType.text_plain.getText(),
                    EncryptionVersion.NONE
            );

            mController.amsMessages.addMessage(warning, true).execute();
        }
    }

}
