package com.liveperson.messaging.commands.pusher

import com.liveperson.infra.Command
import com.liveperson.infra.ICallback
import com.liveperson.infra.Infra
import com.liveperson.infra.InternetConnectionService
import com.liveperson.infra.errors.ErrorCode
import com.liveperson.infra.log.LPLog
import com.liveperson.infra.network.http.HttpHandler
import com.liveperson.infra.network.http.request.HttpDeleteRequest
import com.liveperson.infra.otel.LPTraceType
import com.liveperson.messaging.LpError
import com.liveperson.messaging.Messaging
import com.liveperson.messaging.model.SynchronizedInternetConnectionCallback
import javax.net.ssl.SSLPeerUnverifiedException

class DeletePendingProactiveMessagesCommand(
    val controller: Messaging, val brandId: String,
    private val transactionIds: Set<String>,
    val callback: ICallback<Boolean, Exception>) : Command {

    private var mPusherHelper: PusherHelper? = null

    override fun execute() {
        mPusherHelper = PusherHelper(brandId, controller)
        if (transactionIds.isNullOrEmpty()) {
            notifyError(Exception("$ERROR_UNABLE_TO_MAKE_REQUEST} Error: No transactionId"))
            return
        }
        validateStateAndSendRequest()
    }

    private fun validateStateAndSendRequest() {
        if (InternetConnectionService.isNetworkAvailable()) {
            sendRequest()
        } else {
            SynchronizedInternetConnectionCallback { sendRequest() }.execute()
        }
    }

    private fun sendRequest() {
        if (mPusherHelper == null) {
            mPusherHelper = PusherHelper(brandId, controller)
        }

        val pusherDomain = mPusherHelper?.getPusherDomain()
        if (pusherDomain.isNullOrBlank()) {
            notifyError(Exception("$ERROR_UNABLE_TO_MAKE_REQUEST} Error: Missing Domain"))
            return
        }

        val consumerId = mPusherHelper?.getConsumerId()
        if (consumerId.isNullOrBlank()) {
            notifyError(Exception("$ERROR_UNABLE_TO_MAKE_REQUEST} Error: Missing consumerID"))
            return
        }

        val token = mPusherHelper?.getAuthToken()
        if (token.isNullOrBlank()) {
            notifyError(Exception("$ERROR_UNABLE_TO_MAKE_REQUEST} Error: Missing auth token"))
            return
        }

        val appId = mPusherHelper?.getAppId()
        if (appId.isNullOrBlank()) {
            notifyError(Exception("$ERROR_UNABLE_TO_MAKE_REQUEST Error: Missing appID"))
            return
        }

        // convert the set to a string, use comma to separate ids
        val sTransactionIds = transactionIds.joinToString(",")

        val requestURL = String.format(
            DELETE_URL, pusherDomain,
            brandId, appId,
            consumerId, sTransactionIds)
        val certificates: List<String>? = mPusherHelper?.getCertificatePinningKeys()

        val httpDeleteRequest = HttpDeleteRequest(requestURL,
            LPTraceType.PUSHER_DELETE_PENDING_PROACTIVE_MESSAGES_REQ)
        httpDeleteRequest.addHeader("Authorization", "Bearer $token")
        httpDeleteRequest.timeout = REQUEST_TIMEOUT
        httpDeleteRequest.certificatePinningKeys = certificates
        httpDeleteRequest.callback = object : ICallback<String, Exception> {
            override fun onSuccess(value: String?) {
                callback.onSuccess(true)
            }

            override fun onError(exception: Exception) {
                notifyError(exception)
            }
        }
        HttpHandler.execute(httpDeleteRequest)
    }

    private fun notifyError(exception: Exception) {
        LPLog.e(TAG, ErrorCode.ERR_0000016J, exception.message ?: "unknown")
        Infra.instance.postOnMainThread {
            if (exception is SSLPeerUnverifiedException) {
                controller.mEventsProxy.onError(LpError.INVALID_CERTIFICATE, exception.message)
            }
            callback.onError(exception)
        }
    }

    companion object {
        private const val TAG = "DeletePendingProactiveMessagesCommand"
        private const val REQUEST_TIMEOUT = 30000
        private const val ERROR_UNABLE_TO_MAKE_REQUEST = "Unable to $TAG."
        private const val DELETE_URL = "https://%1\$s/api/account/%2\$s/proactive/pending-messages?appId=%3\$s&consumerId=%4\$s&transactionIds=%5\$s&v=1"
    }
}
