package com.liveperson.messaging.commands.pusher

import com.liveperson.infra.Command
import com.liveperson.infra.ICallback
import com.liveperson.infra.Infra
import com.liveperson.infra.InternetConnectionService
import com.liveperson.infra.auth.LPAuthenticationParams
import com.liveperson.infra.errors.ErrorCode
import com.liveperson.infra.log.LPLog
import com.liveperson.infra.model.PushMessage
import com.liveperson.infra.utils.PushMessageParser
import com.liveperson.messaging.LpError
import com.liveperson.messaging.Messaging
import com.liveperson.messaging.model.SynchronizedAuthenticationCompletedCallback
import com.liveperson.messaging.model.SynchronizedInternetConnectionCallback
import com.liveperson.messaging.network.http.GetPendingProactiveMessagesRequest
import javax.net.ssl.SSLPeerUnverifiedException

class GetPendingProactiveMessagesCommand(
    private val controller: Messaging,
    private val brandId: String,
    private val appId: String,
    private val authParams: LPAuthenticationParams,
    private val callback: ICallback<List<PushMessage>, Exception>
) : Command {

    override fun execute() {
        if (appId.isNullOrBlank()) {
            val exceptionMsg = "$ERROR_UNABLE_TO_MAKE_REQUEST Error: Missing appID"
            callback.onError(Exception(exceptionMsg))
            notifyError(Exception(exceptionMsg))
            return
        }
        mPusherHelper = PusherHelper(brandId, controller)
        validateNetworkState()
    }

    private fun validateNetworkState() {
        if (InternetConnectionService.isNetworkAvailable()) {
            authorizeAndSendRequest()
        } else {
            SynchronizedInternetConnectionCallback { authorizeAndSendRequest() }.execute()
        }
    }

    private fun authorizeAndSendRequest() {
        val isExecuting = SynchronizedAuthenticationCompletedCallback(
            controller.mAccountsController,
            brandId,
            object : ICallback<Void, Exception> {
                override fun onSuccess(value: Void?) {
                    LPLog.d(TAG, "onSuccess")
                    sendRequest()
                }

                override fun onError(exception: Exception?) {
                    LPLog.e(TAG, ErrorCode.ERR_0000016G, "Failed when execute SynchronizedAuthenticationCompletedCallback")
                    notifyError(exception ?: Exception("Failed when execute SynchronizedAuthenticationCompletedCallback"))
                }
            }
        ).executeWithReturnValue()

        if (!isExecuting) {
            controller.connectForApiUsage(brandId, authParams, null, true)
        }
    }

    private fun sendRequest() {
        if (mPusherHelper == null) {
            mPusherHelper = PusherHelper(brandId, controller)
        }

        val pusherDomain = mPusherHelper?.getPusherDomain()
        if (pusherDomain.isNullOrBlank()) {
            notifyError(Exception("$ERROR_UNABLE_TO_MAKE_REQUEST Error: Missing Domain"))
            return
        }

        val consumerId = mPusherHelper?.getConsumerId()
        if (consumerId.isNullOrBlank()) {
            notifyError(Exception("$ERROR_UNABLE_TO_MAKE_REQUEST Error: Missing consumerID"))
            return
        }

        val token = mPusherHelper?.getAuthToken()
        if (token.isNullOrBlank()) {
            notifyError(Exception("$ERROR_UNABLE_TO_MAKE_REQUEST Error: Missing auth token"))
            return
        }

        val certificates: List<String>? = mPusherHelper?.getCertificatePinningKeys()
        val url = String.format(URL, pusherDomain, brandId, appId, consumerId, LIMIT)

        GetPendingProactiveMessagesRequest(url, token, certificates,
            object : ICallback<String?, Exception?> {
                override fun onSuccess(value: String?) {
                    if (!value.isNullOrBlank()) {
                        val pushMessages = parseResponse(value)
                        notifySuccess(pushMessages)
                    } else {
                        notifyError(Exception("null response"))
                    }
                }

                override fun onError(exception: Exception?) {
                    notifyError(exception ?: Exception("$ERROR_UNABLE_TO_MAKE_REQUEST"))
                }
            }
        ).execute()
    }

    private fun parseResponse(jsonString: String) : List<PushMessage> {
        var pushMessages = PushMessageParser.parsePendingProactiveMessages(brandId, controller.applicationContext, jsonString)

        return pushMessages
    }

    private fun notifySuccess(pushMessages: List<PushMessage>) {
        Infra.instance.postOnMainThread { callback.onSuccess(pushMessages) }
    }

    private fun notifyError(exception: Exception) {
        LPLog.e(TAG, ErrorCode.ERR_0000016G, exception.message!!)
        Infra.instance.postOnMainThread {
            if (exception is SSLPeerUnverifiedException) {
                controller.mEventsProxy.onError(LpError.INVALID_CERTIFICATE, exception.message)
            }
            callback.onError(exception)
        }
    }

    private var mPusherHelper: PusherHelper? = null
    companion object {
        private const val TAG = "GetPendingProactiveMessagesCommand"
        private const val ERROR_UNABLE_TO_MAKE_REQUEST = "Unable to GetPendingProactiveMessagesCommand."
        private const val LIMIT = 50
        private const val URL = "https://%1\$s/api/account/%2\$s/proactive/pending-messages?appId=%3\$s&consumerId=%4\$s&os=android&v=1&limit=%5\$s"
    }
}