package com.liveperson.messaging.commands.tasks

import android.content.Context
import com.liveperson.infra.ICallback
import com.liveperson.infra.log.LPLog.d
import com.liveperson.infra.messaging.R
import com.liveperson.infra.network.http.requests.CSDSRequest
import com.liveperson.messaging.LpError
import com.liveperson.messaging.TaskType
import com.liveperson.messaging.controller.AccountsController
import com.liveperson.messaging.model.AmsConnectionAnalytics
import java.util.*
import javax.net.ssl.SSLPeerUnverifiedException

/**
 * Base connection task to fetch CSDS
 */
class CsdsTask(context: Context, accountsController: AccountsController, brandId: String) : BaseAmsAccountConnectionTask() {

	companion object {
		private const val TAG = "CsdsTask"
	}

	private var csdsDomain: String? = null
	private val mAccountsController: AccountsController

	init {
		mBrandId = brandId
		csdsDomain = if (brandId.startsWith("qa") || brandId.startsWith("le")) {
			context.resources.getString(R.string.csds_qa_url)
		} else {
			context.resources.getString(R.string.csds_url)
		}
		mAccountsController = accountsController
	}

	override fun execute() {
		d(TAG, "Running CSDS task...")

		AmsConnectionAnalytics.csdsTaskStart()
		val certificates = mAccountsController.getCertificatePinningKeys(mBrandId)

		CSDSRequest(csdsDomain, mBrandId, certificates, object : ICallback<HashMap<String?, String?>?, Exception?> {

			override fun onSuccess(cSDSUrls: HashMap<String?, String?>?) {
				// Set the CSDS values to persistent memory. If one of the values is new, return error
				if (mAccountsController.setCSDSMap(mBrandId, cSDSUrls)) {
					d(TAG, "onSuccess: One of the CSDS values had changed. Return error")

					AmsConnectionAnalytics.csdsTaskEnd()
					mCallback.onTaskError(TaskType.CSDS, LpError.CSDS, Exception("New CSDS values were set."))

					return
				}

				AmsConnectionAnalytics.csdsTaskEnd()
				mCallback.onTaskSuccess()
			}

			override fun onError(e: Exception?) {
				if (e is SSLPeerUnverifiedException) {
					mCallback.onTaskError(TaskType.INVALID_CERTIFICATE, LpError.INVALID_CERTIFICATE, e)
				} else {
					mCallback.onTaskError(TaskType.CSDS, LpError.CSDS, e)
				}
			}
		}).execute()
	}

	override fun getName(): String {
		return TAG
	}
}