package com.liveperson.messaging.commands.tasks;

import android.text.TextUtils;
import android.util.SparseIntArray;

import com.liveperson.api.response.model.DialogData;
import com.liveperson.api.response.model.MultiDialog;
import com.liveperson.api.response.model.UserProfile;
import com.liveperson.api.response.types.ConversationState;
import com.liveperson.api.response.types.DialogState;
import com.liveperson.infra.ICallback;
import com.liveperson.infra.Infra;
import com.liveperson.infra.InternetConnectionService;
import com.liveperson.infra.LocalBroadcastReceiver;
import com.liveperson.infra.callbacks.AuthStateSubscription;
import com.liveperson.infra.log.FlowTags;
import com.liveperson.infra.log.LPLog;
import com.liveperson.infra.managers.ConsumerManager;
import com.liveperson.infra.model.Consumer;
import com.liveperson.infra.network.http.HttpException;
import com.liveperson.infra.network.http.HttpUtilsKt;
import com.liveperson.infra.otel.LPTelemetryManager;
import com.liveperson.infra.otel.LPTraceSpan;
import com.liveperson.infra.otel.LPTraceType;
import com.liveperson.messaging.Messaging;
import com.liveperson.messaging.SocketTaskType;
import com.liveperson.messaging.TaskExecutionState;
import com.liveperson.messaging.commands.BasicQueryMessagesCommand;
import com.liveperson.messaging.commands.QueryMessagesUMSCommand;
import com.liveperson.messaging.controller.ConnectionsController;
import com.liveperson.messaging.model.AmsConnection;
import com.liveperson.messaging.model.AmsDialogs;
import com.liveperson.messaging.model.Conversation;
import com.liveperson.messaging.model.ConversationData;
import com.liveperson.messaging.model.ConversationUtils;
import com.liveperson.messaging.model.Dialog;
import com.liveperson.messaging.model.DialogUtils;
import com.liveperson.messaging.model.MessagingUserProfile;
import com.liveperson.messaging.network.http.QueryMessagesINCACommand;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;

import static com.liveperson.infra.errors.ErrorCode.ERR_00000064;
import static com.liveperson.infra.errors.ErrorCode.ERR_00000065;
import static com.liveperson.infra.errors.ErrorCode.ERR_00000067;
import static com.liveperson.infra.errors.ErrorCode.ERR_00000068;
import static com.liveperson.infra.managers.ConsumerManager.getConsumerJWT;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

/**
 * Created by shiranr on 28/01/2016.
 */
public class FetchConversationManager {

    private static final String TAG = "FetchConversationManager";
    protected final Messaging mController;
    private ConversationUtils mConversationUtils;
    private DialogUtils mDialogUtils;
    private Set<String> usersUpdating = new HashSet<>();
    private int numConversationToUpdateUI;
    final private HashMap<String, LocalBroadcastReceiver> didDialogGetMessages = new HashMap<>();
    private static Map<String, Conversation> pendingConversationsToRefresh = new HashMap<>();
    private static final Map<String, TaskExecutionState> dialogRequestState = new ConcurrentHashMap<>();

    private final Map<Integer, Boolean> mRetrievalStatusesMap = new ConcurrentHashMap<>();

    public enum DATA_SOURCE{UMS, INCA}

    private final int NUM_OF_CONVERSATIONS_AT_FIRST_LOAD = 2;

    public FetchConversationManager(Messaging controller){
        mController = controller;
        init();
    }

    protected void init() {
        mConversationUtils = new ConversationUtils(mController);
        mDialogUtils = new DialogUtils(mController);
    }

    /**
     * Clear map which maintains conversation dialog request status after moving to background.
     */
    public static void clearDialogRequestStatusMap() {
        LPLog.INSTANCE.d(TAG, "clearDialogRequestStatusMap");
        dialogRequestState.clear();
    }

    public void fetchConversationsFirstTime(final String brandId, final List<ConversationData> dataFromUMS, List<ConversationData> dataFromInca){
        if (isCollectionEmpty(dataFromUMS) && isCollectionEmpty(dataFromInca)) {
            ConnectionsController controller = mController.mConnectionController;
            synchronized (controller) {
                AmsConnection connection = controller.getConnection(brandId);
                if (connection != null) {
                    boolean isNetworkEnabled = InternetConnectionService.isNetworkAvailable();
                    connection.setIsUpdated(isNetworkEnabled);
                    mController.amsConversations.setFailedToFetchConversationFirstTime(brandId, !isNetworkEnabled);
                }
            }
            mController.amsMessages.updateFetchHistoryEnded(true);
            return;
        }

        int numConversationToFetch = NUM_OF_CONVERSATIONS_AT_FIRST_LOAD;
        int umsSize = isCollectionEmpty(dataFromUMS) ? 0 : dataFromUMS.size();
        int incaSize = isCollectionEmpty(dataFromInca) ? 0 : dataFromInca.size();
        //for ex. - history has only 1 conversation and default numConversationToFetch is 2.
        if (umsSize + incaSize < numConversationToFetch){
            numConversationToFetch = umsSize + incaSize;
        }
        numConversationToUpdateUI = numConversationToFetch;


        if (numConversationToFetch > 0){
            final BlockingQueue<QueryRequestMessage> queue = new ArrayBlockingQueue<>(numConversationToFetch * 2);
            mRetrievalStatusesMap.put(queue.hashCode(), false);
            waitFetchConversationFinished(brandId, numConversationToFetch, queue, true, true);

            int i = 0;
            if (dataFromUMS != null) {
                //fetch from UMS
                for (ConversationData conversationData : dataFromUMS) {
                    // We don't fetch any closed conversations from UMS unless it is missing from INCA list.
                    // Additional condition: check data masking setting -> dataSource is INCA instead of UMS
                    if (conversationData.state == ConversationState.CLOSE && isConversationPresentInHistory(conversationData, dataFromInca) &&
                            mController.amsConversations.getDataSource(conversationData.endTs) == DATA_SOURCE.INCA) {
                        continue;
                    }
                    // In case the time to view unmasked data is still valid -> fetch closed conversation from UMS.
                    boolean shouldFetchMessages = i < numConversationToFetch;
                    LPLog.INSTANCE.d(TAG, "Saving conversation #" + i + " source: UMS. bringing messages: " + shouldFetchMessages);
                    fetchConversation(conversationData, shouldFetchMessages, queue, i, DATA_SOURCE.UMS);
                    i++;
                }
            }
            if (dataFromInca != null){
                //Saving conversations from INCA.
                //if we still need to fetch more conversation (shouldFetchMessages), we'll fetch it from INCA.
                for(ConversationData conv : dataFromInca) {
                    boolean shouldFetchMessages = i < numConversationToFetch;
                    LPLog.INSTANCE.d(TAG, "Saving conversation #" + i + " source: INCA. bringing messages: " + shouldFetchMessages);
                    fetchConversation(conv, shouldFetchMessages, queue, i, DATA_SOURCE.INCA);
                    i++;
                }
            }

            LPLog.INSTANCE.d(TAG, "Finished updating conversations in DB. waiting for query messages responses (if there is))");
        }else{
            int i = 0;
            //Saving conversations from UMS
            for(ConversationData conversationData : dataFromUMS){
                fetchConversation(conversationData,false,null, i , DATA_SOURCE.UMS);
                i++;
            }
            //Saving conversations from INCA.
            for(ConversationData conversationData : dataFromInca){
                fetchConversation(conversationData,false,null, i , DATA_SOURCE.INCA);
                i++;
            }
            mController.amsMessages
                    .updateOnCommand().setPreQueryOnBackground(() -> {
	                    LPLog.INSTANCE.d(TAG, "Finished updating conversations in DB. waiting for query messages responses (if there is))");
	                    if (null != mController.mConnectionController.getConnection(brandId)) {
                            mController.mConnectionController.getConnection(brandId).setIsUpdated(true);
                        }
                    }).execute();
        }

    }

    private boolean isCollectionEmpty(List<ConversationData> collection) {
        return collection == null || collection.size() == 0;
    }

    /**
     * Check If conversation from UMS is present in INCA conversations list
     * @param UMSConversation Conversation from list of UMS conversations
     * @param historyConversations List of INCA conversations
     * @return boolean
     */
    private boolean isConversationPresentInHistory(ConversationData UMSConversation, List<ConversationData> historyConversations) {
        if (UMSConversation == null) {
            return true;
        }
        if (historyConversations == null) {
            return false;
        }
        for (ConversationData conversation: historyConversations) {
            if (conversation.conversationId.equals(UMSConversation.conversationId)) {
                return true;
            }
        }
        LPLog.INSTANCE.i(TAG, "isConversationPresentInHistory: UMS conversation "
                + UMSConversation.conversationId + " is missing from INCA conversations list. Fetch it from UMS.");
        return false;
    }

    public void fetchConversation(Conversation conversation, ArrayList<Dialog> dialogs) {
        if (!mController.mConnectionController.isSocketReady(conversation.getBrandId())) {
	        LPLog.INSTANCE.i(TAG, "Brand is not connected. can't fetch dialog for " + conversation.getConversationId());
            mController.amsMessages.updateFetchHistoryEnded(false);
            return;
        }
        fetchConversation(conversation, dialogs, false, true);
    }

    /**
     * Perform previously incomplete update for a conversation
     * @param conversationId pending conversation to refresh
     */
    public void refreshPendingConversation(String conversationId) {
        if (conversationId == null || pendingConversationsToRefresh == null) {
            return;
        }
        Conversation conversation = pendingConversationsToRefresh.get(conversationId);
        if (conversation != null) {
            pendingConversationsToRefresh.remove(conversationId);
            refreshConversation(conversation);
        }
    }

    /**
     * Refresh conversation. It could be previously failed to update or recently closed conversation.
     * @param conversation conversation to update
     */
    public void refreshConversation(Conversation conversation) {
        ArrayList<Dialog> dialogs = mController.amsDialogs.getDialogsByConversationId(conversation.getConversationId());
        if (dialogs.size() == 0) {
            LPLog.INSTANCE.d(TAG, "refreshConversation: dialogs not found in memory, checking database...");
            mController.amsDialogs.queryDialogsByConversationId(conversation.getConversationId()).setPostQueryOnBackground(fetchedDialogs -> {
                if (fetchedDialogs != null && fetchedDialogs.size() > 0) {
                    LPLog.INSTANCE.d(TAG, "refreshConversation: dialogs found in database. Re-fetch conversation: " + conversation.getConversationId());
                    fetchConversation(conversation, fetchedDialogs, true, false);
                } else {
                    pendingConversationsToRefresh.put(conversation.getConversationId(), conversation);
                    LPLog.INSTANCE.e(TAG, ERR_00000064, "refreshConversation: Missing dialogs for conversation ID:" + conversation.getConversationId());
                    throw new RuntimeException("refreshConversation: Missing dialogs for conversation ID:" + conversation.getConversationId());
                }
            }).execute();
        } else {
            LPLog.INSTANCE.d(TAG, "refreshConversation: dialogs found in memory. Re-fetch conversation: " + conversation.getConversationId());
            fetchConversation(conversation, dialogs, true, false);
        }
    }

    private void fetchConversation(Conversation conversation, ArrayList<Dialog> dialogs, boolean forceUpdateAgent, boolean addResolveMessage) {
        final BlockingQueue<QueryRequestMessage> queue = new ArrayBlockingQueue<>(dialogs.size());

        ArrayList<Dialog> filteredDialogs = new ArrayList<>();
        // We prevent a fetch conversation request If one is already in progress for same dialogId.
        // This happens sometimes when we are initially loading conversations after log in.
        synchronized (this) {
            for (Dialog dialog : dialogs) {
                if (dialog != null && dialog.getDialogId() != null) {
                    // If request is already in progress, ignore dialog
                    if (dialogRequestState.get(dialog.getDialogId()) != TaskExecutionState.STARTED) {
                        filteredDialogs.add(dialog);
                        dialogRequestState.put(dialog.getDialogId(), TaskExecutionState.STARTED);
                    } else {
                        LPLog.INSTANCE.i(TAG, "fetchConversation: Ignore request for dialogId: : " + dialog.getDialogId());
                    }
                }
            }
        }
        mRetrievalStatusesMap.put(queue.hashCode(), false);
        waitFetchConversationFinished(conversation.getBrandId(), 1, queue, addResolveMessage, false);

        for (Dialog dialog : filteredDialogs) {
            if (dialog != null) {
                String assignedAgentId = dialog.getAssignedAgentId();
                numConversationToUpdateUI = 1;
                //after creating closed dialog, we can send query request & add resolved message
                ConversationData conversationData = new ConversationData(conversation.getBrandId(), conversation, filteredDialogs);

                DATA_SOURCE source;
                if (conversation.isConversationOpen()) {
                    //get it from UMS
                    source = DATA_SOURCE.UMS;
                } else {
                    source = DATA_SOURCE.INCA;
                }
                LPLog.INSTANCE.i(TAG, "Fetching dialog for " + conversation.getConversationId() + " sending request to query unread messages via " + source.name());
                queryMessages(conversationData, dialog, queue, 0, source);

                mConversationUtils.updateParticipants(conversation.getTargetId(), new String[]{assignedAgentId},
                        UserProfile.UserType.AGENT, dialog.getConversationId(), false, forceUpdateAgent);

            } else {
                LPLog.INSTANCE.e(TAG, ERR_00000065, "fetchConversation: Missing open dialog in conversation: " + conversation.getConversationId());
            }
        }
    }

    private void waitFetchConversationFinished(
            final String brandId,
            final int maxConversationHistory,
            final BlockingQueue<QueryRequestMessage> queue,
            final boolean addResolveMessage,
            final boolean firstTimeFetching
    ) {

        //because this thread is gonna be dedicated to messaging queue, we don't want it to take over a thread in ThreadPullExecutor for 15 seconds.
       new Thread(new Runnable() {
           LocalBroadcastReceiver mConnectionReceiver;

            @Override
            public void run() {
                final ArrayList<ICallback<Void, Exception>> resolveMessagesListeners = new ArrayList<>();
                boolean isTimedOut = false;
                QueryRequestMessage prevQueryRequestMessage = null;
                Exception occurredException = null;
                try {
                    registerNoConnectionEvents();

                    final SparseIntArray conversationsQueryMap = new SparseIntArray(maxConversationHistory);
                    boolean done = false;

                    while (!done) {
                        if (!InternetConnectionService.isNetworkAvailable()) {
                            LPLog.INSTANCE.d(TAG, "Can't fetch history from server. no network. ");
                            throw new IllegalStateException("Can't fetch history from server.  no network.");
                        }

                        QueryRequestMessage queryRequestMessage = queue.poll(5, TimeUnit.SECONDS);
                        if (queryRequestMessage != null) {

                            if (queryRequestMessage.conversationData != null && queryRequestMessage.conversationData.dialogs != null) {
                                DialogData[] dialogs = queryRequestMessage.conversationData.dialogs;
                                for (DialogData dialog : dialogs) {
                                    final String dialogId = dialog.dialogId;

                                    synchronized (didDialogGetMessages) {
                                        if (didDialogGetMessages.get(dialogId) == null) {
                                            LocalBroadcastReceiver broadcastReceiver = new LocalBroadcastReceiver.Builder()
                                                    .addAction(MessagingEventSubscriptionManager.Companion.getMESSAGE_EVENT_COMPLETED() + dialogId)
                                                    .build((context, intent) -> {
                                                        LocalBroadcastReceiver receiver = didDialogGetMessages.get(dialogId);
                                                        if (receiver != null) {
                                                            receiver.unregister();
                                                        }
                                                    });
                                            didDialogGetMessages.put(dialogId, broadcastReceiver);
                                        }
                                    }
                                }
                            }
                            LPLog.INSTANCE.d(TAG, "waitFetchConversationFinished: didDialogGetMessages == " + didDialogGetMessages.size());
                            // Save the request when it is added so it could be referenced in case of a time out.
                            // Time out event can happen as a result of the following: Network time out or an empty body response.
                            prevQueryRequestMessage = queryRequestMessage;
                        }

                        if (queryRequestMessage == null || queryRequestMessage.conversationIndex == -1) {
                            LPLog.INSTANCE.w(TAG, "Fetching history thread was interrupted or timeout expired");
                            //interrupt by no connection event
                            done = true;
                            isTimedOut = true;
                            occurredException = new NullPointerException();
                        } else {

                            int oldValue = conversationsQueryMap.get(queryRequestMessage.conversationIndex);
                            int newValue = oldValue + queryRequestMessage.value;
                            boolean success = queryRequestMessage.success;
                            conversationsQueryMap.put(queryRequestMessage.conversationIndex, newValue);
                            LPLog.INSTANCE.d(TAG, "query maps requests " + conversationsQueryMap +
                                    " new value = " + newValue +
                                    " addResolveMessage = " + addResolveMessage +
                                    " success = " + success);

                            if (newValue == 0) {//all query messages returned.

                                //adding resolve message after all query messages are back.
                                ConversationData conversationData = queryRequestMessage.conversationData;
                                if (success && addResolveMessage &&
                                        // TODO: check whether this condition is required or not
                                        containsClosedDialog(conversationData)) {

                                    //only if the conversation closed add resolve message
                                    final QueryRequestMessage finalQueryRequestMessage = prevQueryRequestMessage;
                                    ICallback<Void, Exception> listener = new ICallback<Void, Exception>() {

                                        @Override
                                        public void onSuccess(Void v) {
                                            checkIsDoneFetching();
                                        }

                                        @Override
                                        public void onError(Exception e) {
                                            LPLog.INSTANCE.w(TAG, "Failed adding resolve message.", e);
                                            checkIsDoneFetching();
                                        }

                                        private void checkIsDoneFetching() {
                                            resolveMessagesListeners.remove(this);
                                            //after finishing adding resolve message, we can check if we are done.
                                            if (resolveMessagesListeners.isEmpty() && conversationsQueryMap.size() == 0) {
                                                finishFetching(finalQueryRequestMessage, false);
                                            }
                                        }
                                    };

                                    resolveMessagesListeners.add(listener);
                                    addResolveMessageToClosedConversation(queryRequestMessage, listener);
                                }

                                //check if all query conversations returned.
                                done = isDone(conversationsQueryMap);
                            }
                        }
                    }
                    //when we are done - clear map. will be helpful when resolve messages are done.
                    conversationsQueryMap.clear();

                } catch (Exception e) {
                    LPLog.INSTANCE.w(TAG, "Failed fetching messages from history: ", e);
                    isTimedOut = true;
                    occurredException = e;
                } finally {
                    mRetrievalStatusesMap.put(queue.hashCode(), true);
                    usersUpdating.clear();
                    if (mConnectionReceiver != null) {
                        mConnectionReceiver.unregister();
                    }
                    mController.amsConversations.setFailedToFetchConversationFirstTime(brandId, firstTimeFetching && occurredException != null);
                    if (!resolveMessagesListeners.isEmpty()) {
                        //wait for resolve messages to finished.
                        LPLog.INSTANCE.d(TAG, "Finished fetching messages from history but we are still waiting for Resolve messages to be added to DB...");
                    } else {
                        finishFetching(prevQueryRequestMessage, isTimedOut);
                    }
                }
            }

            protected boolean isDone(SparseIntArray conversationsQueryMap) {
                //check if all query conversations returned.
                if (conversationsQueryMap.get(maxConversationHistory - 1, -1) == 0) {

                    //last conversation exists and it's 0
                    //check if we have all the keys for all the conversation and all the values equals 0;
                    for (int j = 0; j < maxConversationHistory; j++) {
                        int value = conversationsQueryMap.get(j, -1);
                        if (value != 0) {
                            return false;
                        }

                        //check if all the responses are back
                        if (j == maxConversationHistory - 1) {
                            return true;
                        }
                    }
                }

                return false;
            }

            protected void finishFetching(QueryRequestMessage queryRequestMessage, boolean timedOut) {
                // if we are not updated already (if we had 10 conversations - we load only x (2)
                // from history, when those 2 finished we will update the ui already )
                if (null != mController.mConnectionController.getConnection(brandId)
                        && !mController.mConnectionController.getConnection(brandId).isUpdated()
                        && mController.mConnectionController.isSocketReady(brandId)) {
                    mController.mConnectionController.getConnection(brandId).setIsUpdated(true);
                }
                updateFetchHistoryListener(queryRequestMessage, timedOut);

                LPLog.INSTANCE.d(TAG, "Finished fetching messages from history ! ");
            }

            private void registerNoConnectionEvents() {
                mConnectionReceiver = new LocalBroadcastReceiver.Builder()
                        .addAction(AmsConnection.BROADCAST_KEY_SOCKET_READY_ACTION)
                        .build((context, intent) -> {
                            boolean isConnected = intent.getBooleanExtra(AmsConnection.BROADCAST_KEY_SOCKET_READY_EXTRA, false);
                            if (!isConnected) {
                                if (mConnectionReceiver != null) {
                                    mConnectionReceiver.unregister();
                                }
                                //break from running process.
                                queue.add(new QueryRequestMessage(-1, -1, null, false));
                            }
                        });
            }

            private boolean containsClosedDialog(ConversationData conversationData) {
                if (conversationData.state == ConversationState.CLOSE) {
                    return true;
                }
                ArrayList<Dialog> conversationDialogs = AmsDialogs.extractDialogs(conversationData);
                for (Dialog dialog : conversationDialogs) {
                    if (dialog.getState() == DialogState.CLOSE) {
                        return true;
                    }
                }
                return false;
            }

            private void addResolveMessageToClosedConversation(QueryRequestMessage queryRequestMessage, ICallback<Void, Exception> listener) {
                ConversationData conversationData = queryRequestMessage.conversationData;

                LPLog.INSTANCE.d(TAG, "Adding resolve message to " + conversationData.conversationId + " index = " + queryRequestMessage.conversationIndex + " numConversationToUpdateUI = " + numConversationToUpdateUI);

                ArrayList<Dialog> conversationDialogs = AmsDialogs.extractDialogs(conversationData);
                for (Dialog dialog : conversationDialogs) {
                    if (dialog.getState() == DialogState.CLOSE) {
                        mDialogUtils.addClosedDialogDivider(conversationData.targetId, dialog, conversationData.getAssignedAgentId(), dialog.getCloseReason(),
                                //update ui only if conversation is beyond the range of numConversationToUpdateUI
                                queryRequestMessage.conversationIndex >= numConversationToUpdateUI, listener);
                    }
                }
            }
        }).start();
    }

    private void updateFetchHistoryListener(QueryRequestMessage queryRequestMessage, boolean timedOut) {
        if (timedOut) {
            synchronized(didDialogGetMessages) {
                for (String dialogId : didDialogGetMessages.keySet()) {
                    LocalBroadcastReceiver broadcastReceiver = didDialogGetMessages.get(dialogId);
                    if (broadcastReceiver != null) {
                        // Time out reached. Update the relevant dialog that it has 0 messages.
                        if (broadcastReceiver.isRegistered()) {
                            LPLog.INSTANCE.d(TAG, "updateFetchHistoryListener: dialog ID has no messages: " + dialogId);
                            handleEmptyDialog(queryRequestMessage, dialogId, new BaseAmsSocketConnectionCallback() {

                                @Override
                                public void onTaskSuccess() {
                                    mController.amsMessages.updateFetchHistoryEnded(true);
                                }

                                @Override
                                public void onTaskError(SocketTaskType type, Throwable exception) {
                                    mController.amsMessages.updateFetchHistoryEnded(true);
                                }
                            });
                        }
                        broadcastReceiver.unregister();
                    }
                }
                didDialogGetMessages.clear();
            }
        } else {
            mController.amsMessages.updateFetchHistoryEnded(true);
        }
    }

    /**
     * Update the relevant dialog that it has no messages
     * */
    private void handleEmptyDialog(QueryRequestMessage queryRequestMessage, String dialogId, BaseAmsSocketConnectionCallback callback) {

        if (queryRequestMessage != null && queryRequestMessage.conversationData != null) {
            // Log ERR_00000066 as INFO log. Check more here: INC0048106
            LPLog.INSTANCE.i(TAG, FlowTags.DIALOGS, "QueryRequest timed out - ERR_00000066");

            UpdateEmptyDialogCommand command = new UpdateEmptyDialogCommand(mController,
                    queryRequestMessage.conversationData.brandId,
                    queryRequestMessage.conversationData.conversationId,
                    dialogId,
                    true);

            command.setResponseCallBack(callback);
            command.execute();
        }
    }

    /**
     * We are getting data WITHOUT updating ui. will be updated after all data will received from server.
     * @param data
     * @param queryMessages
     * @param queue
     * @param conversationIndex
     * @param source
     */
    private void fetchConversation(final ConversationData data, final boolean queryMessages, final BlockingQueue<QueryRequestMessage> queue, final int conversationIndex, final DATA_SOURCE source){
        switch (data.state) {
            case CLOSE: {
                mController.amsConversations.updateClosedConversation(data, false).setPostQueryOnBackground(updatedConversation -> {
                    ArrayList<Dialog> dialogs = AmsDialogs.extractDialogs(data);
                    for (Dialog dialog: dialogs) {
                        mController.amsDialogs.updateClosedDialog(data, dialog, false).executeSynchronously();
                        if (queryMessages) {
                            if (updatedConversation != null) {
                                //after creating closed dialog, we can send query request & add resolved message
	                            LPLog.INSTANCE.d(TAG, "Sending request to query unread messages... newer than sequence: " + updatedConversation.getLastServerSequence() +" source = " + source);
                                queryMessages(data, dialog, queue, conversationIndex, source);
                            } else {
                                //dialog already exists and closed. notifying it's already handled.
                                addRequestToQueue(queue, conversationIndex, data, 0);
                            }
                        }
                    }

                    updateAllUsersRelatedToConversation(data, queryMessages, queue, conversationIndex);
                }).execute();
            }
            break;
            case OPEN: {
                mController.amsConversations.createNewCurrentConversation(data);
                ArrayList<Dialog> dialogs = AmsDialogs.extractDialogs(data);
                Dialog dialog = AmsDialogs.getOpenDialog(dialogs);
                for (Dialog d: dialogs) {
                    //add dialogId into database dialog table for all dialogs in the open conversation.
                    mController.amsDialogs.createNewCurrentDialog(d);
                }

                mController.amsDialogs.setActiveDialog(dialog);
				// This is implemented this way since in next version we will need to revert it, so left updateTTR the same
                long effectiveTTR = mConversationUtils.calculateEffectiveTTR(data.brandId, data.ttrValue, data.manualTTR ,data.delayTillWhen);
                mConversationUtils.updateTTR(data.conversationTTRType, effectiveTTR, data.brandId);
                if (dialog != null) {
	                LPLog.INSTANCE.d(TAG, "We have a new Current Dialog! " + dialog.getDialogId() + ". Sending request to query messages and update assigned agent details");
                }
                updateAllUsersRelatedToConversation(data, queryMessages, queue, conversationIndex);
                if(queryMessages) {
                    queryMessages(data, dialog, queue, conversationIndex, source);
                }
            }
            break;
        }
    }

    private void updateAllUsersRelatedToConversation(ConversationData data, boolean isBringingData, final BlockingQueue<QueryRequestMessage> queue, int conversationIndex) {

        for (String user : data.participants.ALL_AGENTS) {
            getDataForUser(data, user, UserProfile.UserType.AGENT, isBringingData, queue, conversationIndex);
        }
        for (String user : data.participants.CONTROLLER) {
            getDataForUser(data, user, UserProfile.UserType.CONTROLLER, isBringingData, queue, conversationIndex);
        }
    }

    private void getDataForUser(final ConversationData data, String user, UserProfile.UserType userType, boolean isBringingData, final BlockingQueue<QueryRequestMessage> queue, final int conversationIndex) {
        if (!usersUpdating.contains(user)) {
            usersUpdating.add(user);

            //we are updated with our assigned agent. updating other agents (reader/manager type)
            //we send null as the conversation id cause we don't want to relate the agent details to a specific conversation.
            ICallback<MessagingUserProfile, Exception> callback = null;

            //the first conversation that brings the agent details is responsible for it, all the other after it will be updated automatically
            if (isBringingData){
                LPLog.INSTANCE.d(TAG, "Bringing user data for conversation index: " + conversationIndex +" agent: " + user);
                addRequestToQueue(queue, conversationIndex, data, 1);
                callback = new ICallback<MessagingUserProfile, Exception>() {
                    @Override
                    public void onSuccess(MessagingUserProfile value) {
                        LPLog.INSTANCE.d(TAG, "onSuccess Bringing user data for conversation index: " + conversationIndex);

                        addResponseToQueue(queue, conversationIndex, data, -1, true);
                    }

                    @Override
                    public void onError(Exception exception) {
                        LPLog.INSTANCE.d(TAG, "onError Bringing user data for conversation index: " + conversationIndex);

                        //we send success 'true' because we don't want to block the operation if an error occurred trying to bring agent details.
                        addResponseToQueue(queue, conversationIndex, data, -1, true);
                    }
                };
            }


            mConversationUtils.updateParticipants(data.targetId, new String[]{user},userType, null,false, true, callback);
        }
    }

    private void queryMessages(final ConversationData conversationData, Dialog dialog, final BlockingQueue<QueryRequestMessage> queue, final int conversationIndex, DATA_SOURCE source) {
        final BasicQueryMessagesCommand command;
        LPTraceSpan traceSpan = null;
        if (dialog.getChannelType() == MultiDialog.ChannelType.COBROWSE ) {
            command = new QueryCobrowseDialogStateCommand(mController, conversationData.brandId, conversationData, dialog, false);
        } else if (source.equals(DATA_SOURCE.UMS)) {
            command = new QueryMessagesUMSCommand(mController, conversationData.brandId, conversationData.conversationId, dialog.getDialogId(), dialog.getLastServerSequence(), false);
            mController.amsConversations.addUMSConversationId(conversationData.brandId, conversationData.conversationId);
            traceSpan = LPTelemetryManager.INSTANCE.begin(LPTraceType.UMS_GET_CONV_MESSAGES);
        } else {
            if (Conversation.TEMP_CONVERSATION_ID.equals(conversationData.conversationId) || Dialog.TEMP_DIALOG_ID.equals(dialog.getDialogId())) {
                return;
            }
            conversationData.source = source;
            command = new QueryMessagesINCACommand(mController, conversationData.brandId, conversationData.conversationId, dialog.getDialogId(),false);
        }

        LPTraceSpan finalTraceSpan = traceSpan;
        command.setResponseCallBack(new BaseAmsSocketConnectionCallback() {
            @Override
            public void onTaskSuccess() {
                if (finalTraceSpan != null) {
                    finalTraceSpan.end();
                }
                dialogRequestState.put(dialog.getDialogId(), TaskExecutionState.SUCCESS);
                addResponseToQueue(queue, conversationIndex, conversationData, -1, true);
                if (conversationData.source == DATA_SOURCE.INCA) {
                    mController.amsConversations.removeUMSConversationId(conversationData.brandId, conversationData.conversationId);
                }
            }

            @Override
            public void onTaskError(SocketTaskType type, Throwable exception) {
                if (finalTraceSpan != null) {
                    finalTraceSpan.cancel();
                }
                if (HttpUtilsKt.isTokenExpired(exception)) {
                    setupFailedQueryCommand(command, exception, (command1, error) -> {
                        dialogRequestState.put(dialog.getDialogId(), TaskExecutionState.FAILURE);
                        addResponseToQueue(queue, conversationIndex, conversationData, -1, false);
                    });
                } else {
                    dialogRequestState.put(dialog.getDialogId(), TaskExecutionState.FAILURE);
                    addResponseToQueue(queue, conversationIndex, conversationData, -1, false);
                }
            }
        });

        addRequestToQueue(queue, conversationIndex, conversationData, 1);
        command.execute();
    }

    /**
     * Method used to handle situation when Inca/Ums query fails.
     *
     * @param command   - failed instance of command to restart if jwt is expired.
     * @param exception - cause of command's failure
     * @param callback  - callback, when cause of command failure is not jwt expiration
     *
     * @see FetchConversationManager#queryMessages(ConversationData, Dialog, BlockingQueue, int, DATA_SOURCE)
     */
    private void setupFailedQueryCommand(
            BasicQueryMessagesCommand command,
            Throwable exception,
            OnTaskFailedCallback callback
    ) {

        HttpException cause = (HttpException) exception;
        final ConsumerManager manager = Infra.instance.getConsumerManager();
        AuthStateSubscription subscription = new AuthStateSubscription() {
            @Override
            public void onAuthStateChanged(
                    @NonNull ConsumerManager.AuthState oldState,
                    @NonNull ConsumerManager.AuthState newState,
                    @Nullable Consumer oldConsumer,
                    @Nullable Consumer newConsumer
            ) {
                boolean shouldRestart = oldState.equals(ConsumerManager.AuthState.AUTH_IN_PROGRESS);
                shouldRestart &= newState.equals(ConsumerManager.AuthState.AUTHENTICATED);
                shouldRestart &= getConsumerJWT(oldConsumer) != null;
                shouldRestart &= getConsumerJWT(newConsumer) != null;
                shouldRestart &= !TextUtils.equals(getConsumerJWT(oldConsumer), getConsumerJWT(newConsumer));

                boolean shouldFailed = newState.equals(ConsumerManager.AuthState.AUTH_FAILED);
                if (shouldRestart) {
                    manager.unsubscribeFromAuthStateChanges(this);
                    command.execute();
                } else if (shouldFailed) {
                    manager.unsubscribeFromAuthStateChanges(this);
                    callback.onFailed(command, exception);
                }
            }
        };
        manager.handleTokenExpiration(cause, () -> {
            mController.mEventsProxy.onTokenExpired();
            return null;
        });
        manager.subscribeToAuthStateChanges(subscription);
    }

    private void addRequestToQueue(BlockingQueue<QueryRequestMessage> queue, int conversationIndex, ConversationData conversationData, int value) {
        try {
            LPLog.INSTANCE.d(TAG, "#" + conversationIndex + " Adding " + value + " to queue");
            QueryRequestMessage requestMessage = new QueryRequestMessage(
                    conversationIndex,
                    value,
                    conversationData,
                    true
            );
            addToQueueOrThrowException(queue, requestMessage);
        } catch (InterruptedException e) {
            LPLog.INSTANCE.e(TAG, ERR_00000067, "#" + conversationIndex + " Problem adding to query messages queue", e);
        }
    }

    private void addResponseToQueue(BlockingQueue<QueryRequestMessage> queue, int conversationIndex, ConversationData conversationData, int value, boolean success) {
        try {
            LPLog.INSTANCE.d(TAG, "#" + conversationIndex + " Adding " + value + " to queue");
            QueryRequestMessage responseMessage = new QueryRequestMessage(
                    conversationIndex,
                    value,
                    conversationData,
                    success
            );
            addToQueueOrThrowException(queue, responseMessage);
        } catch (InterruptedException e) {
            LPLog.INSTANCE.e(TAG, ERR_00000068, "#" + conversationIndex + " Problem adding to query messages queue", e);
        }
    }

    /**
     * Method used to add request ore response message to queue while history retrieval.
     * This method suspends thread, but doesn't block it if queue is busy or could not emit
     * new messages.
     * If history retrieval failed or finished for provided queue this method will throw exception.
     *
     * @param queue - blocking queue used to synchronize conversation messages requests.
     * @param message - message used to notify about starting/ending of query.
     * @throws InterruptedException if required queue doesn't emit any messages.
     */
    private void addToQueueOrThrowException(BlockingQueue<QueryRequestMessage> queue, QueryRequestMessage message) throws InterruptedException {
        checkRetrievalCompletion(queue);
        while (!queue.offer(message, 500, TimeUnit.MILLISECONDS)) {
            checkRetrievalCompletion(queue);
        }
    }

    /**
     * Method used to check whether queue is available to emit data.
     * If exception was thrown in {@link #waitFetchConversationFinished(String, int, BlockingQueue, boolean, boolean)} }
     * this method will also throw an Exception.
     *
     * @param queue queue of requests and responses used for synchronization.
     * @throws InterruptedException if fetching of conversation finished
     */
    private void checkRetrievalCompletion(BlockingQueue<QueryRequestMessage> queue) throws InterruptedException {
        Boolean isSessionCompleted = mRetrievalStatusesMap.get(queue.hashCode());
        if (isSessionCompleted == null || isSessionCompleted) {
            throw new InterruptedException("Query session was already finished");
        }
    }

    /**
     * Internal callback that will be used for failed inca request to receive more
     * data.
     *
     * @see FetchConversationManager#queryMessages(ConversationData, Dialog, BlockingQueue, int, DATA_SOURCE)
     * @see FetchConversationManager#setupFailedQueryCommand(BasicQueryMessagesCommand, Throwable, OnTaskFailedCallback)
     */
    private interface OnTaskFailedCallback {

        /**
         * Method used to handle situation when Inca/Ums query command finished with failure.
         *
         * @param command - failed command (Note: error could not be JWT-expiration)
         * @param error   - cause of failed command
         */
        void onFailed(BasicQueryMessagesCommand command, Throwable error);
    }

    private class QueryRequestMessage {
        int conversationIndex;
        int value;
        boolean success;
        ConversationData conversationData;

        public QueryRequestMessage(int conversationIndex, int value, ConversationData conversationData, boolean success) {
            this.conversationIndex = conversationIndex;
            this.value = value;
            this.conversationData = conversationData;
            this.success = success;
        }
    }
}
