package com.liveperson.messaging.commands.tasks

import android.content.Context
import android.text.TextUtils
import com.liveperson.infra.ICallback
import com.liveperson.infra.Infra
import com.liveperson.infra.log.LPLog.d
import com.liveperson.infra.log.LPLog.w
import com.liveperson.messaging.LpError
import com.liveperson.messaging.MessagingFactory
import com.liveperson.messaging.TaskType
import com.liveperson.messaging.controller.AccountsController
import com.liveperson.messaging.controller.connection.ConnectionParamsCache
import com.liveperson.messaging.model.AmsConnectionAnalytics
import com.liveperson.messaging.network.http.GetActiveConversationsRequest
import org.json.JSONObject

/**
 * Wrapper task for GetActiveConversationRequest.
 */
open class GetActiveConversationsUmsTask(
    val context: Context,
    val accountsController: AccountsController,
    var umsDomain: String?,
    brandId: String,
) : BaseAmsAccountConnectionTask() {

    companion object {
        private const val TAG = "GetActiveConversationTask"
    }

    var result: MutableList<String>? = null
    var getActiveConversationsRequest: GetActiveConversationsRequest? = null
    var consumerId: String? = null
    var lpToken: String? = null

    init {
        mBrandId = brandId
    }

    override fun execute() {
        if (consumerId == null) {
            consumerId = Infra.instance.consumerManager.getActiveConsumer()?.consumerId
        }
        if (lpToken == null) {
            lpToken = Infra.instance.consumerManager.getActiveConsumer()?.lpToken
        }
        if (umsDomain == null){
            umsDomain = MessagingFactory.getInstance().controller.mAccountsController
                .getServiceUrl(
                    mBrandId,
                    ConnectionParamsCache.CSDS_UMS_DOMAIN_KEY
                )
        }

        val domain = umsDomain ?: return notifyError("Failed to receive umsDomain")
        val brand = mBrandId ?: return notifyError("Failed to receive mBrandId")
        val consumer = consumerId ?: return notifyError("Failed to receive ums consumerId")
        val token = lpToken ?: return notifyError("Failed to receive ums lpToken")
        d(TAG, "Running GetActiveConversations Task...")
        AmsConnectionAnalytics.getActiveConversationTaskStart()
        if (getActiveConversationsRequest == null) {
            val data = GetActiveConversationsRequest.GetActiveConversationsRequestData(
                baseURL = domain,
                siteId = brand,
                consumerId = consumer,
                lpToken = token,
                certificates = accountsController.getCertificatePinningKeys(mBrandId)
            )
            getActiveConversationsRequest = GetActiveConversationsRequest(
                data,
                callback = prepareCallback()
            )
        }
        getActiveConversationsRequest?.execute()
    }

    private fun notifyError(message: String) {
        w(TAG, message)
        callback.onTaskError(TaskType.GET_ACTIVE_CONVERSATIONS, LpError.GET_ACTIVE_CONVERSATIONS_FAILURE, Exception(message))
    }

    private fun prepareCallback(): ICallback<String?, Exception?>{
        return object : ICallback<String?, Exception?> {
            override fun onSuccess(value: String?) {
                if (!TextUtils.isEmpty(value)) {
                    result = mutableListOf()
                    try {
                        val jsonObject = JSONObject(value!!)
                        val array = jsonObject.getJSONArray("data")
                        if (array.length() > 0) {
                            for (i in 0 until array.length()) {
                                array.optJSONObject(i)?.getString("id")?.let { result?.add(it) }
                            }
                        }
                        callback.onTaskSuccess()
                    } catch (ex: Exception) {
                        onError(ex)
                    }

                } else {
                    onError(Exception("GetActiveConversations task failure. Value: $value"))
                }
                AmsConnectionAnalytics.getActiveConversationTaskEnd()
            }

            override fun onError(exception: Exception?) {
                callback.onTaskError(
                    TaskType.GET_ACTIVE_CONVERSATIONS,
                    LpError.GET_ACTIVE_CONVERSATIONS_FAILURE,
                    exception
                )
                AmsConnectionAnalytics.getActiveConversationTaskEnd()
            }
        }
    }

    override fun getName(): String {
        return TAG
    }
}