package com.liveperson.messaging.commands.tasks

import com.liveperson.infra.ICallback
import com.liveperson.infra.errors.ErrorCode
import com.liveperson.infra.log.LPLog
import com.liveperson.infra.network.socket.SocketManager
import com.liveperson.messaging.LpError
import com.liveperson.messaging.Messaging
import com.liveperson.messaging.SocketTaskType
import com.liveperson.messaging.model.ConversationField
import com.liveperson.messaging.network.socket.requests.UpdateConversationFieldRequest

/**
 * This task is intended to be executed when performing step up authentication.
 * The purpose is to Update participant of a conversation to successfully perform step up.
 * We remove old consumer id (from UnAuth conversation) and add new one (to Auth conversation) with same conversation id.
 */
class UpdateConversationFieldTask(val controller: Messaging) : BaseAmsSocketConnectionTask() {
    private val TAG = "UpdateConversationFieldTask"
    private val OPEN_CONVERSATION_ERROR = "Failed to update conversation field, response: Consumer step up failed.  User already has open conversation"

    private val PARTICIPANTS_CHANGE_FIELD = "ParticipantsChange"
    private val TYPE_ADD = "ADD"
    private val TYPE_REMOVE = "REMOVE"
    private val CONSUMER = "CONSUMER"

    private var conversationId: String? = null

    override fun execute() {
        LPLog.i(TAG, "Running UpdateConversationFieldTask task...")
        val performStepUp =
            controller.mAccountsController.getAccount(mBrandId)?.isPerformStepUp

        if (performStepUp == false) {
            mCallback.onTaskSuccess()
            LPLog.d(TAG, "Step up disabled. No request")
            return
        }

        LPLog.i(TAG, "Step up is enabled. Send request to update conversation participants")

        val conversationFieldList = generateConversationFieldArray()
        val socketUrl = controller.mAccountsController.getConnectionUrl(mBrandId)

        if (conversationFieldList.isNotEmpty() && !socketUrl.isNullOrEmpty() && !conversationId.isNullOrEmpty()) {
            val conversationFieldRequest =
                UpdateConversationFieldRequest(socketUrl, conversationId!!, conversationFieldList)

            conversationFieldRequest.setResponseCallBack(object : ICallback<String?, Throwable?> {
                override fun onSuccess(value: String?) {
                    LPLog.d(TAG, "Successfully updated conversation fields: $value")
                    controller.mAccountsController.getAccount(mBrandId)?.isPerformStepUp = false
                    mCallback.onTaskSuccess()
                }

                override fun onError(exception: Throwable?) {
                    LPLog.e(TAG, ErrorCode.ERR_00000164, "Failed to update conversation field, response: $exception")
                    controller.mAccountsController.getAccount(mBrandId)?.isPerformStepUp = false

                    // If Update Conversation Fields task is failed with valid error, we ignore the step-up.
                    if (!exception?.message.isNullOrEmpty() &&
                        sanitizeText(exception?.message) == sanitizeText(OPEN_CONVERSATION_ERROR)) {
                        controller.mConnectionController?.notifySocketTaskFailure(mBrandId, LpError.STEP_UP_FAILURE, exception)
                        mCallback.onTaskSuccess()
                        return
                    }

                    mCallback.onTaskError(SocketTaskType.UPDATE_CONVERSATION_FIELD, exception)
                }

            })
            LPLog.d(TAG, "Updating conversation participants")
            SocketManager.getInstance().send(conversationFieldRequest)
        } else {
            LPLog.w(
                TAG,
                "Cannot update conversation field. conversationFieldList size: ${conversationFieldList.size}, socketUrl: $socketUrl, conversationId: $conversationId"
            )
            controller.mConnectionController?.notifySocketTaskFailure(mBrandId, LpError.STEP_UP_FAILURE,
                Exception("Failed to update conversation field. Missing data."))
            mCallback.onTaskSuccess()
        }
    }

    /**
     * Generate an array of 'ConversationField' based on consumer id to add vs remove from a conversation.
     */
    private fun generateConversationFieldArray(): ArrayList<ConversationField> {
        val conversationFieldList = ArrayList<ConversationField>()

        try {
            val oldConsumerId = controller.amsUsers?.stepUpConsumerId
            val newConsumerId = controller.amsUsers?.getConsumerId(mBrandId)

            val cachedConversationId =
                controller.amsConversations?.getConversationIdInPreferences(mBrandId)

            if (!oldConsumerId.isNullOrEmpty() && !newConsumerId.isNullOrEmpty() && !cachedConversationId.isNullOrEmpty()) {
                this.conversationId = cachedConversationId

                conversationFieldList.add(
                    ConversationField(
                        PARTICIPANTS_CHANGE_FIELD,
                        TYPE_REMOVE,
                        oldConsumerId,
                        CONSUMER
                    )
                )
                conversationFieldList.add(
                    ConversationField(
                        PARTICIPANTS_CHANGE_FIELD,
                        TYPE_ADD,
                        newConsumerId,
                        CONSUMER
                    )
                )
            } else {
                LPLog.w(
                    TAG,
                    "Missing data to form ConversationField. oldConsumerId: $oldConsumerId, newConsumerId: $newConsumerId, cachedConversationId: $cachedConversationId"
                )
            }
        } catch (exception: Exception) {
            LPLog.w(TAG, "Failed to get data to form ConversationField. $exception")
        }

        return conversationFieldList
    }

    /**
     * Remove white spaces from a text
     */
    private fun sanitizeText(text: String?): String {
        if (text.isNullOrEmpty()) {
            return ""
        }
        return text.replace("\\s".toRegex(), "") // remove white spaces
    }
}