package com.liveperson.messaging.controller;

import android.net.Uri;
import android.text.TextUtils;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.liveperson.infra.Clearable;
import com.liveperson.infra.Infra;
import com.liveperson.infra.auth.LPAuthenticationParams;
import com.liveperson.infra.auth.LPAuthenticationType;
import com.liveperson.infra.errors.ErrorCode;
import com.liveperson.infra.log.LPLog;
import com.liveperson.messaging.controller.connection.ConnectionParamsCache;
import com.liveperson.messaging.controller.connection.IConnectionParamsCache;
import com.liveperson.messaging.model.AmsAccount;

import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import kotlin.text.StringsKt;

/**
 * Created by ofira on 07/12/2015.
 */
public class AccountsController implements Clearable{

	private static final String TAG = "AccountsController";

    private static final String SOCKET_URL = "wss://%1$s/ws_api/account/%2$s/messaging/consumer";
    private static final String DOMAIN_VERSION_QUERY = "v";
    private static final int DOMAIN_VERSION_VALUE = 3;

	private Map<String, AmsAccount> mAccounts;
    private ClientProperties mClientProperties;


    public AccountsController(ClientProperties clientProperties) {
        mAccounts = new HashMap<>();
        mClientProperties = clientProperties;
    }


	/**
     * @param brand
     */
    public void addNewAccount(String brand) {
    	AmsAccount account = getAccount(brand);
        if (account == null) {
            LPLog.INSTANCE.i(TAG, "Adding new account: " + brand);
            mAccounts.put(brand, new AmsAccount(brand));
        }
    }


    /**
     * @param brand
     * @return
     */
    public String getToken(String brand) {
        AmsAccount account = getAccount(brand);
        if (account == null) {
			LPLog.INSTANCE.e(TAG, ErrorCode.ERR_00000162, "getToken: getAccount returns null for brand with id: " + brand);
			return null;
        }
        return account.getToken();
    }

    /**
     * Certificate Pinning public key
     * @param brand
     * @return
     */
    public List<String> getCertificatePinningKeys(String brand) {
        AmsAccount account = getAccount(brand);
        if (account == null) {
			LPLog.INSTANCE.e(TAG, ErrorCode.ERR_00000162, "getCertificatePinningKeys: getAccount returns null for brand with id: " + brand);
			return Collections.emptyList();
        }
        return account.getCertificatePinningKeys();
    }


    /**
     * @param brand
     * @return
     */
    public boolean isTokenExpired(String brand) {
        AmsAccount account = getAccount(brand);
        if (account == null) {
			LPLog.INSTANCE.e(TAG, ErrorCode.ERR_00000162, "isTokenExpired: getAccount returns null for brand with id: " + brand);
			return false;
        }
        return account.isTokenExpired() || account.isHostAppJWTExpired();
    }

    /**
     * @param brand
     * @return
     */
    public LPAuthenticationParams getLPAuthenticationParams(String brand) {
        AmsAccount account = getAccount(brand);
        if (account == null) {
			LPLog.INSTANCE.e(TAG, ErrorCode.ERR_00000162, "getLPAuthenticationParams: getAccount returns null for brand with id: " + brand);
			return null;
        }
        return account.getLPAuthenticationParams();
    }

    /**
     * @param brand
     * @return
     */
    public void setLPAuthenticationParams(String brand, LPAuthenticationParams lpAuthenticationParams) {
        AmsAccount account = getAccount(brand);
        if (account == null) {
			LPLog.INSTANCE.e(TAG, ErrorCode.ERR_00000162, "setLPAuthenticationParams: getAccount returns null for brand with id: " + brand);
			return;
        }
        resetAuthStateIfNeeded(account, lpAuthenticationParams);
        account.setLPAuthenticationParams(lpAuthenticationParams);
    }

	/**
	 * Set CSDS values to persistent memory
	 * @param brand
	 * @param csdsUrls
	 * @return true - if one of the relevant values were changed, false - if non were changed
	 */
    public boolean setCSDSMap(String brand, HashMap<String, String> csdsUrls) {

		boolean dataUpdatedWithNewValue;

		AmsAccount account = getAccount(brand);
        if (account == null) {
			LPLog.INSTANCE.e(TAG, ErrorCode.ERR_00000162, "setCSDSMap: getAccount returns null for brand with id: " + brand);
			return false;
        }

		dataUpdatedWithNewValue = account.setCSDSMap(csdsUrls);

        initServices(brand);

		return dataUpdatedWithNewValue;
	}

	private void initServices(String brand) {
		// Set Event manager domain
		Infra.instance.getEventManagerService().setDomain(getServiceUrl(brand, ConnectionParamsCache.CSDS_EVENT_MANAGER_DOMAIN_KEY));
	}

	public boolean isCsdsDataMissing(String brand) {
        AmsAccount account = getAccount(brand);
		if (account == null) {
			LPLog.INSTANCE.e(TAG, ErrorCode.ERR_00000162, "isCsdsDataMissing: getAccount returns null for brand with id: " + brand);
			return false;
		}

		return account.isCsdsDataMissing();
    }

    public String getServiceUrl(String brand, String serviceName) {
        AmsAccount account = getAccount(brand);
        if (account == null) {
			LPLog.INSTANCE.e(TAG, ErrorCode.ERR_00000162, "getServiceUrl: getAccount returns null for brand with id: " + brand);
			return null;
        }
        return account.getServiceUrl(serviceName);
    }

	/**
	 * Get the {@link com.liveperson.messaging.controller.connection.ConnectionParamsCache} for the given brandId
	 * @param brandId
	 * @return
	 */
	public IConnectionParamsCache getConnectionParamsCache(String brandId) {
		AmsAccount account = getAccount(brandId);
		if (account == null) {
			LPLog.INSTANCE.e(TAG, ErrorCode.ERR_00000162, "getConnectionParamsCache: getAccount returns null for brand with id: " + brandId);
			return null;
		}
		return account.getConnectionParamsCache();

	}

	/**
	 * Check whether AutoMessages is enabled on the given account (according to LPTag)
	 * @param brandId
	 * @return true - enabled, false - disabled
	 */
	public boolean isAutoMessagesEnabled(String brandId){
		AmsAccount account = getAccount(brandId);
		if (account == null) {
			LPLog.INSTANCE.e(TAG, ErrorCode.ERR_00000162, "isAutoMessagesEnabled: getAccount returns null for brand with id: " + brandId);
			return false;
		}
		return account.isAutoMessagesEnabled();
	}
	/**
     * Return connection url for socket based on brand id
     *
     * @param brandId
     * @return
     */
    public String getConnectionUrl(String brandId) {

        AmsAccount account = getAccount(brandId);
        if (account == null) {
			LPLog.INSTANCE.e(TAG, ErrorCode.ERR_00000162, "getConnectionUrl: getAccount returns null for brand with id: " + brandId);
			return null;
        }
		String connectionUrl = account.getServiceUrl(ConnectionParamsCache.CSDS_UMS_DOMAIN_KEY);
		String token = account.getToken();
		if (TextUtils.isEmpty(connectionUrl) || TextUtils.isEmpty(token)) {
			return null;
		}

		String path = String.format(SOCKET_URL, connectionUrl, brandId);
        Uri fullUri = Uri.parse(path).buildUpon()
                .appendQueryParameter(DOMAIN_VERSION_QUERY, String.valueOf(DOMAIN_VERSION_VALUE)).build();

        //Header: authorization : jwt token
        //Header: client properties : json
        return fullUri.toString();//path + mQueriesParams;
	}

	public String getTokenizerUrl(String brandId) {
		AmsAccount account = getAccount(brandId);
		if (account == null) {
			LPLog.INSTANCE.e(TAG, ErrorCode.ERR_00000162, "getTokenizerUrl: getAccount returns null for brand with id: " + brandId);
			return null;
		}
		return account.getServiceUrl(ConnectionParamsCache.CSDS_AMS_TOKENIZER_DOMAIN_KEY);
	}

    public String getConnectionUrlForLogs(String brandId) {
        AmsAccount account = getAccount(brandId);
        if (account == null) {
			LPLog.INSTANCE.e(TAG, ErrorCode.ERR_00000162, "getConnectionUrlForLogs: getAccount returns null for brand with id: " + brandId);
			return null;
        }
        String connectionUrl = account.getServiceUrl(ConnectionParamsCache.CSDS_UMS_DOMAIN_KEY);
        String token = account.getToken();
        if (TextUtils.isEmpty(connectionUrl) || TextUtils.isEmpty(token)) {
            return null;
        }
        return String.format(SOCKET_URL, connectionUrl, brandId);
    }

    /**
     * Return the {@link AmsAccount} of specific brand
     *
     * @param brand
     * @return
     */
    public AmsAccount getAccount(String brand) {
        return mAccounts.get(brand);
    }

    @Override
    public void clear() {
        mAccounts.clear();
    }

    public ClientProperties getClientProperties() {
        return mClientProperties;
    }

    public boolean isInUnAuthMode(String brandId){
        AmsAccount account = getAccount(brandId);
        if (account == null) {
			LPLog.INSTANCE.e(TAG, ErrorCode.ERR_00000162, "isInUnAuthMode: getAccount returns null for brand with id: " + brandId);
			return false;
        }
        return account.isInUnAuthMode();
    }

    /**
     * Method used to reset auth state for account
     * when authentication params were changed in background.
     * Once authentication params were changed - reconnecting flow started
     * and IDP call would be executed.
     *
     * @param account associated account for which authentication params were changed
     * @param newAuthParams new authentication params for requested account.
     */
    private void resetAuthStateIfNeeded(@NonNull AmsAccount account, @Nullable LPAuthenticationParams newAuthParams) {
        LPAuthenticationParams oldAuthParams = account.getLPAuthenticationParams();

        if (newAuthParams == null || oldAuthParams == null) {
            return;
        }

        if (!TextUtils.isEmpty(newAuthParams.getAuthKey()) && !StringsKt.equals(newAuthParams.getAuthKey(), oldAuthParams.getAuthKey(), true)) {
            LPLog.INSTANCE.d(TAG, "Resetting auth state for code flow.");
            account.resetConsumerAuthState();
        } else if (!TextUtils.isEmpty(newAuthParams.getHostAppJWT()) && !StringsKt.equals(newAuthParams.getHostAppJWT(), oldAuthParams.getHostAppJWT(), true)) {
            LPLog.INSTANCE.d(TAG, "Resetting auth state for implicit flow.");
            account.resetConsumerAuthState();
        } else if (newAuthParams.getAuthType() != LPAuthenticationType.AUTH && !newAuthParams.equals(oldAuthParams)) {
            LPLog.INSTANCE.d(TAG, "Resetting auth state for " + newAuthParams.getAuthType().name() + " flow.");
            account.resetConsumerAuthState();
        } else if (account.isTokenExpired()
                && oldAuthParams.getAuthType() == LPAuthenticationType.UN_AUTH
                && newAuthParams.getAuthType() == LPAuthenticationType.UN_AUTH) {
            account.resetConsumerAuthState();
        }
    }
}
