package com.liveperson.messaging.model;

import static com.liveperson.infra.errors.ErrorCode.ERR_00000161;

import android.text.TextUtils;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.liveperson.api.response.model.UserProfile;
import com.liveperson.api.response.types.TTRType;
import com.liveperson.infra.ICallback;
import com.liveperson.infra.analytics.LPAnalytics;
import com.liveperson.infra.auth.LPAuthenticationParams;
import com.liveperson.infra.configuration.Configuration;
import com.liveperson.infra.controller.DBEncryptionHelper;
import com.liveperson.infra.database.DataBaseCommand;
import com.liveperson.infra.eventmanager.Event;
import com.liveperson.infra.log.LPLog;
import com.liveperson.infra.messaging.R;
import com.liveperson.infra.model.LPWelcomeMessage;
import com.liveperson.infra.model.MessageOption;
import com.liveperson.infra.model.PushMessage;
import com.liveperson.infra.preferences.PushMessagePreferences;
import com.liveperson.infra.utils.EncryptionVersion;
import com.liveperson.infra.utils.LocaleUtils;
import com.liveperson.messaging.Messaging;
import com.liveperson.messaging.network.http.AgentProfileRequest;
import com.liveperson.api.response.model.Event.Types;
import com.liveperson.messaging.wm.WelcomeMessageManager;

import org.json.JSONException;
import org.json.JSONObject;

import java.util.HashSet;
import java.util.List;
import java.util.Set;

/**
 * Created by shiranr on 28/01/2016.
 */
public class ConversationUtils {

    private static final String TAG = "ConversationUtils";
    protected final Messaging mController;

    public ConversationUtils(Messaging controller) {
        mController = controller;
    }

    /**
     * Method used to run
     * {@link AmsMessages#showWelcomeMessage(MessagingChatMessage, LPWelcomeMessage, AmsMessages.MessagesListener.OnWelcomeMessageShownCallback)}
     * which represents welcome message immediately.
     *
     * If provided welcome message frequency is set to first time conversation
     * and first message is not equals to true provided welcome message would
     * be ignored.
     *
     * @param brandId identifier of brand required to request welcome message.
     * @param message possible welcome message that would be represented
     *                in conversation screen.
     * @see ConversationUtils#showInitialWelcomeMessage(String, LPWelcomeMessage) 
     */
    public void showWelcomeMessage(String brandId, LPWelcomeMessage message) {
        showWelcomeMessage(brandId, message, AmsDialogs.KEY_WELCOME_DIALOG_ID, MessagingChatMessage.MessageState.READ);
    }


    /**
     * Method used to run
     * {@link AmsMessages#showWelcomeMessage(MessagingChatMessage, LPWelcomeMessage, AmsMessages.MessagesListener.OnWelcomeMessageShownCallback)}
     * which represents welcome message immediately.
     *
     * If provided welcome message frequency is set to first time conversation
     * and first message is not equals to true provided welcome message would
     * be ignored.
     *
     * @param brandId      identifier of brand required to request welcome message.
     * @param message      possible welcome message that would be represented
     *                     in conversation screen.
     * @param dialogId     related dialog id
     * @param messageState state for welcome message
     * @see ConversationUtils#showInitialWelcomeMessage(String, LPWelcomeMessage)
     */
    public void showWelcomeMessage(
            String brandId,
            LPWelcomeMessage message,
            String dialogId,
            MessagingChatMessage.MessageState messageState
    ) {
        long messageTimestamp;
        if (messageState.equals(MessagingChatMessage.MessageState.OFFLINE)) {
            messageTimestamp = AmsMessages.OFFLINE_WELCOME_MESSAGE_TIMESTAMP;
        } else {
            messageTimestamp = System.currentTimeMillis();
        }
        final MessagingChatMessage contentMessage = buildWelcomeMessagingChatMessage(
                brandId,
                messageTimestamp,
                message.getWelcomeMessage(),
                AmsMessages.WELCOME_MSG_SEQUENCE_NUMBER,
                MessagingChatMessage.MessageType.BRAND,
                dialogId,
                messageState
        );
        List<MessageOption> options = message.getMessageOptions();

        final String quickRepliesContent;
        if (options == null || options.isEmpty()) {
            quickRepliesContent = null;
        } else {
            String quickRepliesJson = DBEncryptionHelper.decrypt(
                    EncryptionVersion.VERSION_1,
                    message.getQuickReplies(true)
            );
            QuickRepliesMessageHolder holder = QuickRepliesMessageHolder.fromJsonString(brandId, quickRepliesJson);
            if (holder == null) {
                quickRepliesContent = null;
            } else {
                quickRepliesContent = holder.getQuickRepliesString();
            }
        }

        if (contentMessage != null) {
            OnWelcomeMessageShownCallbackImpl callback = new OnWelcomeMessageShownCallbackImpl(contentMessage, quickRepliesContent);
            mController.amsMessages.showWelcomeMessage(contentMessage, message, callback);
        }
    }

    /**
     * Method used to request welcome message to conversation screen
     * if outbound welcome message is not presented in database.
     * @param brandId identifier of brand required to request welcome message.
     */
    public void addWelcomeMessageRequest(final String brandId) {
        mController.amsMessages.getLatestOutboundMessage().setPostQueryOnBackground(outboundMessage -> {
            // Add welcome message only If last message is not outbound campaign message
            boolean isOfflineMessagesExists = mController.amsMessages
                    .getOfflineMessagesRepository()
                    .areOfflineMessagesExists(brandId);
            if (TextUtils.isEmpty(outboundMessage) && !isOfflineMessagesExists) {
                LPLog.INSTANCE.d(TAG, "addWelcomeMessage: Requesting welcome message");
                requestWelcomeMessage(brandId);
            }
        }).execute();
    }

    /**
     * Method used to show initial welcome message from conversation params
     * for brand-new consumers.
     * @param brandId identifier of brand required to request welcome message.
     * @param welcomeMessage initial welcome message for brand new consumer.
     */
    public void showInitialWelcomeMessage(
            final String brandId,
            LPWelcomeMessage welcomeMessage
    ) {
        mController.amsMessages.getLatestOutboundMessage().setPostQueryOnBackground(outboundMessage -> {
            // Add welcome message only If last message is not outbound campaign message
            boolean isOfflineMessagesExists = mController.amsMessages
                    .getOfflineMessagesRepository()
                    .areOfflineMessagesExists(brandId);
            if (TextUtils.isEmpty(outboundMessage) && !isOfflineMessagesExists) {
                LPLog.INSTANCE.d(TAG, "addWelcomeMessage: Adding welcome message from view params");
                showWelcomeMessage(brandId, welcomeMessage);
            }
        }).execute();
    }



    /**
     * Add or update outbound welcome message into the database If necessary.
     *
     * @param brandId respective account id
     */
    public void updateOutboundCampaignMessage(String brandId) {
        if (PushMessagePreferences.INSTANCE.isPushNotificationClicked()) {
            String id = PushMessagePreferences.INSTANCE.getClickedNotificationId();
            displayOutboundCampaignMessage(brandId, id);
        } else if (Configuration.getBoolean(R.bool.show_outbound_in_app_message)) {
            String id = PushMessagePreferences.INSTANCE.getLatestNotificationIdForBrand(brandId);
            displayOutboundCampaignMessage(brandId, id);
        } else {
            // Clean up existing outbound campaign message if required
            LPLog.INSTANCE.d(TAG, "updateOutboundCampaignMessage: Removing existing outbound welcome message");
            mController.amsMessages.removeLastOutboundMessage().execute();
        }
    }

    /**
     * Add outbound welcome message into database and notify UI
     *
     * @param brandId   respective account id
     * @param messageId id of message to be fetched from cache
     */
    private void displayOutboundCampaignMessage(String brandId, String messageId) {
        PushMessage pushMessage = null;
        if (!TextUtils.isEmpty(messageId)) {
            pushMessage = PushMessagePreferences.INSTANCE.getCachedPushMessage(messageId, brandId);
        }

        // check  if messageEvent is null (payload does not have "event" json) then need to get cached welcome msg
        final String newOutboundMessage = (pushMessage != null && !TextUtils.isEmpty(pushMessage.getMessageEvent())) ?
                getContentFromMessageEvent(pushMessage.getMessageEvent(), pushMessage) : PushMessagePreferences.INSTANCE.getCachedPushWelcomeMessage(messageId, brandId);
        // payload has "event" (rich content) && has type = RichContentEvent -> MessageType = AGENT_STRUCTURED_CONTENT
        // else -> MessageType = BRAND
        final MessagingChatMessage.MessageType msgType = (pushMessage != null && isRichContentMessage(pushMessage.getMessageEvent())) ?
                MessagingChatMessage.MessageType.AGENT_STRUCTURED_CONTENT : MessagingChatMessage.MessageType.BRAND;
        String transactionId = PushMessagePreferences.INSTANCE.getTransactionId(messageId, brandId);

        LPLog.INSTANCE.d(TAG, "newOutboundMessage: " + newOutboundMessage + "\nmsgType: " + msgType.name());

        if (!TextUtils.isEmpty(newOutboundMessage)) {
            final PushMessage currentPushMessage = pushMessage;
            mController.amsMessages.getLatestOutboundMessage().setPostQueryOnBackground(storedOutboundMessage -> {
                removeTemporalWelcomeMessage().executeSynchronously();
                if (TextUtils.isEmpty(storedOutboundMessage)) {
                    long messageTimestamp = System.currentTimeMillis();
                    MessagingChatMessage messagingChatMessage = buildWelcomeMessagingChatMessage(
                            brandId,
                            messageTimestamp,
                            newOutboundMessage,
                            AmsMessages.OUTBOUND_CAMPAIGN_MSG_SEQUENCE_NUMBER,
                            msgType,
                            AmsDialogs.KEY_WELCOME_DIALOG_ID,
                            MessagingChatMessage.MessageState.READ
                    );

                    // Remove any existing welcome message or an already added/old outbound campaign message
                    mController.amsMessages.removeLastOutboundMessage().execute();

                    if (currentPushMessage != null && !currentPushMessage.isExpired()) {
                        LPLog.INSTANCE.d(TAG, "displayOutboundCampaignMessage: Adding new outbound campaign welcome message.");
                        mController.amsMessages.addMessage(messagingChatMessage, true).execute();
                        mController.getEventManagerService().logEvent(brandId, transactionId, true, Event.READ, null, null, mController.getApplicationContext());

                        Set<String> keeperIds = new HashSet<>();
                        keeperIds.add(messageId);
                        mController.deleteOtherPendingProactiveMessages(brandId, keeperIds);
                    } else {
                        LPLog.INSTANCE.d(TAG, "displayOutboundCampaignMessage: message expired, skipped");
                    }
                } else if (!newOutboundMessage.equals(storedOutboundMessage)) {
                    // Update outbound message if we got new one
                    long messageTimestamp = System.currentTimeMillis();
                    MessagingChatMessage messagingChatMessage = buildWelcomeMessagingChatMessage(
                            brandId,
                            messageTimestamp,
                            newOutboundMessage,
                            AmsMessages.OUTBOUND_CAMPAIGN_MSG_SEQUENCE_NUMBER,
                            msgType,
                            AmsDialogs.KEY_WELCOME_DIALOG_ID,
                            MessagingChatMessage.MessageState.READ
                    );

                    mController.getEventManagerService().logEvent(brandId, transactionId, true, Event.READ, null, null, mController.getApplicationContext());

                    Set<String> keeperIds = new HashSet<>();
                    keeperIds.add(messageId);
                    mController.deleteOtherPendingProactiveMessages(brandId, keeperIds);

                    if (currentPushMessage == null || currentPushMessage.isExpired()) {
                        LPLog.INSTANCE.d(TAG, "displayOutboundCampaignMessage: Removing existing outbound welcome message");
                        mController.amsMessages.removeLastOutboundMessage().execute();
                    } else {
                        LPLog.INSTANCE.d(TAG, "displayOutboundCampaignMessage: Updating existing outbound campaign welcome message with new one.");
                        mController.amsMessages.updateLastOutboundMessage(messagingChatMessage).execute();
                        trackProactiveAsWelcomeMessage(brandId, transactionId, msgType);
                    }
                }
            }).execute();
        } else {
            // Clear existing outbound campaign message if we have one stored in database but nothing
            // in shared preferences (may be expired one)
            LPLog.INSTANCE.d(TAG, "displayOutboundCampaignMessage: Removing existing outbound welcome message");
            mController.amsMessages.removeLastOutboundMessage().execute();
        }
    }

    /**
     * Event to track when proactive message is processed and displayed on conversation window.
     * It can be pure message or rich content with StructuredContent in the payload.
     *
     * @param brandId       - brand required to be tracked;
     * @param transactionId - id of transaction;
     * @param type          - type of message (could be plain message or SC message);
     */
    private void trackProactiveAsWelcomeMessage(
            @NonNull String brandId,
            @NonNull String transactionId,
            @NonNull MessagingChatMessage.MessageType type) {
        LPAuthenticationParams params = mController.mAccountsController.getLPAuthenticationParams(brandId);
        if (params != null) {
            LPAnalytics.ConsumerAction.INSTANCE.trackShowProactiveAsWelcomeMessage(
                    params.getAuthType(),
                    LocaleUtils.getInstance().getLocaleCode(),
                    type == MessagingChatMessage.MessageType.AGENT_STRUCTURED_CONTENT,
                    transactionId
            );
        } else {
            LPLog.INSTANCE.d(TAG, "getLPAuthenticationParams: getLPAuthenticationParams returns null for brand with id: " + brandId);
        }
    }

    /**
     * Build a new temporary welcome message/Campaign for consumer. This could be Brand set welcome message or
     * outbound campaign message with or without rich content (StructuredContent)
     *
     * @param brandId          respective brand's account id
     * @param messageTimestamp current time
     * @param message          message text
     * @param messageSequence  message sequence number (Currently -4 or -5)
     * @param msgType          it can be:
     *                         BRAND: old content
     *                         AGENT_STRUCTURED_CONTENT: structured content
     * @return chat message objects
     */
    private MessagingChatMessage buildWelcomeMessagingChatMessage(
            String brandId,
            long messageTimestamp,
            String message,
            int messageSequence,
            MessagingChatMessage.MessageType msgType,
            String dialogId,
            MessagingChatMessage.MessageState messageState
    ) {
        String conversationId = AmsConversations.KEY_WELCOME_CONVERSATION_ID;
        String originatorId = "";
        String eventId = dialogId + AmsMessages.WELCOME_MESSAGE_EVENT_ID_POSTFIX;


        mController.amsConversations.createDummyConversationForWelcomeMessage(brandId, conversationId, messageTimestamp);
        mController.amsDialogs.createDummyDialogForWelcomeMessage(brandId, conversationId, dialogId, messageSequence, messageTimestamp);

        MessagingChatMessage messagingChatMessage = new MessagingChatMessage(
                originatorId,
                message,
                messageTimestamp,
                dialogId,
                eventId,
                msgType,
                messageState,
                EncryptionVersion.NONE
        );
        messagingChatMessage.setServerSequence(messageSequence);
        return messagingChatMessage;
    }

    /**
     * Get content string if it's available
     *
     * @param msgEvent    event json string in payload
     * @param pushMessage push notification message
     * @return - content payload if it's available <br>
     * - message payload || notification message (if message in payload is empty) when event type = ContentEvent
     */
    private String getContentFromMessageEvent(String msgEvent, PushMessage pushMessage) {
        if (TextUtils.isEmpty(msgEvent)) {
            return "";
        }

        try {
            JSONObject jsonObject = new JSONObject(msgEvent);
            if (jsonObject.has("content")) {
                return jsonObject.optString("content");
            } else if (jsonObject.has("type") && jsonObject.optString("type").equalsIgnoreCase(Types.ContentEvent.name())) {
                // if there is no message in payload -> use notification message
                String message = jsonObject.optString("message");
                if (TextUtils.isEmpty(message)) {
                    return pushMessage.getMessage();
                } else {
                    return message;
                }
            }
            return "";
        } catch (JSONException e) {
            LPLog.INSTANCE.e(TAG, ERR_00000161, "Exception while parsing json.", e);
            return "";
        }
    }

    /**
     * Check message is Rich content or not
     *
     * @param msgEvent event json string in payload
     * @return true if it has event type = RichContentEvent
     */
    private boolean isRichContentMessage(String msgEvent) {
        if (TextUtils.isEmpty(msgEvent)) {
            return false;
        }

        try {
            JSONObject jsonObject = new JSONObject(msgEvent);
            if (jsonObject.has("type") && jsonObject.optString("type").equalsIgnoreCase(Types.RichContentEvent.name())) {
                return true;
            }
            return false;
        } catch (JSONException e) {
            LPLog.INSTANCE.e(TAG, ERR_00000161, "Exception while parsing json.", e);
            return false;
        }
    }

    public void updateParticipants(final String targetId, String[] participants, final UserProfile.UserType userType,
                                   final String conversationId, final boolean updateUI, final boolean forceUpdate) {
        updateParticipants(targetId, participants, userType, conversationId, updateUI, forceUpdate, null);
    }

    public void updateParticipants(final String targetId, String[] participants, final UserProfile.UserType userType,
                                   final String conversationId, final boolean updateUI, final boolean forceUpdate, final ICallback<MessagingUserProfile, Exception> callback) {
        for (final String userId : participants) {
            if (!TextUtils.isEmpty(userId)) {
                //foreach participant - if does not exist in db - send getUserProfile request
                mController.amsUsers.getUserById(userId).setPostQueryOnBackground(userProfile -> {
                    if (userProfile == null) {
                        userProfile = new MessagingUserProfile("", "", userType);
                        userProfile.setOriginatorID(userId);
                        mController.amsUsers.updateUserProfile(userProfile);
                        mController.amsMessages.onAgentReceived(userProfile);

                        LPLog.INSTANCE.i(TAG, "First time bringing information for another participant that joined conversation " + conversationId);
                        sendUpdateUserRequest(targetId, userId, conversationId, updateUI, callback);
                    } else if (userProfile.isEmptyUser() || forceUpdate) {
                        sendUpdateUserRequest(targetId, userId, conversationId, updateUI, callback);
                    } else if (callback != null) {
                        callback.onSuccess(userProfile);
                    }
                }).execute();
            }
        }
    }

    /**
     * @param targetId
     * @param userId
     * @param conversationId
     * @param shouldUpdateUi
     * @param callback
     */
    private void sendUpdateUserRequest(String targetId, String userId, String conversationId, boolean shouldUpdateUi, ICallback<MessagingUserProfile, Exception> callback) {
        if (!TextUtils.isEmpty(userId)) {
            // There is an assigned agent, get his details and update
            new AgentProfileRequest(mController, targetId, userId, conversationId, shouldUpdateUi).setCallback(callback).execute();
        } else {
            if (TextUtils.isEmpty(conversationId)) {
                LPLog.INSTANCE.d(TAG, "sendUpdateUserRequest: no dialog id");
                return;
            }

            // There is no assigned agent, get the conversation from the DB and update the callback
            mController.amsConversations.queryConversationById(conversationId)
                    .setPostQueryOnBackground(data -> {
                        LPLog.INSTANCE.d(TAG, "onResult: Calling agent details callback with null agent");
                        mController.onAgentDetailsChanged(null, data.isConversationOpen());
                    }).execute();
        }

    }

    public void updateTTR(TTRType type, long effectiveTTR, String targetId) {
        LPLog.INSTANCE.d(TAG, "update TTR type to - " + type + ". EffectiveTTR = " + effectiveTTR);
        mController.amsConversations.updateTTRType(targetId, type, effectiveTTR);

    }

    public long calculateEffectiveTTR(final String targetId, final long ttrValue, final long manualTTR, final long delayTTR) {
        long clockDiff = mController.mConnectionController.getClockDiff(targetId);

        return mController.amsConversations.calculateEffectiveTTR(targetId, ttrValue, manualTTR, delayTTR, clockDiff);
    }

    public static void showTTR(final Messaging controller, String targetId) {
        controller.amsConversations.showTTR(targetId);
    }

    // region welcome message manager

    /**
     * Method used to cancel all pending delayed tasks
     * if welcome message manager is initialized and active(not null)
     * @see WelcomeMessageManager#cancelTimeoutTasks()
     */
    public void cancelWelcomeMessageTimeoutTasks() {
        WelcomeMessageManager manager = mController.getWelcomeMessageManager();
        if (manager != null) {
            manager.cancelTimeoutTasks();
        }
    }

    /**
     * Method used to request a welcome message
     * for particular brand
     * if welcome message manager is initialized and active(not null)
     *
     * @param brandId identifier of particular brand
     *
     * @see WelcomeMessageManager#requestNewWelcomeMessage(String)
     */
    public void requestWelcomeMessage(String brandId) {
        WelcomeMessageManager manager = mController.getWelcomeMessageManager();
        if (manager != null) {
            manager.requestNewWelcomeMessage(brandId);
        }
    }

    /**
     * Method used to get an actual welcome message
     * for particular brand.
     * if welcome message manager is initialized and active(not null)
     *
     * @param brandId identifier of particular brand
     *
     * @see WelcomeMessageManager#getWelcomeMessage(String)
     *
     * @return actual welcome message for particular brand or null
     * if welcome message is not active(equals null).
     */
    public LPWelcomeMessage getWelcomeMessage(String brandId) {
        WelcomeMessageManager manager = mController.getWelcomeMessageManager();
        if (manager == null) {
            return null;
        } else {
            return manager.getWelcomeMessage(brandId);
        }
    }

    /**
     * Method used to clear metadata of welcome message for active brand.
     */
    private void clearWelcomeMessageMetadata() {
        mController.amsMessages.clearWelcomeMessageMetadata();
    }

    private DataBaseCommand<Boolean> removeTemporalWelcomeMessage() {
        return mController.amsMessages.removeTemporalWelcomeMessage();
    }

    public void clearLatestWelcomeMessage() {
        removeTemporalWelcomeMessage()
                .setPostQueryOnBackground(data -> { if (data) clearWelcomeMessageMetadata();})
                .execute();
    }
    // endregion welcome message manager

    private class OnWelcomeMessageShownCallbackImpl implements AmsMessages.MessagesListener.OnWelcomeMessageShownCallback {

        @NonNull
        private final MessagingChatMessage contentMessage;
        @Nullable
        private final String quickRepliesContent;

        private OnWelcomeMessageShownCallbackImpl(@NonNull MessagingChatMessage contentMessage, @Nullable String quickRepliesContent) {
            this.contentMessage = contentMessage;
            this.quickRepliesContent = quickRepliesContent;
        }

        @Override
        public void onWelcomeMessageShown(boolean isAdded) {
            if (isAdded) {
                mController.amsMessages.addMessage(contentMessage, false).execute();
                if (contentMessage.getMessageState() != MessagingChatMessage.MessageState.OFFLINE) {
                    mController.amsMessages.prepareWelcomeMessageMetadata(contentMessage, quickRepliesContent);
                } else {
                    mController.amsMessages.clearWelcomeMessageMetadata();
                }
            } else {
                clearLatestWelcomeMessage();
            }
        }

        @Override
        public void onOutboundMessageShown() {
            removeTemporalWelcomeMessage().execute();
        }
    }
}
