package com.liveperson.messaging.network.http;

import android.text.TextUtils;

import com.liveperson.api.response.model.UserProfile;
import com.liveperson.infra.Command;
import com.liveperson.infra.ICallback;
import com.liveperson.infra.database.DataBaseExecutor;
import com.liveperson.infra.log.LPLog;
import com.liveperson.infra.network.http.HttpHandler;
import com.liveperson.infra.network.http.request.HttpGetRequest;
import com.liveperson.infra.otel.LPTraceType;
import com.liveperson.messaging.Messaging;
import com.liveperson.messaging.controller.connection.ConnectionParamsCache;
import com.liveperson.messaging.model.AcUserData;
import com.liveperson.messaging.model.AcUserType;
import com.liveperson.messaging.model.MessagingUserProfile;

import static com.liveperson.infra.errors.ErrorCode.ERR_000000C6;

import java.util.Set;

/**
 * Created by ofira on 3/17/16.
 */
public class AgentProfileRequest implements Command {

    private static final String TAG = "AgentProfileRequest";

    private static final String AC_USERS_URL = "https://%s/api/account/%s/configuration/le-users/users/%s";
    private static final int AGENT_REQUEST_TIMEOUT = 30000;

    private final Messaging mController;

    private final String mBrand;
    private final String mUserId;
    private final String mDialogId;
    private final boolean mShouldUpdateUI;
    private ICallback<MessagingUserProfile, Exception> callback;

    public AgentProfileRequest(Messaging controller, String brand, String userId, String dialogId, boolean shouldUpdateUi) {
        mBrand = brand;
        mUserId = userId;
        mDialogId = dialogId;
        mShouldUpdateUI = shouldUpdateUi;
        mController = controller;
    }

    @Override
    public void execute() {
        String acCdnDomain = mController.mAccountsController.getServiceUrl(mBrand, ConnectionParamsCache.CSDS_AC_CDN_DOMAIN_KEY);

        if (!TextUtils.isEmpty(acCdnDomain)) {
            String cSDSRequestUrl = String.format(AC_USERS_URL, acCdnDomain, mBrand, mUserId);

            LPLog.INSTANCE.d(TAG, "Getting agent details url " + cSDSRequestUrl);
            HttpGetRequest httpGetRequest = new HttpGetRequest(cSDSRequestUrl,
                    LPTraceType.ACCDN_USER_DETAILS_REQ);
            httpGetRequest.setTimeout(AGENT_REQUEST_TIMEOUT);
            httpGetRequest.setCallback( new ICallback<String, Exception>() {
                @Override
                public void onSuccess(String profile) {
                    if (!TextUtils.isEmpty(profile)) {
                        LPLog.INSTANCE.d(TAG, "onSuccess with agent details " + profile);
                        onRequestCompleted();

                        AcUserData userData = new AcUserData(profile);
                        final MessagingUserProfile userProfile = new MessagingUserProfile(userData.getFirstName(), userData.getLastName(), UserProfile.UserType.AGENT);
                        userProfile.setNickname(userData.getNickName());
                        userProfile.setBrandID(mBrand);
                        userProfile.setDescription(userData.getEmployeeId());
                        userProfile.setOriginatorID(mUserId);
                        userProfile.setAvatarUrl(userData.getPictureUrl());
                        userProfile.setEmail(userData.getEmail());
                        // if user data has type of bot, then
                        // SDK should save its originator id
                        // and update messages sent by particular user
                        // within dialog.
                        if (userData.getUserType().equals(AcUserType.BOT)
                            || userData.getUserType().equals(AcUserType.CONTROLLER)) {
                            DataBaseExecutor.execute(() -> {
                                Set<String> botUsers = mController.amsUsers.getBotUserIds(mBrand);
                                if (botUsers.add(mUserId)) {
                                    mController.amsUsers.setBotUserIds(mBrand, botUsers);
                                    if (mDialogId != null && mUserId != null) {
                                        mController.amsMessages.updateBotMessagesForDialog(mDialogId, mUserId).execute();
                                    } else {
                                        // this code will handle a situation when
                                        // mDialogId is null but sdk has just loaded messages
                                        // for particular dialog.
                                        Set<String> syncedDialogs = mController.amsDialogs.getSyncedDialogIds();
                                        for (String dialogId : syncedDialogs) {
                                            mController.amsMessages.updateBotMessagesForDialog(dialogId, mUserId).execute();
                                        }
                                    }
                                }
                            });
                        }

                        mController.amsMessages.onAgentReceived(userProfile);
                        mController.amsUsers.updateUserProfile(userProfile);

                        if(callback != null){
                            callback.onSuccess(userProfile);
                        }

                        if (TextUtils.isEmpty(mDialogId)){
                            LPLog.INSTANCE.d(TAG, "onResult: updating agent details. without conversation id.");
                            return;
                        }
                        mController.amsDialogs.queryDialogById(mDialogId)
                                .setPostQueryOnBackground(data -> {
                                    if (data != null && TextUtils.equals(data.getAssignedAgentId(), mUserId)) {
	                                    LPLog.INSTANCE.d(TAG, "onResult: Calling agent details callback");
                                        mController.onAgentDetailsChanged(userProfile, data.isOpen());
                                    }
                                    //update all massages that filtered by brand id.
                                    //if we have open conversation with the assigned agent that we just got his profile -
                                    //update also messages that filtered by conversationID.
	                                LPLog.INSTANCE.d(TAG, "got agent details (" + mUserId + ") related to dialog ID: " + mDialogId);
                                    //if we finished fetchHistory we are updated. any agent details after that moment should update ui.
                                    if (mShouldUpdateUI /* || mController.mConnectionController.getConnection(mBrand).isUpdated()*/) {
	                                    LPLog.INSTANCE.d(TAG, "Updating ui with agent details! ");
                                        mController.amsMessages.updateAgentDetailsUpdated(mDialogId);
                                    }
                                }).execute();
                    }else{
                        if(callback != null){
                            callback.onError(new Exception("Empty response"));
                        }
                    }
                }

                @Override
                public void onError(Exception exception) {
                    LPLog.INSTANCE.e(TAG, ERR_000000C6, "JSONException", exception);
                    if(callback != null){
                        callback.onError(exception);
                    }
                }
            });
            HttpHandler.execute(httpGetRequest);

            onRequestSent();
        }
    }


    private void onRequestSent() {
        if (!TextUtils.isEmpty(mDialogId)){
            mController.amsDialogs.addUpdateRequestInProgress(mDialogId);
        }

    }

    private void onRequestCompleted() {
        if (!TextUtils.isEmpty(mDialogId)){
            mController.amsDialogs.removeUpdateRequestInProgress(mDialogId);
        }
    }

    public ICallback<MessagingUserProfile, Exception> getCallback() {
        return callback;
    }

    public AgentProfileRequest setCallback(ICallback<MessagingUserProfile, Exception> callback) {
        this.callback = callback;
        return this;
    }
}
