package com.liveperson.messaging.offline

import android.content.ContentValues
import androidx.annotation.WorkerThread
import com.liveperson.api.response.types.ConversationState
import com.liveperson.api.response.types.TTRType
import com.liveperson.infra.database.BaseDBRepository
import com.liveperson.infra.database.tables.ConversationsTable
import com.liveperson.infra.log.LPLog.d
import com.liveperson.messaging.cache.ConversationsCacheProvider
import com.liveperson.messaging.model.Conversation
import com.liveperson.messaging.offline.api.OfflineConversationRepository

/**
 * Repository used for storing and retrieving a conversation in offline mode.
 * The conversation will be stored in the in-memory cache and in the database.
 */
internal class OfflineConversationRepositoryImpl(
    private val conversationsCacheProvider: ConversationsCacheProvider
) : BaseDBRepository(ConversationsTable.TABLE_NAME), OfflineConversationRepository {

    companion object {
        private const val TAG = "OfflineConversationRepositoryImpl"
    }

    @WorkerThread
    override fun getOrCreateOfflineConversation(
        targetId: String,
        brandId: String,
        requestId: Long
    ): Conversation {
        val currentOfflineConversation =
            conversationsCacheProvider.getConversationFromCache(Conversation.OFFLINE_CONVERSATION_ID)
            ?: queryOfflineConversation(targetId, brandId)
            ?: createAndSaveOfflineConversation(targetId, brandId, requestId)
        conversationsCacheProvider.cacheConversation(currentOfflineConversation)
        return currentOfflineConversation
    }

    override fun queryRealConversationsCount(): Int {
        val whereClause = (ConversationsTable.Key.CONVERSATION_ID + " != ?"
                + " AND " + ConversationsTable.Key.CONVERSATION_ID + " != ?")
        val whereArgs = arrayOf(
            Conversation.TEMP_CONVERSATION_ID,
            Conversation.OFFLINE_CONVERSATION_ID
        )
        try {
            db.query(null, whereClause, whereArgs, null, null, null)
                .use { cursor -> return cursor.count }
        } catch (exception: Exception) {
            d(TAG, "Error occurred while getting real conversation count", exception)
            return -1
        }
    }

    override fun clearOfflineConversation(targetId: String) {
        val whereClause = (ConversationsTable.Key.CONVERSATION_ID + " = ?"
                + " AND " + ConversationsTable.Key.TARGET_ID + " = ?")
        val whereArgs = arrayOf(Conversation.OFFLINE_CONVERSATION_ID, targetId)
        db.removeAll(whereClause, whereArgs)
        conversationsCacheProvider.removeConversationFromCache(
            targetId,
            Conversation.OFFLINE_CONVERSATION_ID
        )
        d(TAG, "Finished removing offline conversation")
    }

    /**
     * Retrieve the offline conversation from the database if it exists
     * @param targetId the target id
     * @param brandId the brand id
     * @return the stored offline conversation or null if not found
     */
    private fun queryOfflineConversation(targetId: String, brandId: String): Conversation? {
        val query = (ConversationsTable.Key.CONVERSATION_ID + " = ?"
                + " AND " + ConversationsTable.Key.TARGET_ID + " = ?"
                + " AND " + ConversationsTable.Key.BRAND_ID + " = ?"
                + " AND " + ConversationsTable.Key.STATE + " = ?")
        val arguments = arrayOf(
            Conversation.OFFLINE_CONVERSATION_ID,
            targetId,
            brandId,
            ConversationState.OFFLINE.ordinal.toString()
        )
        return runCatching {
            db.query(null, query, arguments, null, null, null, "1")
                ?.use { it.takeIf { it.moveToFirst() }?.let { Conversation(it) } }
        }.onFailure {
            d(TAG, "Failed to query offline conversation", it)
        }.getOrNull()
    }

    /**
     * Create an offline conversation and store it in the database
     * @param targetId the target id
     * @param brandId the brand id
     * @param requestId the request id
     * @return the newly created offline conversation
     */
    private fun createAndSaveOfflineConversation(
        targetId: String,
        brandId: String,
        requestId: Long
    ): Conversation {
        val conversation = Conversation(targetId, brandId)
        conversation.conversationId = Conversation.OFFLINE_CONVERSATION_ID
        conversation.state = ConversationState.OFFLINE
        conversation.conversationTTRType = TTRType.NORMAL
        conversation.requestId = requestId
        conversation.unreadMessages = -1
        conversation.startTimestamp = System.currentTimeMillis()
        val insertValues = ContentValues()
        insertValues.put(ConversationsTable.Key.BRAND_ID, conversation.brandId)
        insertValues.put(ConversationsTable.Key.TARGET_ID, conversation.targetId)
        insertValues.put(ConversationsTable.Key.CONVERSATION_ID, conversation.conversationId)
        insertValues.put(ConversationsTable.Key.STATE, conversation.state.ordinal)
        insertValues.put(ConversationsTable.Key.TTR_TYPE, conversation.conversationTTRType.ordinal)
        insertValues.put(ConversationsTable.Key.UNREAD_MESSAGES_COUNT, -1)
        insertValues.put(ConversationsTable.Key.START_TIMESTAMP, conversation.startTimestamp)
        insertValues.put(ConversationsTable.Key.REQUEST_ID, conversation.requestId)
        db.insert(insertValues)
        return conversation
    }
}
