package com.liveperson.messaging.offline

import android.content.ContentValues
import android.text.TextUtils
import com.liveperson.api.response.types.DialogState
import com.liveperson.infra.database.BaseDBRepository
import com.liveperson.infra.database.tables.DialogsTable
import com.liveperson.infra.database.tables.MessagesTable
import com.liveperson.infra.log.LPLog.d
import com.liveperson.infra.managers.PreferenceManager
import com.liveperson.messaging.model.AmsDialogs
import com.liveperson.messaging.model.AmsMessages
import com.liveperson.messaging.model.Dialog
import com.liveperson.messaging.model.FileMessage
import com.liveperson.messaging.model.FullMessageRow
import com.liveperson.messaging.model.MessagingChatMessage
import com.liveperson.messaging.model.MessagingChatMessage.MessageState
import com.liveperson.messaging.offline.api.OfflineMessagesRepository
import com.liveperson.messaging.provider.FileMessageProvider
import com.liveperson.messaging.provider.FileUploadProgressProvider
import com.liveperson.messaging.utils.joinToSQLString

/**
 * Repository used for storing and retrieving messages while in offline mode.
 * The messages will be stored to and retrieved from the database.
 */
internal class OfflineMessagesRepositoryImpl(
    private val preferenceManager: PreferenceManager,
    private val fileMessageProvider: FileMessageProvider,
    private val fileUploadProgressProvider: FileUploadProgressProvider,
    private val offlineMessagesListener: OfflineMessagesListener
) : BaseDBRepository(MessagesTable.MESSAGES_TABLE),
    OfflineMessagesRepository {

    companion object {
        private const val TAG = "OfflineMessagesRepositoryImpl"

        private const val KEY_OFFLINE_MESSAGES_EVENTS = "OFFLINE_MESSAGES_EVENTS"
    }

    override fun loadOfflineMessagesExcept(
        brandId: String,
        eventIds: Set<String>
    ): List<FullMessageRow> {
        val rowIds = fileUploadProgressProvider.getInProgressUploadMessageRowIdsString()
        val whereArgs: Array<String> =
            getPendingMessagesQueryParams(brandId, rowIds, DialogState.OPEN.ordinal.toString())
        val where: String = getPendingMessagesQuery(rowIds)
        val offlineMessages: MutableList<FullMessageRow> = ArrayList()
        try {
            db.query(null, where, whereArgs, null, null, null).use { cursor ->
                if (cursor != null && cursor.moveToFirst()) {
                    do {
                        val message = AmsMessages.getSingleMessageFromCursor(cursor)
                        if (!MessagingChatMessage.MessageType.isConsumer(message.messageType)) {
                            continue
                        }
                        if (eventIds.contains(message.eventId)) {
                            continue
                        }
                        var fileMessage: FileMessage? = null
                        if (MessagingChatMessage.MessageType.isConsumerFile(message.messageType)) {
                            fileMessage = fileMessageProvider.getFileByMessageRowId(message.localId)
                        }
                        offlineMessages.add(FullMessageRow(message, null, fileMessage))
                    } while (cursor.moveToNext())
                }
            }
        } catch (exception: Exception) {
            d(TAG, "Failed to query offline messages", exception)
        }
        return offlineMessages
    }

    override fun removeMessage(eventId: String) {
        val whereClause = MessagesTable.KEY_EVENT_ID + " = ?"
        val whereArgs = arrayOf(eventId)
        db.removeAll(whereClause, whereArgs)
        offlineMessagesListener.removeMessageByEventId(eventId)
    }

    override fun areOfflineMessagesExists(brandId: String): Boolean {
        val joinColumn = buildString {
            append("JOIN ${DialogsTable.TABLE_NAME} ON ")
            append("${DialogsTable.TABLE_NAME}.${DialogsTable.Key.DIALOG_ID}")
            append(" = ")
            append("${MessagesTable.MESSAGES_TABLE}.${MessagesTable.KEY_DIALOG_ID}")
        }
        val whereClause = buildString {
            append("${MessagesTable.MESSAGES_TABLE}.${MessagesTable.KEY_DIALOG_ID} = ? ")
            append("AND ${MessagesTable.MESSAGES_TABLE}.${MessagesTable.KEY_SERVER_SEQUENCE} != ? ")
            append("AND ${MessagesTable.MESSAGES_TABLE}.${MessagesTable.KEY_SERVER_SEQUENCE} != ? ")
            append("AND ${MessagesTable.MESSAGES_TABLE}.${MessagesTable.KEY_STATUS} = ? ")
            append("AND ${DialogsTable.TABLE_NAME}.${DialogsTable.Key.BRAND_ID} = ?")
        }
        val whereArgs = arrayOf(
            Dialog.OFFLINE_DIALOG_ID,
            AmsMessages.WELCOME_MSG_SEQUENCE_NUMBER.toString(),
            AmsMessages.OUTBOUND_CAMPAIGN_MSG_SEQUENCE_NUMBER.toString(),
            MessageState.OFFLINE.ordinal.toString(),
            brandId
        )
        try {
            db.query(joinColumn, null, whereClause, whereArgs, null, null, null, "1")
                .use { cursor -> return cursor.moveToFirst() }
        } catch (exception: Exception) {
            d(TAG, "areOfflineMessagesExists", exception)
            return false
        }
    }

    override fun isOfflineWelcomeMessageExists(dialogId: String): Boolean {
        val whereClause = buildString {
            append("(${MessagesTable.KEY_DIALOG_ID} = ? OR ${MessagesTable.KEY_DIALOG_ID} = ? )")
            append(" AND ")
            append("(${MessagesTable.KEY_SERVER_SEQUENCE} = ? OR ${MessagesTable.KEY_SERVER_SEQUENCE} = ? )")
        }
        val whereArgs = arrayOf(
            AmsDialogs.KEY_WELCOME_DIALOG_ID,
            dialogId,
            AmsMessages.WELCOME_MSG_SEQUENCE_NUMBER.toString(),
            AmsMessages.OUTBOUND_CAMPAIGN_MSG_SEQUENCE_NUMBER.toString()
        )
        try {
            db.query(null, whereClause, whereArgs, null, null, null, "1")
                .use { cursor -> return cursor.moveToFirst() }
        } catch (exception: Exception) {
            d(TAG, "isOfflineWelcomeMessageExists", exception)
            return false
        }
    }

    override fun updateOfflineMessagesDialogId(actualDialogId: String) {
        val contentValues = ContentValues()
        contentValues.put(MessagesTable.KEY_DIALOG_ID, actualDialogId)
        val whereClause = MessagesTable.KEY_DIALOG_ID + " =? "
        val whereArgs = arrayOf(Dialog.OFFLINE_DIALOG_ID)
        val updatedRows: Int = db.update(contentValues, whereClause, whereArgs)
        if (updatedRows > 0) {
            triggerUpdateMessagesForDialogId(actualDialogId)
        }
    }

    override fun removeOfflineWelcomeMessage() {
        val eventId =
            AmsDialogs.KEY_WELCOME_DIALOG_ID + AmsMessages.WELCOME_MESSAGE_EVENT_ID_POSTFIX
        val whereClause = (MessagesTable.KEY_TIMESTAMP + " = ?"
                + " AND " + MessagesTable.KEY_SERVER_SEQUENCE + " = ?"
                + " AND " + MessagesTable.KEY_EVENT_ID + " = ?")
        val whereArgs = arrayOf(
            AmsMessages.OFFLINE_WELCOME_MESSAGE_TIMESTAMP.toString(),
            AmsMessages.WELCOME_MSG_SEQUENCE_NUMBER.toString(),
            eventId
        )
        val result: Int = db.removeAll(whereClause, whereArgs)
        if (result > 0) {
            offlineMessagesListener.removeMessageByEventId(eventId)
        }
    }

    override fun removePendingOfflineMessages(brandId: String) {
        val eventIds = getPendingOfflineMessages(brandId)
        if (eventIds.isNotEmpty()) {
            val whereClause = MessagesTable.KEY_EVENT_ID + " in (?)"
            val whereArgs = arrayOf(eventIds.joinToSQLString(","))
            db.removeAll(whereClause, whereArgs)
        }
        setPendingOfflineMessages(brandId, HashSet())
    }

    override fun setPendingOfflineMessages(brandId: String, eventIds: Set<String>) {
        preferenceManager.setStringsSet(KEY_OFFLINE_MESSAGES_EVENTS, brandId, eventIds)
    }

    override fun getPendingOfflineMessages(brandId: String): Set<String> {
        return preferenceManager.getStringSet(
            KEY_OFFLINE_MESSAGES_EVENTS,
            brandId,
            HashSet<String>()
        )
    }

    override fun removePendingOfflineMessage(brandId: String, eventId: String) {
        val pendingOfflineMessages = getPendingOfflineMessages(brandId).toMutableSet()
        pendingOfflineMessages.remove(eventId)
        setPendingOfflineMessages(brandId, pendingOfflineMessages)
    }

    override fun isMessagePending(brandId: String, eventId: String): Boolean {
        val pendingOfflineMessages = getPendingOfflineMessages(brandId)
        return pendingOfflineMessages.contains(eventId)
    }

    override fun triggerUpdateMessagesForDialogId(dialogId: String) {
        db.query(
            arrayOf(
                "MIN(" + MessagesTable.KEY_TIMESTAMP + ")",
                "MAX(" + MessagesTable.KEY_TIMESTAMP + ")"
            ),
            MessagesTable.KEY_DIALOG_ID + " = ?",
            arrayOf(dialogId),
            null,
            null,
            null
        )?.use { cursor ->
            if (cursor.moveToFirst()) {
                val firstMessageTimestampForConversation = cursor.getLong(0)
                val lastMessageTimestampForConversation = cursor.getLong(1)
                offlineMessagesListener.onUpdateMessages(
                    firstMessageTimestampForConversation,
                    lastMessageTimestampForConversation
                )
            }
        }
    }

    private fun getPendingMessagesQueryParams(
        brandId: String,
        rowIds: String,
        state: String
    ): Array<String> {
        // If there are images in upload progress add them to the where clause
        return if (!TextUtils.isEmpty(rowIds)) {
            arrayOf(brandId, state, rowIds)
        } else { // where without image rowId
            arrayOf(brandId, state)
        }
    }

    private fun getPendingMessagesQuery(rowIds: String): String {
        val whereBuilder = StringBuilder().append(MessagesTable.KEY_ID).append(" in (select m.")
            .append(MessagesTable.KEY_ID).append(" from ")
            .append(MessagesTable.MESSAGES_TABLE).append(" m , ")
            .append(DialogsTable.TABLE_NAME).append(" c ")
            .append("where (").append("m.")
            .append(MessagesTable.KEY_STATUS).append("=")
            .append(MessageState.OFFLINE.ordinal)
            .append(") and c.").append(DialogsTable.Key.BRAND_ID).append("=?")
            .append(" and c.").append(DialogsTable.Key.STATE).append("=?")
            .append(" and m.").append(MessagesTable.KEY_DIALOG_ID).append("= c.")
            .append(DialogsTable.Key.DIALOG_ID)

        // If there are images in upload progress add them to the where clause
        if (!TextUtils.isEmpty(rowIds)) {
            d(
                TAG,
                "resendAllPendingMessages: There is upload images in progress, ignore these messages rowId: $rowIds"
            )
            whereBuilder.append(" and m.").append(MessagesTable.KEY_ID).append(" not in (?)")
        }
        whereBuilder.append(")") // Close the where clause
        val where = whereBuilder.toString()
        d(TAG, "getPendingMessagesQuery: where clause: $where")
        return where
    }

    /**
     * Listener used for triggering events on the UI when certain events occur
     */
    interface OfflineMessagesListener {
        /**
         * Method that is called when messages are updated
         * @param firstTimestamp initial time stamp
         * @param lastTimestamp updated time stamp
         */
        fun onUpdateMessages(firstTimestamp: Long, lastTimestamp: Long)

        /**
         * Method that is called when a message is removed
         * @param eventId the event id associated with the message
         */
        fun removeMessageByEventId(eventId: String)
    }
}
