package com.liveperson.messaging.wm

import android.content.Context
import android.content.SharedPreferences
import com.liveperson.infra.configuration.Configuration
import com.liveperson.infra.messaging.R
import com.liveperson.infra.model.LPWelcomeMessage
import com.liveperson.infra.model.LPWelcomeMessage.MessageFrequency
import com.liveperson.messaging.Messaging
import com.liveperson.messaging.wm.WelcomeMessageBroadcastReceiver.Companion.ACTION_WELCOME_MESSAGE_RESPONSE
import com.liveperson.messaging.wm.WelcomeMessageBroadcastReceiver.Companion.sendWelcomeMessageEvent
import com.liveperson.monitoring.MonitoringFactory
import org.json.JSONObject
import java.util.concurrent.Executors
import java.util.concurrent.ScheduledExecutorService
import java.util.concurrent.ScheduledFuture
import java.util.concurrent.TimeUnit

/**
 * Class used to request or manage a dynamic welcome message.
 * Using this class SDK could set or get welcome message synchoronously,
 * request a welcome message with required timeout (set in welcome_message_delay_in_seconds
 * integer resource).
 *
 * @param sharedPreferences - multi process preferences to store dynamic welcome message structure.
 * @param responseSender - context aware component used to send a broadcast intents about
 * welcome message response.
 * @param defaultWelcomeMessageProvider - default welcome message factory which creates a welcome message
 * from conversation view params.
 * @param timeOutProvider - provider of welcome message delay in minutes.
 *
 * @see Messaging
 * @see WelcomeMessageContentProvider
 */
class WelcomeMessageManager private constructor(
    private val sharedPreferences: SharedPreferences,
    private val responseSender: (String) -> Unit,
    private val defaultWelcomeMessageProvider: () -> LPWelcomeMessage,
    private val timeOutProvider: () -> Long
) {

    companion object {
        private const val KEY_DYNAMIC_WELCOME_MESSAGE = "welcome.message"
        private const val KEY_DYNAMIC_WELCOME_MESSAGE_FREQUENCY = "welcome.message.frequency"

        private const val JSON_KEY_WELCOME_MESSAGE_CONTENT = "content"
        private const val JSON_KEY_WELCOME_MESSAGE_ACTIONS = "actions"
        private const val JSON_KEY_WELCOME_MESSAGE_FREQUENCY = "frequency"


        /**
         * Extension property used to receive welcome message from conversation view
         * params. Note, this method won't return a dynamic welcome message.
         * This method just used to encapsulate logic of welcome message retrieval.
         */
        private val Messaging.defaultWelcomeMessage: LPWelcomeMessage
            get() {
                val message = conversationViewParams.lpWelcomeMessage
                return if (!message.welcomeMessage.isNullOrBlank()) {
                    message
                } else {
                    LPWelcomeMessage(applicationContext.getString(R.string.lp_first_message))
                }
            }

        /**
         * Extension method for shared prefences used to store welcome message
         * structure as json object string.
         * Welcome message's content, actions and frequency are stored as separate
         */
        private fun SharedPreferences.setWelcomeMessage(brandId: String, welcomeMessage: LPWelcomeMessage) {
            val json = JSONObject()
            json.put(JSON_KEY_WELCOME_MESSAGE_CONTENT, welcomeMessage.welcomeMessage)
            json.put(JSON_KEY_WELCOME_MESSAGE_ACTIONS, welcomeMessage.messageOptions.serialize())
            json.put(JSON_KEY_WELCOME_MESSAGE_FREQUENCY, welcomeMessage.messageFrequency.ordinal)

            edit().putString(KEY_DYNAMIC_WELCOME_MESSAGE + brandId, json.toString())
                .putInt(KEY_DYNAMIC_WELCOME_MESSAGE_FREQUENCY + brandId, welcomeMessage.messageFrequency.ordinal)
                .commit();
        }

        /**
         * Method used to create welcome message from stored in share preference json.
         */
        private fun createWelcomeMessageFromJson(content: String): LPWelcomeMessage = with(JSONObject(content)) {
            val welcomeMessage = LPWelcomeMessage(optString(JSON_KEY_WELCOME_MESSAGE_CONTENT)).apply {
                val frequency = optInt(JSON_KEY_WELCOME_MESSAGE_FREQUENCY, MessageFrequency.FIRST_TIME_CONVERSATION.ordinal)
                messageFrequency = MessageFrequency.fromOrdinal(frequency)
                messageOptions = optJSONArray(JSON_KEY_WELCOME_MESSAGE_ACTIONS)?.toMessageOptions() ?: emptyList()
            }
            return welcomeMessage
        }

        @JvmStatic
        fun newInstance(messaging: Messaging): WelcomeMessageManager {
            val context = messaging.applicationContext
            return WelcomeMessageManager(
                context.getWelcomeMessagePreferences(),
                responseSender = { context.sendWelcomeMessageEvent(ACTION_WELCOME_MESSAGE_RESPONSE + it) },
                defaultWelcomeMessageProvider = { messaging.defaultWelcomeMessage },
                timeOutProvider = { Configuration.getInteger(R.integer.lp_welcome_message_delay_in_seconds).toLong() }
            )
        }

        /**
         * Method used to set a dynamic welcome message
         * without initialization of welcome message manager.
         * This method should be used in public APIs
         * that doesn't contain a messaging class.
         */
        @JvmStatic
        fun setWelcomeMessage(context: Context, brandId: String, welcomeMessage: LPWelcomeMessage?) {
            val preferences = context.applicationContext.getWelcomeMessagePreferences()
            if (welcomeMessage == null) {
                preferences.edit().clear().commit()
            } else {
                preferences.setWelcomeMessage(brandId, welcomeMessage)
            }
        }

        /**
         * Method used get a dynamic welcome message without initialization of welcome message manager.
         * This method could be used
         */
        @JvmStatic
        fun getWelcomeMessage(context: Context, brandId: String): LPWelcomeMessage? {
            val preferences = context.applicationContext.getWelcomeMessagePreferences()
            val content = preferences.getString(KEY_DYNAMIC_WELCOME_MESSAGE + brandId, "")
                .takeUnless { it.isNullOrBlank() }
                ?: return null

            return createWelcomeMessageFromJson(content)
        }
    }

    private val requestsExecutor: ScheduledExecutorService = Executors.newSingleThreadScheduledExecutor()

    private val pendingRequests: MutableList<ScheduledFuture<*>> = mutableListOf()

    /**
     * Method used to request welcome message from brand's app.
     *
     * This method just provides a delay for welcome message response
     * to wait until brand changes welcome message or timeout reaches.
     *
     * Once timeout (provided in R.integer.welcome_message_delay_in_seconds xml integer value)
     * was reached welcome message manager will send a broadcast event.
     *
     * @param brandId id of brand which should provide a welcome message. Actually it would be an
     * active brand for which conversation screen was previously shown.
     */
    @Synchronized
    fun requestNewWelcomeMessage(brandId: String) {
        val delay = timeOutProvider()
        val task = NotifierTask(brandId, responseSender)
        if (delay <= 0L) {
            requestsExecutor.execute(task)
        } else {
            pendingRequests.add(requestsExecutor.schedule(task, delay, TimeUnit.SECONDS))
        }
    }

    /**
     * Method used to cancel all pending task used to broadcast event
     * about welcome message responses. Used to prevent multiple responses
     * for requested welcome messages.
     * Note: this method doesn't produce any events about request's cancellation.
     */
    @Synchronized
    fun cancelTimeoutTasks() {
        pendingRequests.forEach { if (!it.isCancelled && !it.isDone) it.cancel(true) }
        pendingRequests.clear()
    }

    /**
     * Method used to clear a welcome message storage.
     * @see Messaging.clear
     */
    fun clearWelcomeMessage() {
        sharedPreferences.edit().clear().commit()
    }

    /**
     * Method used to receive welcome message frequency without parsing
     * the content of stored welcome message.
     */
    fun getWelcomeMessageFrequency(brandId: String): MessageFrequency {
        if (MonitoringFactory.monitoring.initialized &&
            MonitoringFactory.monitoring.welcomeMessageMap.containsKey(brandId)) {
            val ccuiWM = MonitoringFactory.monitoring.welcomeMessageMap[brandId]
            if (ccuiWM != null) {
                return ccuiWM.messageFrequency
            }
        }

        val message = defaultWelcomeMessageProvider()
        val frequency = sharedPreferences.getInt(KEY_DYNAMIC_WELCOME_MESSAGE_FREQUENCY + brandId, -1)
        return if (frequency in 0..1) {
            MessageFrequency.fromOrdinal(frequency)
        } else {
            message.messageFrequency
        }
    }

    /**
     * Method used to get welcome message for required brand by its id.
     * If there is no dynamic welcome message stored in preferences this method
     * will return a welcome message provided in conversation view params.
     */
    fun getWelcomeMessage(brandId: String): LPWelcomeMessage {
        if (MonitoringFactory.monitoring.initialized &&
            MonitoringFactory.monitoring.welcomeMessageMap.containsKey(brandId)) {
            val ccuiWM = MonitoringFactory.monitoring.welcomeMessageMap[brandId]
            if (ccuiWM != null) {
                return ccuiWM
            }
        }

        val content = sharedPreferences.getString(KEY_DYNAMIC_WELCOME_MESSAGE + brandId, "")
        return if (!content.isNullOrBlank()) {
            val welcomeMessage = createWelcomeMessageFromJson(content)
            welcomeMessage.takeUnless { it.welcomeMessage.isNullOrBlank() } ?: defaultWelcomeMessageProvider()
        } else {
            defaultWelcomeMessageProvider()
        }
    }

    /**
     * Method used shut down internal queue to release resource, once
     * logout from SDK occurred.
     */
    fun dispose() {
        requestsExecutor.shutdown()
    }

    /**
     * Implementation of deferred task used to notify SDK components about
     * welcome message response.
     */
    private class NotifierTask(private val brandId: String, private val intentSender: (String) -> Unit): Runnable {
        override fun run() {
            intentSender(brandId)
        }
    }
}