package com.liveperson.messaging.wm.core

import android.content.ContentValues
import android.content.SharedPreferences
import org.json.JSONArray

/**
 * Implementation of shared preferences editor for
 * multi-process environment.
 *
 * @param contentValues default empty content values
 * that would be written to shared preferences.
 *
 * @param editorHelper helper class used to write content
 * values passed to editor using ContentResolver API.
 */
internal class MultiProcessEditor constructor(
    private val contentValues: ContentValues,
    private val editorHelper: MultiProcessEditorHelper
): SharedPreferences.Editor {
    override fun putString(key: String?, value: String?): SharedPreferences.Editor = apply {
        contentValues.put(key, value)
    }

    override fun putStringSet(
        key: String?,
        values: MutableSet<String>?
    ): SharedPreferences.Editor = apply {
        val serializedArray = JSONArray()
            .apply { values?.forEach { put(it) } }
            .toString()
        contentValues.put(key, serializedArray)
    }

    override fun putInt(key: String?, value: Int): SharedPreferences.Editor = apply {
        contentValues.put(key, value)
    }

    override fun putLong(key: String?, value: Long): SharedPreferences.Editor = apply {
        contentValues.put(key, value)
    }

    override fun putFloat(key: String?, value: Float): SharedPreferences.Editor = apply {
        contentValues.put(key, value)
    }

    override fun putBoolean(key: String?, value: Boolean): SharedPreferences.Editor = apply {
        contentValues.put(key, value)
    }

    override fun remove(key: String): SharedPreferences.Editor = apply {
        editorHelper.remove(key)
    }

    override fun clear(): SharedPreferences.Editor = apply {
        editorHelper.clear()
    }

    override fun commit(): Boolean = editorHelper.commit(contentValues)

    override fun apply() = editorHelper.apply(contentValues)
}