package com.liveperson.messaging.model;

import androidx.annotation.Nullable;
import android.text.TextUtils;

import com.liveperson.api.response.model.MultiDialog;
import com.liveperson.api.response.model.UserProfile;
import com.liveperson.api.response.types.CloseReason;
import com.liveperson.api.response.types.DialogType;
import com.liveperson.infra.ICallback;
import com.liveperson.infra.log.LPLog;
import com.liveperson.infra.utils.EncryptionVersion;
import com.liveperson.messaging.Messaging;
import com.liveperson.messaging.network.http.AgentProfileRequest;
import com.liveperson.infra.utils.ClockUtils;

import static com.liveperson.infra.errors.ErrorCode.ERR_00000075;

public class DialogUtils {

    private static final String TAG = "DialogUtils";
    protected final Messaging mController;

    public DialogUtils(Messaging controller) {
        mController = controller;
    }

    /**
     * Add resolve message and notify to the UI that the conversation has resolved
     *
     * @param brandId The brand id
     * @param dialog  The dialog
     * @param assignedAgentServerId The agent id
     * @param resolvedBy Close reason
     * @param updateUI If UI should be updated
     * @param callback The callback
     */
    public void addClosedDialogDivider(final String brandId, final Dialog dialog, String assignedAgentServerId, CloseReason resolvedBy, final boolean updateUI, @Nullable final ICallback<Void, Exception> callback) {
        mController.amsMessages.isResolveMessageForDialogAdded(dialog.getDialogId()).setPostQueryOnBackground(resolveMessageExists -> {
            LPLog.INSTANCE.d(TAG, "DIALOG " + dialog.getDialogId() + " resolved? " + resolveMessageExists + " reason " + resolvedBy);
            if (resolveMessageExists) {
                LPLog.INSTANCE.d(TAG, "resolved divider for dialog: " + dialog.getDialogId() + " already exists!");
                if (callback != null) {
                    callback.onSuccess(null);
                }
            } else if (dialog.getChannelType() != MultiDialog.ChannelType.COBROWSE) {
                final String originatorId;
                if (resolvedBy == CloseReason.CONSUMER) {
                    originatorId = mController.getOriginatorId(brandId);
                } else if (Dialog.isAutoClose(resolvedBy)) {
                    originatorId = null;
                } else {
                    originatorId = assignedAgentServerId;
                }
                LPLog.INSTANCE.d(TAG, "adding resolved divider for dialog: " + dialog.getDialogId());
                addResolvedDivider(dialog, originatorId, resolvedBy, updateUI, callback);
            } else if (callback != null) {
                RuntimeException exception = new RuntimeException("Probably empty cursor. Look at AmsMessages.isResolveMessageForDialogAdded(...) method");
                LPLog.INSTANCE.w(TAG,"Probably empty cursor for dialog " + dialog.getDialogId() + " channel " + dialog.getChannelType(), exception);
                callback.onError(exception);
            }
        }).execute();
    }

    /**
     * Adds a divider between the messages, this methods usually helps to add a simple divider without a timestamp between dialogs.
     *
     * @param dialog       closed for which conversation resolved message should be added.
     * @param originatorId identifier of participant who resolved conversation.
     * @param resolvedBy   the reason of dialog resolving.
     * @param updateUI     flag whether divider should be represented on conversation screen after
     *                     saving it to database.
     * @param callback     callback to notify that conversation resolved message was added to
     *                     database successfully.
     */
    private void addResolvedDivider(final Dialog dialog, final String originatorId, CloseReason resolvedBy, final boolean updateUI, @Nullable final ICallback<Void, Exception> callback) {
        final long timestamp = ClockUtils.addDiffToTimestamp(dialog.getEndTimestamp());
        LPLog.INSTANCE.d(TAG, "Dialog + " + dialog.getDialogId() + " + was closed. Reason: " + resolvedBy.name());
        if (dialog.getDialogType() == DialogType.MAIN) {
            if (Dialog.isAutoClose(resolvedBy)) {
                LPLog.INSTANCE.d(TAG, "Dialog + " + dialog.getDialogId() + " was auto closed.");
                addResolveMessage(dialog.getDialogId(), timestamp, "", originatorId, updateUI, MessagingChatMessage.MessageType.SYSTEM_RESOLVED, AmsMessages.RESOLVED_AUTO_CLOSED_MSG_SEQUENCE_NUMBER, callback);
            } else {
                addResolveMessage(dialog.getDialogId(), timestamp, "", originatorId, updateUI, MessagingChatMessage.MessageType.SYSTEM_RESOLVED, AmsMessages.RESOLVE_MSG_SEQUENCE_NUMBER, callback);
            }
        } else {
            addResolveMessage(dialog.getDialogId(), timestamp, "dialog closed", originatorId, updateUI, MessagingChatMessage.MessageType.SYSTEM_DIALOG_RESOLVED, AmsMessages.RESOLVE_MSG_SEQUENCE_NUMBER, callback);
        }
    }

    /**
     * Method used to add conversation resolved or auto-closed dialog divider message
     * to database and represent it in conversation screen..
     * @param dialogId       identifier of closed dialog.
     * @param timestamp      date in unix-timestamp format when associated dialog was closed/resolved.
     * @param message        text of conversation resolved message.
     * @param originatorId   identifier of participant who resolved conversation.
     * @param updateUI       flag whether divider should be represented on conversation screen after
     *                       saving it to database.
     * @param messageType    type of message stored in database and represented on conversation screen.
     *                       Message type could be equal to {@link MessagingChatMessage.MessageType#SYSTEM_RESOLVED}
     *                       for main dialogs or {@link MessagingChatMessage.MessageType#SYSTEM_DIALOG_RESOLVED}
     *                       for other types of dialogs.
     * @param serverSequence server sequence of conversation resolved message.
     *                       Sequence could be equal to {@link AmsMessages#RESOLVE_MSG_SEQUENCE_NUMBER}
     *                       for conversations that were resolved by agent or consumer or
     *                       {@link AmsMessages#RESOLVED_AUTO_CLOSED_MSG_SEQUENCE_NUMBER}
     *                       for auto-closed messages.
     * @param callback       callback to notify that conversation resolved message was added to
     *                       database successfully.
     */
    private void addResolveMessage(
            String dialogId,
            long timestamp,
            String message,
            String originatorId,
            boolean updateUI,
            MessagingChatMessage.MessageType messageType,
            int serverSequence,
            final ICallback<Void, Exception> callback
    ) {
	    LPLog.INSTANCE.d(TAG, "creating message resolve at time: " + timestamp
                + " timestamp: " + timestamp);
        String eventId = dialogId + AmsMessages.RESOLVE_MESSAGE_EVENT_ID_POSTFIX;

        MessagingChatMessage systemResolved = new MessagingChatMessage(originatorId, message, timestamp, dialogId, eventId,
                messageType, MessagingChatMessage.MessageState.RECEIVED, EncryptionVersion.NONE);
        systemResolved.setServerSequence(serverSequence);
        mController.amsMessages.addMessage(systemResolved, updateUI).setPostQueryOnBackground(data -> {
            if (callback != null) {
                callback.onSuccess(null);
            }
        }).execute();
    }

    public void updateParticipants(final String targetId, String[] participants, final UserProfile.UserType userType,
                                   final String dialogId, final boolean updateUI, final boolean forceUpdate) {
        updateParticipants(targetId, participants, userType, dialogId, updateUI, forceUpdate, null);
    }

    public void updateParticipants(final String targetId, String[] participants, final UserProfile.UserType userType,
                                   final String dialogId, final boolean updateUI, final boolean forceUpdate, final ICallback<MessagingUserProfile, Exception> callback) {
        for (final String userId : participants) {
            if (!TextUtils.isEmpty(userId)) {
                //foreach participant - if does not exist in db - send getUserProfile request
                mController.amsUsers.getUserById(userId).setPostQueryOnBackground(userProfile -> {
                    if (userProfile == null) {
                        userProfile = new MessagingUserProfile("", "", userType);
                        userProfile.setOriginatorID(userId);
                        mController.amsMessages.onAgentReceived(userProfile);
                        mController.amsUsers.updateUserProfile(userProfile);

	                    LPLog.INSTANCE.i(TAG, "First time bringing information for another participant that joined dialog " + dialogId);
                        sendUpdateUserRequest(targetId, userId, dialogId, updateUI, callback);
                    } else if (userProfile.isEmptyUser() || forceUpdate) {
                        sendUpdateUserRequest(targetId, userId, dialogId, updateUI, callback);
                    } else if (callback != null) {
                        callback.onSuccess(userProfile);
                    }
                }).execute();
            } else {
                LPLog.INSTANCE.e(TAG, ERR_00000075, "Missing agent ID!");
            }
        }
    }

    /**
     * @param targetId
     * @param userId
     * @param dialogId
     * @param shouldUpdateUi
     * @param callback
     */
    private void sendUpdateUserRequest(String targetId, String userId, String dialogId, boolean shouldUpdateUi, ICallback<MessagingUserProfile, Exception> callback) {
        if (!TextUtils.isEmpty(userId)) {
            // There is an assigned agent, get his details and update
            new AgentProfileRequest(mController, targetId, userId, dialogId, shouldUpdateUi).setCallback(callback).execute();
        } else {
            if (TextUtils.isEmpty(dialogId)){
                LPLog.INSTANCE.d(TAG, "sendUpdateUserRequest: no dialog id");
                return;
            }

            // There is no assigned agent, get the conversation from the DB and update the callback
            mController.amsDialogs.queryDialogById(dialogId)
                    .setPostQueryOnBackground(dialog -> {
	                    LPLog.INSTANCE.d(TAG, "onResult: Calling agent details callback with null agent");
                        mController.onAgentDetailsChanged(null, dialog.isOpen());
                    }).execute();
        }
    }

}
