package com.liveperson.messaging.network.http

import android.net.Uri
import android.text.TextUtils
import com.liveperson.infra.Command
import com.liveperson.infra.ICallback
import com.liveperson.infra.log.LPLog.i
import com.liveperson.infra.log.LPLog.mask
import com.liveperson.infra.network.http.HttpHandler
import com.liveperson.infra.network.http.request.HttpGetRequest
import com.liveperson.infra.otel.LPTraceType

open class GetActiveConversationsRequest @JvmOverloads constructor(
    val data : GetActiveConversationsRequestData,
    val callback: ICallback<String?, Exception?>? = null,
) : Command {

    override fun execute() {
        val mRequestUrl = prepareRequestUrlWithQueryParameters()
        val httpGetRequest = HttpGetRequest(mRequestUrl, LPTraceType.UMS_GET_OPEN_CONVERSATIONS)
        httpGetRequest.timeout = GET_ACTIVE_CONVERSATIONS_REQUEST_TIMEOUT
        addHeaders(httpGetRequest);
        httpGetRequest.certificatePinningKeys = data.certificates
        httpGetRequest.callback = prepareCallback()
        HttpHandler.execute(httpGetRequest)
    }

    private fun addHeaders(httpGetRequest: HttpGetRequest) {
        httpGetRequest.addHeader("authorization", "Bearer ${data.lpToken}")
        httpGetRequest.addHeader("Brand-ID", data.siteId)
        httpGetRequest.addHeader("Client-source", "SDK_Android_UMS_HTTP")
    }

    private fun prepareCallback(): ICallback<String?, Exception?> {
        return object : ICallback<String?, Exception?> {
            override fun onSuccess(value: String?) {
                i(TAG, "onSuccess " + mask(value))
                if (!TextUtils.isEmpty(value)) {
                    callback?.let {
                        it.onSuccess(value)
                        return
                    }
                }
                callback?.onError(Exception("Error: empty response from the server"))
            }

            override fun onError(exception: Exception?) {
                exception?.apply {
                    callback?.let {
                        it.onError(exception)
                        return
                    }
                }
                callback?.onError(Exception("Error: request failed"))
            }
        }
    }

    private fun prepareRequestUrlWithQueryParameters(): String {
        val url = String.format(DOMAIN_TEMPLATE, data.baseURL, data.consumerId)
        val uriBuilder = Uri.parse(url).buildUpon()
        data.limit?.apply {
            uriBuilder.appendQueryParameter("limit", this)
        }

        data.sortBy?.apply {
            uriBuilder.appendQueryParameter("sortBy", this)
        }

        data.fields?.apply {
            uriBuilder.appendQueryParameter("fields", this)
        }

        data.filters?.apply {
            uriBuilder.appendQueryParameter("filters", this)
        }

        data.offset?.apply {
            uriBuilder.appendQueryParameter("offset", this)
        }

        data.sortOrder?.apply {
            uriBuilder.appendQueryParameter("sortOrder", this)
        }
        return uriBuilder.build().toString()
    }

    companion object {
        private const val TAG = "GetActiveConversationsRequest"
        private const val GET_ACTIVE_CONVERSATIONS_REQUEST_TIMEOUT = 30000
        private const val OPEN_CONVERSATION_FILTER_JSON = "{\"stage\":\"OPEN\"}"
        private const val DOMAIN_TEMPLATE =
            "https://%1\$s/messaging/v1/consumers/%2\$s/conversations"
    }

    /**
     * Class which holds a data needed to construct a GetActiveConversationsRequest
     *
     * @param baseUrl Domain used as root to build a request.
     * @param sideId Your brand's ID.
     * @param lpToken Token that is used to authenticate with UMS.
     * @param consumerId Id used to identify consumer.
     * @param limit Number of records to return (default 100)
     * @param offset The number of items to skip before starting to collect the result set (default 0)
     * @param sortBy Name of the field by which to sort the result set. Enum: "createdTs" "lastUpdatedTs". Default: "createdTs".
     * @param sortOrder Allows to order the result set in ascending or descending order. Enum: "ASC" "DESC". Default: "DESC"
     * @param filters Any filters to apply on the collection. This needs to be in a URL encoded json string. For example {stage:"OPEN"} should be passed as %7B%22stage%22%3A%22OPEN%22%7D
     * @param fields List of fields to include in the response. Fields must be separated by a comma. Examples:
     * fields=id - Get ID field only
     * fields=id,dialogs.state - Get nested fields
     * @param certificates List of certificates used for "Certificate pinning" feature.
     */
    data class GetActiveConversationsRequestData(
        val baseURL: String,
        val siteId: String,
        val lpToken: String,
        val consumerId: String,
        val limit: String? = null,
        var offset: String? = null,
        var sortBy: String? = null,
        var sortOrder: String? = null,
        var filters: String? = OPEN_CONVERSATION_FILTER_JSON,
        val fields: String? = "id",
        var certificates: List<String>? = null
        )
}
