package com.liveperson.messaging.offline

import androidx.annotation.WorkerThread
import com.liveperson.api.response.types.DialogType
import com.liveperson.infra.database.DataBaseCommand
import com.liveperson.infra.database.DataBaseCommand.QueryCommand
import com.liveperson.infra.database.DataBaseExecutor
import com.liveperson.infra.log.LPLog
import com.liveperson.infra.model.LPWelcomeMessage.MessageFrequency
import com.liveperson.messaging.model.AmsConnection
import com.liveperson.messaging.model.AmsDialogs
import com.liveperson.messaging.model.AmsMessages
import com.liveperson.messaging.model.Conversation
import com.liveperson.messaging.model.Dialog
import com.liveperson.messaging.model.FullMessageRow
import com.liveperson.messaging.offline.api.OfflineMessagesController
import com.liveperson.messaging.offline.api.OfflineConversationRepository
import com.liveperson.messaging.offline.api.OfflineDialogRepository
import com.liveperson.messaging.offline.api.OfflineMessagesRepository
import com.liveperson.messaging.wm.WelcomeMessageManager

/**
 * Class used as entry point for offline messaging flow.
 *
 * This class is responsible for:
 * - Offline conversation and dialog creation/retrieval.
 * - Offline messages operation(ex: attaching to active dialog, removing from db for PCS dialog).
 * - Offline mode environment(enabled, disabled, offline welcome message status).
 * - Representation of offline welcome message.
 * - Sending new conversation request if welcome message exists and consumer
 * doesn't have any open dialogs.
 */
class OfflineMessagingManager(
    private val offlineMessagesController: OfflineMessagesController,
    private val offlineConversationRepository: OfflineConversationRepository,
    private val offlineDialogRepository: OfflineDialogRepository,
    private val offlineMessagesRepository: OfflineMessagesRepository,
    private val welcomeMessageManager: WelcomeMessageManager
) : OfflineMessagesController by offlineMessagesController {

    companion object {
        private const val BEHAVIOR_DO_NOTHING = 0
        private const val BEHAVIOR_CLEAR_OFFLINE_MESSAGES = 1
        private const val BEHAVIOR_RESOLVE_PCS_DIALOG = 2

        private const val OFFLINE_CONVERSATION_REQUEST_ID = 0L

        private const val TAG = "OfflineManager"
    }

    /**
     * Method used to get an existed offline dialog from database/cache
     * or to create an offline conversation and related dialog for current brand.
     *
     * @param requestId request id for new conversation request associated with
     * offline dialog and conversation.
     */
    @JvmOverloads
    fun getOrCreateOfflineDialog(
        requestId: Long = OFFLINE_CONVERSATION_REQUEST_ID
    ): DataBaseCommand<Dialog> {
        val queryCommand = QueryCommand {
            val brandId = offlineMessagesController.currentBrandId
            val targetId = offlineMessagesController.currentBrandId
            val conversation: Conversation = offlineConversationRepository.getOrCreateOfflineConversation(
                targetId,
                brandId,
                requestId
            )
            offlineDialogRepository.getOrCreateOfflineDialog(
                targetId,
                brandId,
                conversation.conversationId,
                requestId
            )
        }
        return DataBaseCommand(queryCommand)
    }


    /**
     * Method used to process offline message for an active open dialog.
     * This method
     *
     * @param activeDialog current active open dialog for consumer.
     *
     * Depends of type of dialog this method will process existed offline messages
     * in such way:
     *
     * - If active dialog is type of [DialogType.MAIN] then all pending offline messages
     * will be attached to this dialog.
     *
     * - If active dialog is type of [DialogType.POST_SURVEY] then such behavior will be applied:
     *
     * 1. lp_offline_messages_for_pcs_behavior is set to 0 - Default behavior, the offline messages
     * will be sent to the PCS dialog.
     *
     * 2. lp_offline_messages_for_pcs_behavior is set to 1 - Discard the offline messages
     * when coming back online, allowing the PCS dialog to continue.
     * The offline messages will not be sent to the server
     * and they will be deleted from the local cache.
     *
     * 3. lp_offline_messages_for_pcs_behavior is set to 2 - Skip the PCS dialog automatically
     * and send all the offline messages to a new conversation.
     */
    fun processOfflineMessages(
        activeDialog: Dialog
    ) = DataBaseExecutor.execute {
        val areOfflineMessagesExist = offlineMessagesRepository.areOfflineMessagesExists(activeDialog.brandId)
        if (!areOfflineMessagesExist) {
            LPLog.d(TAG, "No offline messages found. Skipping offline messages processing.")
            clearOfflineConversation(activeDialog.targetId)
            return@execute
        }
        when (activeDialog.dialogType) {
            DialogType.POST_SURVEY -> {
                activeDialog.proceedPCSFlow()
            }
            else -> {
                if (activeDialog.lastServerSequence <= 0) {
                    showWelcomeMessage(activeDialog.brandId, activeDialog.dialogId)
                }
                val offlineMessages = loadOfflineMessages(activeDialog.brandId, activeDialog.dialogId)
                activeDialog.sendOfflineMessages(offlineMessages)
            }
        }
    }

    /**
     * Method used to create new conversation if consumer has pending
     * offline message that were sent in offline mode and consumer
     * doesn't have any open conversation after syncing the history.
     *
     * - Check [AmsConnection.setIsUpdated] to understand when it is called
     * - Check [AmsDialogs.updateClosedDialog] to understand when it is called f
     * for recently closed PCS dialog
     */
    fun createNewConversationIfNeeded() = DataBaseExecutor.execute {
        if (!offlineMessagesController.isOfflineModeEnabled) {
            LPLog.d(TAG, "Offline mode is disabled. Skipping sending offline messages.")
            return@execute
        }
        val brandId = offlineMessagesController.currentBrandId
        if (brandId.isEmpty()) {
            LPLog.d(TAG, "Brand id is empty. Skipping sending offline messages.")
            return@execute
        }
        val areOfflineMessagesExist = offlineMessagesRepository.areOfflineMessagesExists(brandId)
        if (!areOfflineMessagesExist) {
            LPLog.d(TAG, "No offline messages found. Skipping creation of new conversation.")
            return@execute
        }

        // we need a real active dialog to determine whether
        // consumer requires new conversation or offline messages
        // could be attached to current active dialog
        val activeDialog = offlineDialogRepository.cachedActiveDialog
            ?.takeUnless { it.dialogId == Dialog.OFFLINE_DIALOG_ID }
            ?.takeUnless { it.dialogId == Dialog.TEMP_DIALOG_ID }
            ?.takeUnless { it.isClosed }
            ?: offlineDialogRepository.queryRealActiveDialog(brandId)

        if (activeDialog == null) {
            val dialog = getOrCreateOfflineDialog().executeSynchronously()
            LPLog.d(TAG, "Sending new conversation request for offline dialog: $dialog")
            offlineMessagesController.sendNewConversationRequest(0)
        }
    }

    /**
     * Method used to clear all pending offline messages from database and shared preferences
     * before loading conversation when user navigates to conversation screen.
     *
     * This method prevents user from representation of pending offline messages
     * that were previously attached to a wrong conversation.
     */
    fun clearPendingOfflineMessages(): DataBaseCommand<Void> {
        return DataBaseCommand {
            LPLog.d(TAG, "Started removing pending offline messages")
            try {
                offlineMessagesRepository.removePendingOfflineMessages(currentBrandId)
                LPLog.d(TAG, "Pending offline messages successfully deleted")
            } catch (ex: Throwable) {
                LPLog.d(TAG, "Error occurred while clearing pending offline messages.")
            }
            return@DataBaseCommand null
        }
    }

    /**
     * Method used for removing the previously stored event ids for pending offline message from
     * shared prefs
     * @param eventId the event id associated with the pending offline message
     */
    fun removePendingOfflineMessage(brandId: String, eventId: String) {
        offlineMessagesRepository.removePendingOfflineMessage(brandId, eventId)
    }

    /**
     * Method used to load all offline messages from database.
     * This method return list of rows that are not in a pending state.
     *
     * @param brandId requested brand's identifier
     * @param brandId requested active dialog's identifier
     */
    private fun loadOfflineMessages(brandId: String, dialogId: String): List<FullMessageRow> {
        offlineMessagesRepository.updateOfflineMessagesDialogId(dialogId)
        return offlineMessagesRepository.loadOfflineMessagesExcept(
            brandId,
            offlineMessagesRepository.getPendingOfflineMessages(brandId)
        )
    }

    /**
     * Method used to show offline welcome message depends on
     * welcome message frequency and offline welcome message
     * configuration flag.
     * If offline message exists for particular dialog id, then
     * offline welcome message would not be shown for active dialog
     * once again.
     *
     * @param brandId requested active brand identifier.
     * @param dialogId requested active dialog identifier.
     */
    @WorkerThread
    private fun showWelcomeMessage(brandId: String, dialogId: String) {
        if (!offlineMessagesController.isOfflineWelcomeMessageEnabled) {
            LPLog.d(TAG, "Offline welcome message is disabled.")
            return
        }
        if (offlineMessagesRepository.isOfflineWelcomeMessageExists(dialogId)) {
            LPLog.d(TAG, "Offline welcome message was already added to database.")
            return
        }
        val currentWelcomeMessage = welcomeMessageManager.getWelcomeMessage(brandId)
        when {
            currentWelcomeMessage.messageFrequency == MessageFrequency.EVERY_CONVERSATION -> {
                LPLog.d(TAG, "Welcome message is set for every conversation. Processing...")
                offlineMessagesController.showOfflineWelcomeMessage(brandId, dialogId, currentWelcomeMessage)
            }
            // Because of data race we need to check
            // whether user contains only less than 2 conversations
            // because conversations list could be cleared or
            // user without history has already created current conversation.
            offlineConversationRepository.queryRealConversationsCount() < 2 -> {
                LPLog.d(TAG, "Welcome message frequency is set to first conversation. Processing...")
                offlineMessagesController.showOfflineWelcomeMessage(brandId, dialogId, currentWelcomeMessage)
            }
            else -> {
                LPLog.d(TAG, "Welcome message should not be shown for this flow.")
            }
        }
    }

    /**
     * Extension function over dialog to process offline messages for
     * active pcs dialog.
     *
     * Behavior of this method relies on lp_offline_messages_for_pcs_behavior value.
     *
     * 1. lp_offline_messages_for_pcs_behavior is set to 0 - Default behavior, the offline messages
     * will be sent to the PCS dialog.
     *
     * 2. lp_offline_messages_for_pcs_behavior is set to 1 - Discard the offline messages
     * when coming back online, allowing the PCS dialog to continue.
     * The offline messages will not be sent to the server
     * and they will be deleted from the local cache.
     *
     * 3. lp_offline_messages_for_pcs_behavior is set to 2 - Skip the PCS dialog automatically
     * and send all the offline messages to a new conversation.
     */
    @WorkerThread
    private fun Dialog.proceedPCSFlow() {
        LPLog.d(TAG, "Handling active PCS dialog.")
        offlineMessagesRepository.removeOfflineWelcomeMessage()
        when (val dialogBehavior = offlineMessagesController.offlineModePCSBehavior) {
            BEHAVIOR_DO_NOTHING -> {
                val offlineMessages = loadOfflineMessages(brandId, dialogId)
                sendOfflineMessages(offlineMessages)
            }
            BEHAVIOR_CLEAR_OFFLINE_MESSAGES -> {
                val offlineMessages = loadOfflineMessages(brandId, dialogId)
                offlineMessages.forEach {
                    offlineMessagesRepository.removeMessage(it.messagingChatMessage.eventId)
                }
                offlineMessagesRepository.triggerUpdateMessagesForDialogId(dialogId)
                clearOfflineConversation(targetId)
            }
            BEHAVIOR_RESOLVE_PCS_DIALOG -> {
                offlineMessagesController.resolveActiveDialog(dialogId, null)
            }
            else -> {
                LPLog.d(
                    TAG,
                    "Unsupported PCS dialog behavior value: $dialogBehavior. Sending messages as is"
                )
                val offlineMessages = loadOfflineMessages(brandId, dialogId)
                sendOfflineMessages(offlineMessages)
            }
        }
    }

    /**
     *  Extension function over dialog used to send offline message to the actice
     *  dialog.
     *
     *  @param messages offline message that were not previously sent.
     *
     *  Note: this method will add all sent messages event ids to pending set until
     *  content event with actual timestamp was not received from UMS.
     *
     *  Check [AmsMessages.requestMessagesUpdates] to understand pending messages are
     *  processed by SDK.
     */
    @WorkerThread
    private fun Dialog.sendOfflineMessages(messages: List<FullMessageRow>) {
        val pendingOfflineMessages =
            offlineMessagesRepository.getPendingOfflineMessages(currentBrandId).toMutableSet()
        messages.asSequence().filterNot {
            pendingOfflineMessages.contains(it.messagingChatMessage.eventId)
        }.forEach { row ->
            val chatMessage = row.messagingChatMessage
            LPLog.d(TAG, "Sending offline message: ${LPLog.mask(chatMessage)}")
            pendingOfflineMessages.add(chatMessage.eventId)
            offlineMessagesController.sendOfflineMessage(row)
        }
        offlineMessagesRepository.setPendingOfflineMessages(brandId, pendingOfflineMessages)
        clearOfflineConversation(targetId)
    }

    /**
     * Message used to remove offline dialog and associated offline conversation
     * from db once all message were removed from DB or all messages were attached
     * to the active conversation.
     *
     * Check [sendOfflineMessages] for case when all messages were attached to active fragment.
     * Check [proceedPCSFlow] for case when all messages were removed from DB
     * or attached to active dialog.
     */
    @WorkerThread
    private fun clearOfflineConversation(targetId: String) {
        LPLog.d(TAG, "Clear offline dialog and conversation")
        offlineDialogRepository.clearOfflineDialog(targetId)
        offlineConversationRepository.clearOfflineConversation(targetId)
    }

    /**
     * Method used to check whether current PCS behavior for offline mode
     * is equal to 2, which means that all sent offline messages should
     * be attach to newly created conversation.
     */
    @get:JvmName("shouldCreateConversationAfterPCS")
    val shouldCreateConversationAfterPCS: Boolean
        get() = offlineModePCSBehavior == BEHAVIOR_RESOLVE_PCS_DIALOG

    @get:JvmName("shouldShowPCSOptions")
    val shouldShowPCSOptions: Boolean
        get() = offlineModePCSBehavior == BEHAVIOR_CLEAR_OFFLINE_MESSAGES
}