package com.liveperson.messaging.utils

import android.content.Context
import com.liveperson.infra.ICallback
import com.liveperson.infra.Infra
import com.liveperson.infra.auth.LPAuthenticationParams
import com.liveperson.infra.errors.ErrorCode
import com.liveperson.infra.log.LPLog.d
import com.liveperson.infra.log.LPLog.e
import com.liveperson.infra.managers.ConsumerManager.Companion.getConsumerJWT
import com.liveperson.infra.model.Consumer
import com.liveperson.infra.model.types.FailureReason
import com.liveperson.messaging.LpError
import com.liveperson.messaging.MessagingFactory
import com.liveperson.messaging.TaskType
import com.liveperson.messaging.commands.tasks.BaseAmsAccountConnectionCallback
import com.liveperson.messaging.commands.tasks.GetActiveConversationsUmsTask
import com.liveperson.messaging.controller.AccountsController
import com.liveperson.messaging.controller.connection.ConnectionParamsCache
import com.liveperson.messaging.model.SynchronizedAuthenticationCompletedCallback
import org.json.JSONException

class GetActiveConversationsHelper @JvmOverloads constructor(
    private val context: Context,
    private val mBrandId: String,
    private val lpAuthParams: LPAuthenticationParams?,
    private val accountController: AccountsController = MessagingFactory.getInstance().controller.mAccountsController,
    private val consumer: Consumer? = Infra.instance.consumerManager.getActiveConsumer(),
    private val umsDomain: String? = MessagingFactory.getInstance().controller.mAccountsController
        .getServiceUrl(mBrandId, ConnectionParamsCache.CSDS_UMS_DOMAIN_KEY),
) {

    var getActiveConversationsUmsTask: GetActiveConversationsUmsTask? = null

    fun checkActiveConversations(
        callback: ICallback<List<String>, Exception>,
    ) {
        if (umsDomain == null || consumer?.consumerId == null
            // user was switched (Implicit)
            || consumer.getConsumerJWT()?.equals(lpAuthParams?.hostAppJWT) == false
            // user was switched (Code)
            || consumer.lpAuthenticationParams?.authKey?.equals(lpAuthParams?.authKey) == false
        ) {
            authorizeAndSendRequest(callback)
        } else {
            launchGetActiveConversationsUmsTask(callback)
        }
    }

    private fun authorizeAndSendRequest(callback: ICallback<List<String>, Exception>) {
        val isExecuting = SynchronizedAuthenticationCompletedCallback(
            accountController,
            mBrandId,
            object : ICallback<Void?, java.lang.Exception?> {

                override fun onSuccess(value: Void?) {
                    try {
                        launchGetActiveConversationsUmsTask(callback)
                    } catch (error: java.lang.Exception) {
                        e(
                            "TAG",
                            ErrorCode.ERR_000000D9,
                            "sendRequest: Failed to obtain domain/consumerId/token to make request ",
                            error
                        )
                    }
                }

                override fun onError(exception: java.lang.Exception?) {
                    e(
                        "TAG",
                        ErrorCode.ERR_000000DA,
                        "authorizeAndSendRequest: Failed to authorize ",
                        exception
                    )
                }
            }).executeWithReturnValue()

        // If we have valid jwt we will execute the request. otherwise, we'll try to connect.
        if (!isExecuting && lpAuthParams != null) {
            MessagingFactory.getInstance().controller.connectForApiUsage(mBrandId, lpAuthParams, null, true)
        }
    }

    private fun launchGetActiveConversationsUmsTask(
        invokerCallback: ICallback<List<String>, Exception>
    ) {
        if (getActiveConversationsUmsTask == null) {
            getActiveConversationsUmsTask = GetActiveConversationsUmsTask(
                context,
                accountController,
                umsDomain,
                mBrandId
            )
        }

        getActiveConversationsUmsTask?.callback = (object : BaseAmsAccountConnectionCallback {
            override fun onTaskSuccess() {
                val result = getActiveConversationsUmsTask?.result
                if (result != null) {
                    try {
                        invokerCallback.onSuccess(result)
                    } catch (e: JSONException) {
                        invokerCallback.onError(e)
                    }
                }
                d(TAG, "onSuccess")
            }

            override fun onTaskError(type: TaskType, lpError: LpError, exception: Exception) {
                d(TAG, "onTaskError")
                invokerCallback.onError(exception)
            }

            override fun onTaskError(
                type: TaskType,
                lpError: LpError,
                reason: FailureReason,
                exception: Exception
            ) {
                d(TAG, "onTaskError")
                invokerCallback.onError(exception)
            }

            override fun setSecondaryTask(secondaryTask: Boolean) {
                d(TAG, "setSecondaryTask")
            }
        })
        getActiveConversationsUmsTask?.execute()
    }

    companion object {
        private const val TAG = "GetActiveConversationsHelper"
    }
}
