package com.liveperson.messaging.commands.pusher

import android.text.TextUtils
import com.liveperson.infra.Command
import com.liveperson.infra.ForegroundService
import com.liveperson.infra.ICallback
import com.liveperson.infra.InternetConnectionService
import com.liveperson.infra.errors.ErrorCode
import com.liveperson.infra.log.LPLog.d
import com.liveperson.infra.log.LPLog.e
import com.liveperson.infra.log.LPLog.i
import com.liveperson.infra.network.http.HttpHandler
import com.liveperson.infra.network.http.body.HttpRequestBody
import com.liveperson.infra.network.http.body.LPJSONObjectBody
import com.liveperson.infra.network.http.request.HttpPutRequest
import com.liveperson.infra.otel.LPTraceType
import com.liveperson.messaging.Messaging
import com.liveperson.messaging.controller.connection.ConnectionParamsCache
import com.liveperson.messaging.model.SynchronizedInternetConnectionCallback
import org.json.JSONArray
import org.json.JSONObject

/**
 * Request to clear unread messages count / badge from pusher for a given consumer id.
 * This command is similar to [SendReadAcknowledgementCommand] but did not rely up on conversation id.
 * This is helpful when we want all the unread badges for the consumer to be set to 0 despite
 * a specific conversation state (loaded or not on the screen).
 */
class ClearUnreadMessagesCountCommand(val messagingController: Messaging, val brandId: String, val callback: ICallback<List<String>, Exception>?) : Command {

    companion object {
        private const val TAG = "ClearUnreadMessagesCountCommand"
        private const val PUSHER_URL = "https://%s/api/account/%s/device/clear-badge-count"
        private const val CONSUMER_ID = "consumerId"

        private var hasClearedBadge = false

        fun clearRequestState() {
            i(TAG, "cleanRequestState")
            hasClearedBadge = false
        }
    }

    override fun execute() {
        if (hasClearedBadge || !ForegroundService.getInstance().isBrandForeground(brandId)) {
            i(TAG, "Unread count is already cleared. Ignore request.")
            return
        }
        hasClearedBadge = true
        val pusherDomain: String = messagingController.mAccountsController.getServiceUrl(
            brandId,
            ConnectionParamsCache.CSDS_PUSHER_DOMAIN_KEY
        )
        val requestURL =
            String.format(PUSHER_URL, pusherDomain, brandId)
        validateStateAndSendRequest(requestURL)
    }

    private fun validateStateAndSendRequest(requestURL: String) {
        if (InternetConnectionService.isNetworkAvailable()) {
            sendRequest(requestURL)
        } else {
            SynchronizedInternetConnectionCallback { sendRequest(requestURL) }.execute()
        }
    }

    private fun sendRequest(requestURL: String) {
        val token = messagingController.mAccountsController?.getToken(brandId)
        if (token.isNullOrEmpty()) {
            e(TAG, ErrorCode.ERR_00000169, "Authorization failed. Token is missing")
            callback?.onError(Exception("Authorization failed. Token is missing"))
            return
        }
        val requestJson = getBody()
        val httpPutRequest = HttpPutRequest(requestURL, LPTraceType.PUSHER_CLEAR_BADGE_COUNT_REQ)
        val jsonBody: HttpRequestBody = LPJSONObjectBody(requestJson)
        httpPutRequest.setBody(jsonBody)
        httpPutRequest.addHeader("authorization", "bearer $token")

        httpPutRequest.callback = object : ICallback<String, Exception?> {
            override fun onSuccess(value: String?) {
                try {
                    if (!TextUtils.isEmpty(value)) {
                        d(TAG, "onSuccess: $value")
                        val result = JSONArray(value)
                        callback?.onSuccess(result.toArrayList())
                    } else {
                        i(
                            TAG,
                            "Received empty response from pusher for clear-badge-count request"
                        )
                    }
                } catch (error: Exception) {
                    hasClearedBadge = false
                    e(
                        TAG,
                        ErrorCode.ERR_00000167,
                        "Failed to parse clear-badge-count response: ",
                        error
                    )
                    callback?.onError(error)
                }
            }

            override fun onError(exception: Exception?) {
                hasClearedBadge = false
                if (exception != null) {
                    e(TAG, ErrorCode.ERR_00000168, "Failed to clear badge count for the consumer")
                    callback?.onError(exception)
                }
            }
        }

        i(TAG, "Sending clear-badge-count to pusher")
        HttpHandler.execute(httpPutRequest)
    }

    private fun getBody(): JSONObject {
        val requestBody = JSONObject()
        try {
            requestBody.put(
                CONSUMER_ID,
                messagingController.amsUsers?.getConsumerId(brandId)
            )
        } catch (error: Exception) {
            hasClearedBadge = false
            e(TAG, ErrorCode.ERR_0000016A, "getBody: ", error)
        }
        return requestBody
    }

    private fun JSONArray.toArrayList(): ArrayList<String> {
        val list = arrayListOf<String>()
        for (i in 0 until this.length()) {
            list.add(this.getString(i))
        }
        return list
    }
}