package com.liveperson.messaging.commands.tasks

import com.liveperson.infra.log.LPLog
import com.liveperson.infra.model.types.AuthFailureReason
import com.liveperson.infra.model.types.FailureReason
import com.liveperson.messaging.LpError
import com.liveperson.messaging.TaskExecutionState
import com.liveperson.messaging.TaskType
import com.liveperson.messaging.model.AmsConnectionAnalytics
import java.lang.Exception

/**
 * ConcurrentTasks Is one of the connection tasks of the SDK. It combines multiple tasks together
 * and executes them simultaneously without waiting on prior task to finish.
 * If any of the tasks in execution fails and SDK starts performing exponential retries, Concurrent
 * tasks ensures execution of only failed task/s and prevents un-necessary execution prior successful tasks.
 */
class ConcurrentTasks : BaseAmsAccountConnectionTask() {
    companion object {
        private const val TAG = "ConcurrentTasks"
    }

    override fun execute() {
        AmsConnectionAnalytics.concurrentTasksStart()
        for ((task, state) in concurrentTasks) {
            synchronized(this) {
                if (state == TaskExecutionState.CREATED || state == TaskExecutionState.FAILURE) {
                    task.mBrandId = mBrandId
                    LPLog.d(TAG, "Executing task: ${task.name} with state: $state")
                    concurrentTasks[task] = TaskExecutionState.STARTED
                    task.execute()
                }
            }
        }
    }

    override fun getName(): String {
        return TAG
    }

    /**
     * Create list of concurrent tasks needs to be executed.
     */
     override fun addConcurrentTask(task: BaseAmsAccountConnectionTask) {
        task.callback = getCallBack(task)
         concurrentTasks[task] = TaskExecutionState.CREATED
    }

    /**
     * Return a new BaseAmsAccountConnectionCallback instance for each task to be executed.
     */
    private fun getCallBack(task: BaseAmsAccountConnectionTask): BaseAmsAccountConnectionCallback {
        return object: BaseAmsAccountConnectionCallback {
            override fun onTaskSuccess() {
                synchronized(this) {
                    LPLog.d(TAG, "Finished executing task: ${task.name}")
                    concurrentTasks[task] = TaskExecutionState.SUCCESS
                    if (successfullyExecutedAllTasks()) {
                        AmsConnectionAnalytics.concurrentTasksEnd()
                        mCallback?.onTaskSuccess()
                    }
                }
            }

            override fun onTaskError(type: TaskType?, lpError: LpError?, exception: Exception?) {
                LPLog.w(TAG, "Failed to execute task: ${task.name}")
                if (shouldNotifyFailedTask(lpError, null)) {
                    mCallback?.onTaskError(type, lpError, exception)
                }
                concurrentTasks[task] = TaskExecutionState.FAILURE
            }

            override fun onTaskError(
                type: TaskType?,
                lpError: LpError?,
                reason: FailureReason?,
                exception: Exception?
            ) {
                LPLog.w(TAG, "Failed to execute task: ${task.name}")
                if (shouldNotifyFailedTask(lpError, reason)) {
                    mCallback?.onTaskError(type, lpError, reason, exception)
                }
                concurrentTasks[task] = TaskExecutionState.FAILURE
            }

            override fun setSecondaryTask(secondaryTask: Boolean) {
                mCallback?.setSecondaryTask(secondaryTask)
            }

            /**
             * Check If all tasks in the list have successfully executed.
             */
            fun successfullyExecutedAllTasks(): Boolean {
                for (value in concurrentTasks.values) {
                    if (value.compareTo(TaskExecutionState.SUCCESS) != 0) {
                        return false
                    }
                }
                return true
            }

            /**
             * Notify task failure event to connection state machine if,
             * 1. Failure reason is IDP error: USER_EXPIRED
             * 2. Failure reason is IDP error: TOKEN_EXPIRED
             * 3. Failure reason is INVALID_CERTIFICATE
             * 4. No other tasks have already marked as failed (To ensure smooth exponential backoff retry)
             */
            fun shouldNotifyFailedTask( lpError: LpError?,
                                        reason: FailureReason?): Boolean {
                if (reason == AuthFailureReason.USER_EXPIRED
                    || reason == AuthFailureReason.TOKEN_EXPIRED
                    || reason == AuthFailureReason.INVALID_CERTIFICATE
                    || lpError == LpError.INVALID_SDK_VERSION
                    || lpError == LpError.INVALID_CERTIFICATE) {
                    return true
                }
                // Check If at least 1 task has already failed
                for (value in concurrentTasks.values) {
                    if (value.compareTo(TaskExecutionState.FAILURE) == 0) {
                        return false
                    }
                }
                return true
            }
        }
    }
}