package com.liveperson.messaging.model;


import com.liveperson.infra.Command;
import com.liveperson.infra.ForegroundService;
import com.liveperson.infra.ICallback;
import com.liveperson.infra.LocalBroadcastReceiver;
import com.liveperson.infra.log.LPLog;
import com.liveperson.messaging.controller.AccountsController;

import static com.liveperson.infra.errors.ErrorCode.ERR_000000D3;

/**
 * Created by Ilya Gazman on 11/26/2015.
 * <p/>
 * A command to perform a synchronized connection and provide a callback
 * TODO: See If we can replace this class by using subscribeToAuthStateChanges method of ConsumerManager
 */
public class SynchronizedAuthenticationCompletedCallback implements Command {
    private static final String TAG = "SynchronizedAuthenticationCompletedCallback";
    private static final String ERROR_UNABLE_TO_MAKE_REQUEST = "Unable to make request.";
    private final AccountsController mAccountController;

    private String mBrandId;
    private boolean handled = false;
    private LocalBroadcastReceiver mLocalBroadcastReceiver;
    private ICallback<Void, Exception> mCallback;

    /**
     * Synchronously listening to when connection is accomplished
     *
     * @param accountsController
     * @param callback
     */
    public SynchronizedAuthenticationCompletedCallback(AccountsController accountsController, String brandId, ICallback<Void, Exception> callback) {
        mAccountController = accountsController;
        mBrandId = brandId;
        mCallback = callback;
    }

    @Override
    public void execute() {
        executeWithReturnValue();
    }

    public boolean executeWithReturnValue() {
        if (isValidAuthentication()) {
            handleConnection();
            return true;
        } else {
            registerToConnectionStateChanges();
            validateStatusDidNotChangedDuringRegistration();
            return false;
        }
    }

    /**
     * We have a valid authentication if consumer is authenticated as well as conversation
     * screen is in foreground. Foreground check is because we want to talk to IDP in case
     * host app switched to different user while SDK was in background.
     * @return boolean
     */
    private boolean isValidAuthentication() {
        AmsAccount account = mAccountController.getAccount(mBrandId);
        return account != null && account.hasFinishedAuthenticating()
                && ForegroundService.getInstance().isBrandForeground(mBrandId);
    }

    private void registerToConnectionStateChanges() {
        try {
            if (mLocalBroadcastReceiver == null) {
                mLocalBroadcastReceiver = new LocalBroadcastReceiver.Builder()
                        .addAction(AmsAccount.BROADCAST_KEY_AUTH_COMPLETED_ACTION)
                        .addAction(AmsConnection.BROADCAST_CONNECTING_TO_SERVER_ERROR)
                        .addAction(AmsConnection.BROADCAST_AMS_TOKEN_EXPIRED)
                        .build((context, intent) -> {
                            if (AmsAccount.BROADCAST_KEY_AUTH_COMPLETED_ACTION.equals(intent.getAction())) {
                                handleConnection();
                            } else if (AmsConnection.BROADCAST_CONNECTING_TO_SERVER_ERROR.equals(intent.getAction())) {
                                handleError("Error: Failed to connect to the server");
                            } else if (AmsConnection.BROADCAST_AMS_TOKEN_EXPIRED.equals(intent.getAction())) {
                                handleError("Error: Token expired, refresh the token and try again");
                            }
                        });
            }
            mLocalBroadcastReceiver.register();
        } catch (Exception error) {
            LPLog.INSTANCE.e(TAG, ERR_000000D3, "registerToConnectionStateChanges: Failed to register", error);
        }
    }

    private synchronized void validateStatusDidNotChangedDuringRegistration() {
        if (!handled) {
            if (isValidAuthentication()) {
                handleConnection();
            }
        }
    }

    private synchronized void handleConnection() {
        if (handled) {
            return;
        }
        if (mLocalBroadcastReceiver != null) {
            mLocalBroadcastReceiver.unregister();
        }
        handled = true;
        mCallback.onSuccess(null);
    }
    private synchronized void handleError(String error) {
        if (handled) {
            return;
        }
        if (mLocalBroadcastReceiver != null) {
            mLocalBroadcastReceiver.unregister();
        }
        handled = true;
        mCallback.onError(new Exception(ERROR_UNABLE_TO_MAKE_REQUEST + " " + error));
    }
}