package com.liveperson.api;

import com.liveperson.api.response.types.CloseReason;
import com.liveperson.api.sdk.LPConversationData;
import com.liveperson.api.sdk.PermissionType;
import com.liveperson.messaging.LpError;
import com.liveperson.infra.utils.LPDeprecated;
import com.liveperson.messaging.TaskType;
import com.liveperson.messaging.model.AgentData;

/**
 * Created by shiranr on 09/09/2015.
 */
@SuppressWarnings("DeprecatedIsStillUsed") // We already know. They're also used by customers.
public interface LivePersonCallback {

	// Conversation Events

	/**
	 * Called whenever either the consumer or the agent starts a new conversation.
	 *
	 * @param convData Contains the ID of the new Conversation, and a null CloseReason.
	 */
	void onConversationStarted(LPConversationData convData);

	/**
	 * Called whenever either the consumer or the agent starts a new conversation.
	 *
	 * @deprecated since April 26, 2017
	 */
	@Deprecated @LPDeprecated(since = 1493164800) // April 26, 2017
	void onConversationStarted();

	/**
	 * Called when the current conversation is marked as resolved by either the consumer or
	 * the agent.
	 *
	 * @param convData Contains the ID of the resolved Conversation, and a CloseReason
	 *                 describing why the conversation ended.
	 */
	void onConversationResolved(LPConversationData convData);

	/**
	 * Called when the current conversation is marked as resolved by either the consumer or
	 * the agent.
	 *
	 * @deprecated since February 13, 2017
	 */
	@Deprecated @LPDeprecated(since = 1486944000) // February 13, 2017
	void onConversationResolved();

	/**
	 * Called when the current conversation is marked as resolved by either the consumer or
	 * the agent.
	 *
	 * @deprecated since February 13, 2017
	 *
	 * @param reason The reason why the current conversation was closed.
	 */
	@Deprecated @LPDeprecated(since = 1493164800) // April 26, 2017
	void onConversationResolved(CloseReason reason);

	/**
	 * Called when the Conversation Fragment is closed (only called when using Fragment
	 * Mode).
	 */
	void onConversationFragmentClosed();

	/**
	 * Called when the current conversation gets marked as urgent by either the consumer or
	 * the agent.
	 */
	void onConversationMarkedAsUrgent();

	/**
	 * Called when the current conversation gets marked as normal by either the consumer or
	 * the agent.
	 */
	void onConversationMarkedAsNormal();


	// Agent Events

	/**
	 * Called when the assigned agent is typing a message. When there are 2 seconds of idle
	 * time, the callback returns false to indicate that the agent stopped typing.
	 *
	 * @param isTyping <b>true</b> if the Agent is currently typing, <b>false</b> if the
	 *                 Agent's keyboard is idle.
	 */
	void onAgentTyping(boolean isTyping);

	/**
	 * Called when the assigned agent of the current conversation has changed, or their
	 * details are updated. When no agent is associated with the conversation, the callback
	 * passes a null value. For example, this happens when an agent returns the consumer to
	 * the request queue.
	 *
	 * @param agentData Contains the first and last names, avatar url, and employee ID of
	 *                  the new (or updated) currently-assigned Agent.
	 */
	void onAgentDetailsChanged(AgentData agentData);

	/**
	 * Called when there is a change in agent availability.
	 *
	 * @param isOfflineHoursOn <b>true</b> if the Agent is in Offline Hours mode;
	 *                         <b>false</b> if the Agent returns to an online state.
	 */
	void onOfflineHoursChanges(boolean isOfflineHoursOn);


	// User Action Events

	/**
	 * Called if the user denied a necessary system permission for the action they tried to
	 * perform. For example, when the user clicks on the camera/gallery button to add an
	 * image, the permission system dialog was displayed, and the user denied permission.
	 *
	 * @param permissionType The type of permission required for the action the user was
	 *                       attempting to perform.
	 *
	 * @param doNotShowAgainMarked <b>true</b> if the user directed the Permissions system
	 *                             to never ask again for this permissionType; <b>false</b>
	 *                             otherwise.
	 */
	void onUserDeniedPermission(PermissionType permissionType, boolean doNotShowAgainMarked);

	/**
	 * Called before requiring a permission that the user has not yet accepted. For example,
	 * when the user clicks the camera or gallery buttons to add an image, this callback
	 * gets called just before the permission dialog is displayed. If the user already
	 * allowed permission, this callback does not get called.
	 *
	 * @param permissionType The type of permission required for the action the user is
	 *                       attempting to perform.
	 */
	void onUserActionOnPreventedPermission(PermissionType permissionType);

	/**
	 * Called when the user taps on the agent avatar.
	 * <br/><br/>
	 * The icon is available next to the agent message bubble or on the top of the toolbar
	 * (if using activity mode).
	 *
	 * @param agentData Contains the first and last names, avatar url, and employee ID of
	 *                  the new (or updated) currently-assigned Agent.
	 */
	void onAgentAvatarTapped(AgentData agentData);

	/**
	 * Called when a structured content control with Link action gets clicked.
	 * <br/><br/>
	 * !! This callback only gets called if the `structured_content_link_as_callback`
	 * or the `lp_message_link_as_callback` parameter in the branding.xml
	 * is set to <b>true</b>. !!
	 *
	 * @param uri A link to the content the user was sent to when they tapped on a piece of
	 *            Structured Content.
	 */
	void onStructuredContentLinkClicked(String uri);

	/**
	 * Called when user tries to follow a link in text/file message sent by brand/agent/bot/system or in
	 * structured content message.
	 * <br/><br/>
	 * !! This callback only gets called if the `lp_message_link_as_callback`
	 * parameter in the branding.xml is set to <b>true</b>. !!
	 *
	 * @param uri A link tapped by consumer in text/file message sent by brand/agent/bot/system or in
	 *            structured content message.
	 */
	void onLinkClicked(String uri);

	/**
	 * Called when user taps on one of the quick reply options of welcome message.
	 * @param title Tapped Quick Reply option title
	 */
	void onWelcomeMessageOptionTapped(String title);


	// Survey Events

	/**
	 * Called when the feedback screen launches.
	 */
	void onCsatLaunched();

	/**
	 * Called when the feedback screen gets dismissed with any result, positive or negative.
	 * (The user taps the <b>Submit</b>, <b>Skip</b>, or <b>Back</b> buttons.)
	 */
	void onCsatDismissed();

	/**
	 * Called when the user taps the <b>Submit</b> button on the feedback screen. The
	 * onCsatDismissed callback is called as well.
	 *
	 * @deprecated since August 19, 2019
	 *
	 * @param conversationId ID of the conversation the survey is rating.
	 */
	@Deprecated @LPDeprecated(since = 1566261196) // August 19, 2019
	void onCsatSubmitted(String conversationId);

	/**
	 * Called when the user taps the <b>Submit</b> button on the feedback screen. The
	 * onCsatDismissed callback is called as well.
	 *
	 * @deprecated since August 19, 2019
	 *
	 * @param conversationId ID of the conversation the survey is rating.
	 *
	 * @param starRating How many stars (out of 5) the user rated this conversation.
	 */
	void onCsatSubmitted(String conversationId,int starRating);

	/**
	 * Called when the user taps the <b>Skip</b> or <b>Back</b> buttons. The onCsatDismissed
	 * callback is called as well.
	 */
	void onCsatSkipped();


	// Authentication Events

	/**
	 * Called if the token used in the session has expired and is no longer valid. The host
	 * app needs to reconnect with a new authentication key.
	 */
	void onTokenExpired();

	/**
	 * Called if the temporary user authentication used specifically in an Unauthenticated
	 * type flow expires.
	 */
	void onUnauthenticatedUserExpired();


	// Connection Events

	/**
	 * Called when the connection to the conversation server has been established or
	 * disconnected.
	 *
	 * @param isConnected <b>true</b> for connected, <b>false</b> for disconnected
	 */
	void onConnectionChanged(boolean isConnected);

	// Error Events

	/**
	 * Called to indicate that an internal SDK error occurred.
	 *
	 * @param type    The category or type of error.
	 * @param message A detailed message on the error.
	 * @deprecated Since September 28, 2020
	 */
	@Deprecated
	@LPDeprecated(since = 1601280000)
	//TODO: remove it when upgrade to v6.0.0, we should separate TaskType with Errors.
	void onError(TaskType type, String message);

	/**
	 * Called to indicate that an internal SDK error occurred.
	 *
	 * @param lpError The error type
	 * @param message   The error message.
	 */
	void onError(LpError lpError, String message);


	/**
	 * No-Op-for-all-methods implementation of LivePersonCallback, ideal for extending.
	 */
	class NullLivePersonCallback implements LivePersonCallback {
		// Conversation Events
		@Override
		public void onConversationStarted(LPConversationData convData) {}

		@Override @Deprecated @LPDeprecated(since = 1493164800) // April 26, 2017
		public void onConversationStarted() {}

		@Override
		public void onConversationResolved(LPConversationData convData) {}

		@Override @Deprecated @LPDeprecated(since = 1486944000) // February 13, 2017
		public void onConversationResolved() {}

		@Override @Deprecated @LPDeprecated(since = 1493164800) // April 26, 2017
		public void onConversationResolved(CloseReason reason) {}

		@Override
		public void onConversationFragmentClosed() {}

		@Override
		public void onConversationMarkedAsUrgent() {}

		@Override
		public void onConversationMarkedAsNormal() {}


		// Agent Events
		@Override
		public void onAgentTyping(boolean isTyping) {}

		@Override
		public void onAgentDetailsChanged(AgentData agentData) {}

		@Override
		public void onOfflineHoursChanges(boolean isOfflineHoursOn) {}


		// User Action Events
		@Override
		public void onUserDeniedPermission(PermissionType permissionType, boolean doNotShowAgainMarked) {}

		@Override
		public void onUserActionOnPreventedPermission(PermissionType permissionType) {}

		@Override
		public void onAgentAvatarTapped(AgentData agentData) {}

		@Override
		public void onStructuredContentLinkClicked(String uri) {}

		@Override
		public void onWelcomeMessageOptionTapped(String title) {}

		// Survey Events
		@Override
		public void onCsatLaunched() {}

		@Override
		public void onCsatDismissed() {}

		@Override @Deprecated @LPDeprecated(since = 1566261196) // August 19, 2019
		public void onCsatSubmitted(String conversationId) {}

		@Override
		public void onCsatSubmitted(String conversationId, int starRating) {}

		@Override
		public void onCsatSkipped() {}


		// Authentication Events
		@Override
		public void onTokenExpired() {}

		@Override
		public void onUnauthenticatedUserExpired() {}


		// Connection Events
		@Override
		public void onConnectionChanged(boolean isConnected) {}


		// Error Events
		@Override
		public void onError(TaskType type, String message) {}

		@Override
		public void onError(LpError lpError, String message) {}

		@Override
		public void onLinkClicked(String uri) {}
	}
}
