@file:JvmName("InitialMessageUtils")
package com.liveperson.messaging.utils

import com.liveperson.api.response.types.ConversationState
import com.liveperson.infra.LPConversationsHistoryStateToDisplay
import com.liveperson.infra.configuration.Configuration
import com.liveperson.infra.log.LPLog
import com.liveperson.infra.messaging.R
import com.liveperson.infra.model.LPWelcomeMessage
import com.liveperson.infra.model.types.NotificationType
import com.liveperson.infra.preferences.PushMessagePreferences.getCachedPushMessage
import com.liveperson.infra.preferences.PushMessagePreferences.getClickedNotificationId
import com.liveperson.infra.preferences.PushMessagePreferences.getLatestNotificationIdForBrand
import com.liveperson.infra.preferences.PushMessagePreferences.isPushNotificationClicked
import com.liveperson.messaging.Messaging
import com.liveperson.messaging.model.ConversationData
import com.liveperson.messaging.model.ConversationUtils

private const val TAG = "InitialMessageUtils"

@JvmName("showInitialMessageIfNeeded")
internal fun Messaging.showInitialMessageIfNeeded(
    umsConversations: List<ConversationData>,
    incaConversations: List<ConversationData>?
) {

    val hasOpenConversation = umsConversations.any { it.state == ConversationState.OPEN }
    if (hasOpenConversation) {
        return LPLog.d(TAG, "Consumer has an active conversation. Skipping adding proactive/welcome message")
    } else {
        val nonSyncedConversation = amsConversations?.getActiveConversation(activeBrandId)
            ?.executeSynchronously()
            ?.takeIf { it.state == ConversationState.OPEN }
        val isClosedInUMS = umsConversations.any {
            it.conversationId == nonSyncedConversation?.conversationId && it.state != ConversationState.OPEN
        }
        val isClosedInInca = umsConversations.any {
            it.conversationId == nonSyncedConversation?.conversationId && it.state != ConversationState.OPEN
        }
        if (nonSyncedConversation != null && !isClosedInUMS && !isClosedInInca) {
            return LPLog.d(TAG, "Consumer has an active conversation locally. Skipping adding proactive/welcome message")
        }
    }
    if (conversationViewParams.historyConversationsStateToDisplay == LPConversationsHistoryStateToDisplay.CLOSE) {
        return LPLog.d(TAG, "Conversation params filtering is set to closed conversation. Skipping adding proactive/welcome message")
    }

    val conversationsRepository = amsConversations ?: return LPLog.d(TAG, "Ams conversation is null")
    val welcomeMessage = welcomeMessageManager?.getWelcomeMessage(activeBrandId) ?: return

    val pushMessageId = retrieveProactiveMessageIdToShow()
    val hasConversation = umsConversations.isNotEmpty() || !incaConversations.isNullOrEmpty()

    val conversationUtils = ConversationUtils(this)
    when {
        pushMessageId != null -> {
            LPLog.d(TAG, "Showing proactive message")
            conversationUtils.displayOutboundCampaignMessage(activeBrandId, pushMessageId)
        }
        welcomeMessage.isForEveryConversation -> {
            LPLog.d(TAG, "Showing welcome message for each conversation")
            withHistoryClearenceFlag {
                conversationUtils.addWelcomeMessageRequest(activeBrandId)
            }
        }
        !hasConversation && !conversationsRepository.isHistoryExistForConsumer(activeBrandId) -> {
            LPLog.d(TAG, "Showing welcome message for the first conversation")
            withHistoryClearenceFlag {
                conversationUtils.addWelcomeMessageRequest(activeBrandId)
            }
        }
        else -> {
            LPLog.d(TAG, "Skipping showing message")
        }
    }
}


private inline fun Messaging.withHistoryClearenceFlag(block: () -> Unit) {
    when {
        !getClearHistoryFlag(activeBrandId) -> {
            LPLog.d(TAG, "History is not cleared. Showing welcome message")
            block()
        }
        !Configuration.getBoolean(R.bool.lp_hide_welcome_message_on_clear_history) -> {
            LPLog.d(TAG, "lp_hide_welcome_message_on_clear_history is not enabled. Showing welcome message")
            block()
        }
        else -> {
            LPLog.d(TAG, "History is cleare and lp_hide_welcome_message_on_clear_history. Skipping welcome message")
        }
    }
}

private val LPWelcomeMessage.isForEveryConversation: Boolean
    get() = messageFrequency == LPWelcomeMessage.MessageFrequency.EVERY_CONVERSATION

private fun Messaging.getProactiveMessageId(): String? {
    return when {
        isPushNotificationClicked() -> {
            getClickedNotificationId()
        }
        Configuration.getBoolean(R.bool.show_outbound_in_app_message) -> {
            getLatestNotificationIdForBrand(activeBrandId)
        }
        else -> {
            null
        }
    }
}

private fun Messaging.retrieveProactiveMessageIdToShow(): String? {
    val pushMessageId = getProactiveMessageId()
    if (pushMessageId.isNullOrBlank()) {
        LPLog.d(TAG, "No clicked proactive message found.")
        return null
    }
    val message = getCachedPushMessage(pushMessageId, activeBrandId)
    if (message == null) {
        LPLog.d(TAG, "No push message found for id $pushMessageId")
        return null
    }
    if (message.isExpired) {
        LPLog.d(TAG, "Requested proactive for id $pushMessageId is expired")
        return null
    }
    return when (message.notificationType) {
        null -> {
            LPLog.d(TAG, "Can't define message type for required push message id: $pushMessageId. Skipping showing proactive message")
            null
        }
        NotificationType.REGULAR -> {
            LPLog.d(TAG, "Clicked push notification is a regular one. Skipping showing proactive message")
            null
        }
        else -> {
            pushMessageId
        }
    }
}