package com.liveperson.messaging.wm

import android.annotation.SuppressLint
import android.content.BroadcastReceiver
import android.content.Context
import android.content.Intent
import android.content.IntentFilter
import android.os.Build

/**
 * Broadcast receiver used to handle events from
 * Dynamic Welcome Message API to represent actual
 * welcome message for subscribed brand.
 */
class WelcomeMessageBroadcastReceiver private constructor(
    private val listener: WelcomeMessageEventListener
) : BroadcastReceiver() {

    companion object {

        const val ACTION_WELCOME_MESSAGE_CHANGED = "action.welcome.message.changed"
        const val ACTION_WELCOME_MESSAGE_RESPONSE = "welcome.message.response"

        private const val PERMISSION_WELCOME_MESSAGE_ACTIONS = "com.liveperson.permission.WELCOME_MESSAGE_ACTIONS"

        private const val KEY_CHANGED_VALUE_NAME = "value"

        /**
         * Extension function used to send event and optional key
         * to welcome message broadcast receivers.
         *
         * @param action one of possible action produced
         * by Dynamic Welcome Message components:
         * - @see ACTION_WELCOME_MESSAGE_RESPONSE produced when welcome message request completed.
         * - @see ACTION_WELCOME_MESSAGE_CHANGED produced when welcome message content was changed.
         * @param key optional parameter to determine what exactly was changed
         * in dynamic welcome message structure.
         */
        internal fun Context.sendWelcomeMessageEvent(action: String, key: String? = null) {
            val intent = Intent(action).setPackage(packageName)
            if (key != null) {
                intent.putExtra(KEY_CHANGED_VALUE_NAME, key)
            }
            sendBroadcast(intent)
        }

        private fun intentFilterOf(brandId: String): IntentFilter {
            return IntentFilter(ACTION_WELCOME_MESSAGE_RESPONSE + brandId)
                .apply { addAction(ACTION_WELCOME_MESSAGE_CHANGED) }
        }

        @JvmStatic
        fun createReceiver(listener: WelcomeMessageEventListener): BroadcastReceiver {
            return WelcomeMessageBroadcastReceiver(listener)
        }

        @SuppressLint("UnspecifiedRegisterReceiverFlag")
        @JvmStatic
        fun BroadcastReceiver.registerWelcomeMessageActions(context: Context, brandId: String) {
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.TIRAMISU) {
                context.registerReceiver(this, intentFilterOf(brandId), PERMISSION_WELCOME_MESSAGE_ACTIONS, null, Context.RECEIVER_NOT_EXPORTED)
            } else {
                context.registerReceiver(this, intentFilterOf(brandId), PERMISSION_WELCOME_MESSAGE_ACTIONS, null)
            }
        }
    }

    override fun onReceive(context: Context?, intent: Intent?) {
        intent?.action?.let { listener.onEventReceived(it) }
    }
}