package com.liveperson.messaging.commands.pusher;

import android.text.TextUtils;

import androidx.annotation.Nullable;

import com.liveperson.infra.Command;
import com.liveperson.infra.ICallback;
import com.liveperson.infra.InternetConnectionService;
import com.liveperson.infra.errors.ErrorCode;
import com.liveperson.infra.log.LPLog;
import com.liveperson.infra.managers.PreferenceManager;
import com.liveperson.infra.network.http.requests.UnregisterPushRequest;
import com.liveperson.messaging.controller.AccountsController;
import com.liveperson.messaging.controller.connection.ConnectionParamsCache;
import com.liveperson.messaging.model.AmsUsers;
import com.liveperson.messaging.model.SynchronizedInternetConnectionCallback;

import java.util.List;

import static com.liveperson.infra.managers.PreferenceManager.APP_ID_PREFERENCE_KEY;
import static com.liveperson.infra.managers.PreferenceManager.CONSUMER_UNREGISTER_PUSHER_PREFERENCE_KEY;
import static com.liveperson.infra.managers.PreferenceManager.IS_PUSHER_REGISTERED_PREFERENCE_KEY;

/**
 * Command to unregister from push notifications for a given brandId.
 * <p>
 * Created by nirni on 1/27/16.
 */
public class UnregisterPusherCommand implements Command {

    private static final String TAG = "UnregisterPusherCommand";

    private static final String PUSHER_UNREGISTER_URL = "https://%s/api/account/%s/device/unregister?v=2.0";
    private static final String UN_AUTH_PUSHER_UNREGISTER_URL = "https://%s/api/account/%s/device/unregister";
    private static final int MAX_RETRY_NUMBER = 3;
    private final AccountsController mAccountsController;
    private final AmsUsers mAmsUsers;

    private String mBrandId;
    private String mAppId;
    private String pusherURL;
    private String consumerId;
    @Nullable
    private ICallback<Void, Exception> listener;
    private boolean isFullLogout;
    private int logoutRetry;

    private ICallback<Void, Exception> requestCallback = new ICallback<Void, Exception>() {
        @Override
        public void onSuccess(Void value) {
            PreferenceManager.getInstance().setBooleanValue(IS_PUSHER_REGISTERED_PREFERENCE_KEY, mBrandId, false);
            PreferenceManager.getInstance().remove(CONSUMER_UNREGISTER_PUSHER_PREFERENCE_KEY, mBrandId);
            if (listener != null) {
                listener.onSuccess(value);
            }
        }

        @Override
        public void onError(Exception exception) {
            if (isFullLogout && logoutRetry < MAX_RETRY_NUMBER) {
                logoutRetry++;
                LPLog.INSTANCE.w(TAG, "Unregister pusher retry #: " + logoutRetry, exception);
                runUnregisterPush();
            } else {
                LPLog.INSTANCE.e(TAG, ErrorCode.ERR_00000148, "Failed to unregister pusher", exception);
                if (listener != null) {
                    listener.onError(exception);
                }
            }
        }
    };

    public UnregisterPusherCommand(AccountsController accountsController, AmsUsers amsUsers, String brandId, String consumerId) {
        this(accountsController, amsUsers, brandId, "", null, false);
        this.consumerId = consumerId;
    }

    public UnregisterPusherCommand(AccountsController accountsController, AmsUsers amsUsers, String brandId, String appId, @Nullable ICallback<Void, Exception> listener, boolean isFullLogout) {
        mAccountsController = accountsController;
        mAmsUsers = amsUsers;
        mBrandId = brandId;
        mAppId = appId;
        this.listener = listener;
        this.isFullLogout = isFullLogout;
        logoutRetry = 0;
    }

    @Override
    public void execute() {

        if (!PreferenceManager.getInstance().getBooleanValue(IS_PUSHER_REGISTERED_PREFERENCE_KEY, mBrandId, false)){
            notifySuccess();
            return;
        }

        LPLog.INSTANCE.i(TAG, "execute unregister for brandId " + mBrandId);

        // Create pusher unregister URL
        String pusherDomain = mAccountsController.getServiceUrl(mBrandId, ConnectionParamsCache.CSDS_PUSHER_DOMAIN_KEY);
        if (TextUtils.isEmpty(pusherDomain)) {
            pusherDomain = PreferenceManager.getInstance().getStringValue(ConnectionParamsCache.CSDS_PUSHER_DOMAIN_KEY, mBrandId, null);
            if (TextUtils.isEmpty(pusherDomain)) {
                LPLog.INSTANCE.w(TAG, "pusherDomain does not exists. Quit unregister push");
                notifySuccess();
                return;
            }
        }

        if (TextUtils.isEmpty(mAppId) && !TextUtils.isEmpty(consumerId)) {
            mAppId = PreferenceManager.getInstance().getStringValue(APP_ID_PREFERENCE_KEY, mBrandId, "");
            PreferenceManager.getInstance().setStringValue(CONSUMER_UNREGISTER_PUSHER_PREFERENCE_KEY, mBrandId, consumerId);
        } else {
            PreferenceManager.getInstance().setStringValue(APP_ID_PREFERENCE_KEY, mBrandId, mAppId);
        }

        // Use older version of unregister api to remove UnAuth user from pusher database on sign out.
        if (isFullLogout && (mAccountsController.getAccount(mBrandId) != null && !mAccountsController.getAccount(mBrandId).isAuthenticated())) {
            pusherURL = String.format(UN_AUTH_PUSHER_UNREGISTER_URL, pusherDomain, mBrandId);
        } else {
            pusherURL = String.format(PUSHER_UNREGISTER_URL, pusherDomain, mBrandId);
        }

        if (TextUtils.isEmpty(consumerId)) {
            // Get consumerId from memory
            consumerId = mAmsUsers.getConsumerId(mBrandId);
        }

        // If consumerId is not available in memory try to get from DB
        if (TextUtils.isEmpty(consumerId)) {
			LPLog.INSTANCE.d(TAG, "execute: consumerId is not available. Trying to get from DB...");
            mAmsUsers.getConsumerByBrandIDFromDB(mBrandId).setPostQueryOnBackground(dbConsumerId -> {
                if (!TextUtils.isEmpty(dbConsumerId)) {
                    consumerId = dbConsumerId;
                    runUnregisterPush();
					LPLog.INSTANCE.d(TAG, "onResult: got  consumerId from DB (" + dbConsumerId + "). Unregister push with it...");
                } else {
	                LPLog.INSTANCE.w(TAG, "onResult: Cannot get user profile from DB. Quit unregister push");
                    if (listener != null && isFullLogout) {
                        listener.onSuccess(null); //Don't block logout.
                    }
                }

            }).execute();
        } else {
            runUnregisterPush();
        }
    }

    private void notifySuccess() {
        if (listener != null) {
            listener.onSuccess(null);
        }
    }

    private void runUnregisterPush() {
        final List<String> certificates = mAccountsController.getCertificatePinningKeys(mBrandId);
        if (isFullLogout) { // Run immediately when logout.
            LPLog.INSTANCE.d(TAG, "run: Unregister push immediately");
            if (InternetConnectionService.isNetworkAvailable()) {
                // If available execute immediately
                new UnregisterPushRequest(pusherURL, consumerId, mAppId, certificates).setCallback(requestCallback).execute();
            } else {
                if (listener != null) {
                    listener.onError(new Exception("No network available"));
                }
            }
        } else {
            new SynchronizedInternetConnectionCallback(() -> {
	            LPLog.INSTANCE.d(TAG, "run: Unregister push for consumerId: " + LPLog.INSTANCE.mask(consumerId));
                new UnregisterPushRequest(pusherURL, consumerId, mAppId, certificates).setCallback(requestCallback).execute();
            }).execute();
        }
    }
}
