package com.liveperson.messaging.commands.tasks;

import androidx.annotation.NonNull;
import android.text.TextUtils;

import com.liveperson.infra.ICallback;
import com.liveperson.infra.Infra;
import com.liveperson.infra.callbacks.AuthCallBack;
import com.liveperson.infra.model.Consumer;
import com.liveperson.infra.model.errors.AuthError;
import com.liveperson.infra.network.http.requests.AuthRequest;
import com.liveperson.infra.auth.LPAuthenticationParams;
import com.liveperson.infra.auth.LPAuthenticationType;
import com.liveperson.infra.log.LPLog;
import com.liveperson.infra.managers.PreferenceManager;
import com.liveperson.messaging.MessagingFactory;
import com.liveperson.messaging.TaskType;
import com.liveperson.messaging.utils.TokenUtils;
import com.liveperson.messaging.controller.AccountsController;
import com.liveperson.messaging.controller.connection.ConnectionParamsCache;
import com.liveperson.messaging.model.AmsConnectionAnalytics;
import com.liveperson.messaging.model.AmsUsers;
import com.liveperson.messaging.network.http.IdpRequest;
import com.liveperson.messaging.network.http.UnAuthRequest;

import org.jetbrains.annotations.NotNull;

import java.util.List;

import javax.net.ssl.SSLPeerUnverifiedException;

/**
 * Created by nirni on 1/4/16.
 * <p/>
 * A task to get the token from the IDP service
 */
public class IdpTask extends BaseAmsAccountConnectionTask {

    private static final String TAG = "IdpTask";
    private static final String KEY_CONSUMER_ID = "KEY_CONSUMER_ID";
    private final AmsUsers mAmsUsers;
    private AccountsController mAccountsController;

    private String mHostVersion;

    public IdpTask(AccountsController accountsController, AmsUsers amsUsers, String hostVersion){
        mHostVersion = hostVersion;
        mAccountsController = accountsController;
        mAmsUsers = amsUsers;
    }

    @Override
    public void execute() {
        LPLog.INSTANCE.d(TAG, "Running IDP task...");

        AmsConnectionAnalytics.idpTaskStart();

        // Try to get a locally existing token
        String token = mAccountsController.getToken(mBrandId);
        final List<String> certificates = mAccountsController.getCertificatePinningKeys(mBrandId);
        boolean tokenExpired = mAccountsController.isTokenExpired(mBrandId);
        LPAuthenticationParams auth = mAccountsController.getLPAuthenticationParams(mBrandId);

        //if JWT exists and valid (not expired) // could be only in un-auth flow, cause otherwise we would stop connecting flow in ConnectionStateMachine.
        if (!TextUtils.isEmpty(token) && !tokenExpired) {

            LPLog.INSTANCE.d(TAG, "execute - token exists and valid: " + LPLog.INSTANCE.mask(token));
            AmsConnectionAnalytics.idpTaskEnd();
            mCallback.onTaskSuccess();

        } else { // Token does not exist. Send a request to IDP service

            String idpDomain = getIdpDomain();

            if (auth.getAuthType() == LPAuthenticationType.UN_AUTH) {

                if (tokenExpired) {
                    //clear LP JWT, we are going to generate new one!
                    //we won't clean Non Auth code, cause we need to for refreshing the user.
                    mAccountsController.setToken(mBrandId, null);
                }

                String connectorId = mAccountsController.getAccount(mBrandId).getConnectorId();//mCampaignInfo == null? "" : mCampaignInfo.getConnectorId();

                new UnAuthRequest(mAccountsController.getAccount(mBrandId), idpDomain, mBrandId, auth, mHostVersion, getIdpCallback(), certificates, connectorId).execute();
            } else {
                //Auth and SignUp:
                new IdpRequest(mAccountsController.getAccount(mBrandId), idpDomain, mBrandId, auth, mHostVersion, getIdpCallback(), certificates).execute();
            }
            /*---------------------------------------------------------------------------------------------
            new AuthRequest(Infra.instance.getApplicationContext(), mBrandId, idpDomain, auth, mHostVersion, certificates, mAccountsController.getAccount(mBrandId).getConnectorId(), new AuthCallBack() {
                @Override
                public void onAuthSuccess(@NonNull Consumer consumer) {
                    LPLog.INSTANCE.d(TAG, "onAuthSuccess: Consumer: " + LPLog.INSTANCE.mask(consumer));
                }

                @Override
                public void onAuthFailed(@NotNull AuthError error) {
                    LPLog.INSTANCE.d(TAG, "onAuthFailed: error: " + error.toString());
                }
            }).authenticate();
            ---------------------------------------------------------------------------------------------*/
        }
    }

    @NonNull
    private IDPExceptionICallback getIdpCallback() {
        return new IDPExceptionICallback();
    }

    /**
     * Get the IDP domain from CSDS. If the "IDP" does not exist in the CSDS domains, we get the AsyncMessagingEnt
     * and replace the "msg" to "idp
     *
     * @return
     */
    private String getIdpDomain() {

        String idpDomain;

		// Get the IDP domain from CSDS
		idpDomain = mAccountsController.getServiceUrl(mBrandId, ConnectionParamsCache.CSDS_IDP_DOMAIN_KEY);

		// The IDP domain is not always in CSDS. If we got an empty string we go get the AccountsController.AMS_CSDS_DOMAIN_KEY instead
		if (TextUtils.isEmpty(idpDomain)) {
			String asyncMessagingEnt = mAccountsController.getServiceUrl(mBrandId, ConnectionParamsCache.CSDS_UMS_DOMAIN_KEY);

            // Replace the "msg" to "idp"
            idpDomain = asyncMessagingEnt.replaceFirst("msg", "idp");
        }

        return idpDomain;
    }


    @Override
    public String getName() {
        return TAG;
    }

    public class IDPExceptionICallback implements ICallback<String, Exception> {

        @Override
        public void onSuccess(String token) {
            LPLog.INSTANCE.d(TAG, "onSuccess: got token ");
            mAccountsController.setToken(mBrandId, token);
            mAccountsController.getAccount(mBrandId).setOriginalConsumerId(TokenUtils.getOriginalConsumerIdFromJWT(token));
            //set consumer user id
            mAmsUsers.updateConsumerId(mBrandId, TokenUtils.getConsumerUserId(token));
            AmsConnectionAnalytics.idpTaskEnd();
            mAccountsController.getAccount(mBrandId).sendAuthenticationCompletedStatus();
            mCallback.onTaskSuccess();
        }

        /**
         * Method for testing user switch scenario.
         * TODO: replace with consumer manager.
         *
         * @param token token from IDP response
         */
        private void switchUser(String token) {
            String consumerId = TokenUtils.getConsumerUserId(token);
            boolean shouldRegisterPusher = false;
            if (PreferenceManager.getInstance().contains(KEY_CONSUMER_ID, mBrandId)
                    && !PreferenceManager.getInstance().getStringValue(KEY_CONSUMER_ID, mBrandId, "").equals(consumerId)) {
                LPLog.INSTANCE.i(TAG, "New user login. Performing lite logout.");
                String oldConsumerId = PreferenceManager.getInstance().getStringValue(KEY_CONSUMER_ID, mBrandId, "");
                shouldRegisterPusher = true;
                MessagingFactory.getInstance().getController().liteLogout(mBrandId, oldConsumerId, consumerId);
            } else {
                LPLog.INSTANCE.i(TAG, "Old user login.");
            }
            PreferenceManager.getInstance().setStringValue(KEY_CONSUMER_ID, mBrandId, consumerId);
            mAccountsController.setToken(mBrandId, token);
            mAccountsController.getAccount(mBrandId).setOriginalConsumerId(TokenUtils.getOriginalConsumerIdFromJWT(token));
            mAmsUsers.updateConsumerId(mBrandId, consumerId);
            AmsConnectionAnalytics.idpTaskEnd();
            mAccountsController.getAccount(mBrandId).sendAuthenticationCompletedStatus();
            mCallback.onTaskSuccess();
            if (shouldRegisterPusher) {
                //register pusher after mAccountsController.setToken(mBrandId, token);
                MessagingFactory.getInstance().getController().registerPusherOnLiteLogout(mBrandId, mAccountsController.getLPAuthenticationParams(mBrandId));
            }
        }

        @Override
        public void onError(Exception exception) {
            LPLog.INSTANCE.d(TAG, "onError: failed to connect to idp!", exception);
            if(exception instanceof SSLPeerUnverifiedException){
                mCallback.onTaskError(TaskType.INVALID_CERTIFICATE, exception);
            }else {
                mCallback.onTaskError(TaskType.IDP, exception);
            }
        }

        public void onError(TaskType taskType, Exception exception) {
            LPLog.INSTANCE.d(TAG, "onError: failed to connect to idp! taskType = " + taskType + ", error: ", exception);
            mCallback.onTaskError(taskType, exception);
        }
    }
}
