package com.liveperson.messaging.model;

import android.content.ContentValues;
import android.database.Cursor;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import android.text.TextUtils;
import android.text.format.DateUtils;
import android.util.Patterns;

import com.liveperson.api.request.message.BasePublishMessage;
import com.liveperson.api.request.message.FilePublishMessage;
import com.liveperson.api.request.message.FormPublishMessage;
import com.liveperson.api.request.message.FormSubmissionPublishMessage;
import com.liveperson.api.response.events.ContentEventNotification;
import com.liveperson.api.response.model.ContentType;
import com.liveperson.api.response.model.Event;
import com.liveperson.api.response.model.Participants;
import com.liveperson.api.response.types.ConversationState;
import com.liveperson.api.response.types.DeliveryStatus;
import com.liveperson.api.response.types.DialogState;
import com.liveperson.infra.Clearable;
import com.liveperson.infra.ConversationViewParams;
import com.liveperson.infra.ICallback;
import com.liveperson.infra.Infra;
import com.liveperson.infra.configuration.Configuration;
import com.liveperson.infra.controller.DBEncryptionHelper;
import com.liveperson.infra.controller.DBEncryptionService;
import com.liveperson.infra.database.BaseDBRepository;
import com.liveperson.infra.database.DBUtilities;
import com.liveperson.infra.database.DataBaseCommand;
import com.liveperson.infra.database.DataBaseExecutor;
import com.liveperson.infra.database.tables.BaseTable;
import com.liveperson.infra.database.tables.DialogsTable;
import com.liveperson.infra.database.tables.FilesTable;
import com.liveperson.infra.database.tables.MessagesTable;
import com.liveperson.infra.database.tables.UsersTable;
import com.liveperson.infra.database.transaction_helper.InsertOrUpdateSQLCommand;
import com.liveperson.infra.database.transaction_helper.InsertSQLCommand;
import com.liveperson.infra.database.transaction_helper.SQLiteCommand;
import com.liveperson.infra.database.transaction_helper.UpdateSQLCommand;
import com.liveperson.infra.log.FlowTags;
import com.liveperson.infra.log.LPLog;
import com.liveperson.infra.messaging.R;
import com.liveperson.infra.sdkstatemachine.shutdown.ShutDown;
import com.liveperson.infra.utils.EncryptionVersion;
import com.liveperson.infra.utils.ImageUtils;
import com.liveperson.infra.utils.LinkUtils;
import com.liveperson.infra.utils.UniqueID;
import com.liveperson.messaging.Messaging;
import com.liveperson.messaging.MessagingFactory;
import com.liveperson.messaging.commands.DeliveryStatusUpdateCommand;
import com.liveperson.messaging.commands.pusher.GetUnreadMessagesCountCommand;
import com.liveperson.messaging.commands.pusher.SendReadAcknowledgementCommand;
import com.liveperson.messaging.network.MessageTimeoutQueue;
import com.liveperson.messaging.network.http.MessageTimeoutListener;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.concurrent.TimeUnit;

import static com.liveperson.infra.errors.ErrorCode.ERR_00000076;
import static com.liveperson.infra.errors.ErrorCode.ERR_00000077;
import static com.liveperson.infra.errors.ErrorCode.ERR_00000078;
import static com.liveperson.infra.errors.ErrorCode.ERR_00000079;
import static com.liveperson.infra.errors.ErrorCode.ERR_0000007A;
import static com.liveperson.infra.errors.ErrorCode.ERR_0000007B;
import static com.liveperson.infra.errors.ErrorCode.ERR_0000007C;
import static com.liveperson.infra.errors.ErrorCode.ERR_0000007D;
import static com.liveperson.infra.errors.ErrorCode.ERR_0000007E;
import static com.liveperson.infra.errors.ErrorCode.ERR_0000007F;
import static com.liveperson.infra.errors.ErrorCode.ERR_00000080;
import static com.liveperson.infra.errors.ErrorCode.ERR_00000081;
import static com.liveperson.infra.errors.ErrorCode.ERR_00000082;
import static com.liveperson.infra.errors.ErrorCode.ERR_00000083;
import static com.liveperson.infra.errors.ErrorCode.ERR_00000084;
import static com.liveperson.infra.errors.ErrorCode.ERR_00000085;
import static com.liveperson.infra.errors.ErrorCode.ERR_00000086;
import static com.liveperson.infra.errors.ErrorCode.ERR_00000087;
import static com.liveperson.infra.errors.ErrorCode.ERR_00000088;

/**
 * Created by Ilya Gazman on 11/10/2015.
 * <p/>
 * Messages model
 */
public class AmsMessages extends BaseDBRepository implements ShutDown, Clearable, AmsMessagesLoaderProvider {

	private static final String TAG = "AmsMessages";

	public enum MessagesSortedBy {
		TargetId, ConversationId
	}

	public interface MessagesListener {

		void initMessages(ArrayList<FullMessageRow> searchedMessageList);

		void onQueryMessagesResult(long firstTimestamp, long lastTimestamp);

		void onUpdateMessages(long firstTimestamp, long lastTimestamp);

		void onNewMessage(FullMessageRow fullMessageRow);

		void onUpdateMessage(FullMessageRow fullMessageRow);

		/**
		 * This only being used when clear conversation history button clicked.
		 */
		void removeAllClosedConversations(String targetId);

		/**
		 * Clear all messages. Used when move to background for un-auth account or the SDK version change.
		 */
		void clearAllMessages(String targetId);

		void onHistoryFetched();

		void onExConversationHandled(boolean emptyNotification);

		void onHistoryFetchedFailed();
	}

    private static final int MAX_SQL_VARIABLES = 997;

	public static final int PENDING_MSG_SEQUENCE_NUMBER = -1;
	public static final int RESOLVE_MSG_SEQUENCE_NUMBER = -2;
	public static final int MASKED_CC_MSG_SEQUENCE_NUMBER = -3;
	public static final int WELCOME_MSG_SEQUENCE_NUMBER = -4;

    private final Messaging mController;
	public final FormsManager mFormsManager;

	private MessagesListener mMessagesListener = null;
	private MessagesListener mNullMessagesListener = new NullMessagesListener();

	// Holds the most recent agent message quick replies data
	private QuickRepliesMessageHolder mQuickRepliesMessageHolder = null;

	public AmsMessages(Messaging controller) {
		super(MessagesTable.MESSAGES_TABLE);

		mController = controller;
		MessageTimeoutListener mMessageTimeoutListener = new MessageTimeoutListener() {
			@Override
			public void onMessageTimeout(String brandId) {
				mController.onMessageTimeout(brandId);
				LPLog.INSTANCE.e(TAG, ERR_00000076, "on message timeout received");
			}

			@Override
			public void onPublishMessageTimeout(String brandId, String eventId, String dialogId) {
				updateMessageState(eventId, MessagingChatMessage.MessageState.ERROR);
				LPLog.INSTANCE.e(TAG, ERR_00000077, "on update message timeout");
			}
		};
		mMessageTimeoutQueue = new MessageTimeoutQueue(mMessageTimeoutListener);
		mFormsManager = new FormsManager();
	}

	// A queue for storing messages for a specific timeout so we can mark them as error if not getting ack on them
	public final MessageTimeoutQueue mMessageTimeoutQueue;

	/**
	 * Notify of data change from change of agent profile
	 */
	public DataBaseCommand<Void> updateOnCommand() {

		return new DataBaseCommand<>(() -> null);
	}

	/**
	 * MUST RUN ON DB THREAD!!
	 *
	 * @param brandID
	 * @param limitSize          max num of messages required.  0 or negative value for no limit
	 * @param olderThanTimestamp negative value if there is no need to use this value
	 * @param newerThanTimestamp negative value if there is no need to use this value
	 * @return cursor with messages according to params
	 */
	private Cursor messagesByTarget(String brandID, int limitSize, long olderThanTimestamp, long newerThanTimestamp) {

		StringBuilder sql = getMessagesForTargetQuery().append(" WHERE ").append(DialogsTable.TABLE_NAME)
				.append(".").append(DialogsTable.Key.BRAND_ID).append(" = \"").append(brandID).append("\"");

		if (olderThanTimestamp > -1) {
			sql.append(" AND ").append(MessagesTable.KEY_TIMESTAMP).append(" <= ").append(olderThanTimestamp);
		}
		if (newerThanTimestamp > -1) {
			sql.append(" AND ").append(MessagesTable.KEY_TIMESTAMP).append(" >= ").append(newerThanTimestamp);
		}

		// LE-79838 [Android] History Control API :

		ConversationViewParams conversationViewParams = mController.getConversationViewParams();

		//bring only conversations that started/ended x days ago.
		if (conversationViewParams.getHistoryConversationsMaxDays() > -1){

			long historyConversationsMaxDays = conversationViewParams.getHistoryConversationsMaxDays() * DateUtils.DAY_IN_MILLIS;
			long daysAgo = System.currentTimeMillis() - historyConversationsMaxDays;
			switch (conversationViewParams.getHistoryConversationMaxDaysType()){
				case endConversationDate:
					sql.append(" AND ").append(DialogsTable.TABLE_NAME).append(".").append(DialogsTable.Key.END_TIMESTAMP).append(" >= ").append(daysAgo);
					break;
				case startConversationDate:
					sql.append(" AND ").append(DialogsTable.TABLE_NAME).append(".").append(DialogsTable.Key.START_TIMESTAMP).append(" >= ").append(daysAgo);
					break;
			}
		}
		//bring only conversations under specific state
		switch (conversationViewParams.getHistoryConversationsStateToDisplay()){
			case OPEN:
				sql.append(" AND ").append(DialogsTable.TABLE_NAME).append(".").append(DialogsTable.Key.STATE).append(" = ").append(ConversationState.OPEN.ordinal());
				break;
			case CLOSE:
				sql.append(" AND ").append(DialogsTable.TABLE_NAME).append(".").append(DialogsTable.Key.STATE).append(" = ").append(ConversationState.CLOSE.ordinal());
				break;
			case ALL:
				break;
		}


		//end of LE-79838 [Android] History Control API


		if (limitSize > 0) {
			sql.append(" ORDER BY ").append(MessagesTable.KEY_TIMESTAMP).append(" DESC ");
			sql.append(" LIMIT ").append(limitSize);
			sql = new StringBuilder("Select * FROM ( ").append(sql).append(" ) ORDER BY ").append(MessagesTable.KEY_TIMESTAMP).append(" ASC ");
		} else {
			sql.append(" ORDER BY ").append(MessagesTable.KEY_TIMESTAMP).append(" ASC ");
		}


		return getDB().rawQuery(sql.toString());
	}

	/**
	 * @param conversationID
	 * @param limitSize      -1 for no limit
	 * @return
	 */
	private Cursor messagesByConversationID(String conversationID, int limitSize) {
		StringBuilder sql = getBasicMessagesQuery().append(" WHERE ").append(DialogsTable.TABLE_NAME)
				.append(".").append(DialogsTable.Key.DIALOG_ID).append(" = \"").append(conversationID).append("\"").append(" ORDER BY ")
				.append(MessagesTable.KEY_TIMESTAMP);

		if (limitSize != -1) {
			sql.append(" LIMIT ").append(limitSize);
		}
		return getDB().rawQuery(sql.toString());
	}

	private StringBuilder getMessagesForTargetQuery() {
		return new StringBuilder()
				.append("select ")
				.append(MessagesTable.MESSAGES_TABLE).append(".").append(BaseTable.KEY_ID).append(",")
				.append(MessagesTable.MESSAGES_TABLE).append(".").append(MessagesTable.KEY_EVENT_ID).append(",")
				.append(MessagesTable.MESSAGES_TABLE).append(".").append(MessagesTable.KEY_ORIGINATOR_ID).append(",")
				.append(MessagesTable.MESSAGES_TABLE).append(".").append(MessagesTable.KEY_ENCRYPTION_VERSION).append(" AS ").append(MessagesTable.ENCRYPTION_VERSION_CURSOR_AS_VALUE).append(",")
				.append(UsersTable.USERS_TABLE).append(".").append(UsersTable.KEY_ENCRYPTION_VERSION).append(" AS ").append(UsersTable.ENCRYPTION_VERSION_CURSOR_AS_VALUE).append(",")
				.append(MessagesTable.KEY_SERVER_SEQUENCE).append(",")
				.append(MessagesTable.KEY_DIALOG_ID).append(",")
				.append(MessagesTable.KEY_TEXT).append(",")
				.append(MessagesTable.KEY_CONTENT_TYPE).append(",")
				.append(MessagesTable.KEY_MESSAGE_TYPE).append(",")
				.append(MessagesTable.KEY_STATUS).append(",")
				.append(MessagesTable.KEY_TIMESTAMP).append(",")
				.append(UsersTable.KEY_PROFILE_IMAGE).append(",")
				.append(UsersTable.KEY_NICKNAME).append(",")
				.append(FilesTable.FILES_TABLE).append(".").append(FilesTable.KEY_ID).append(" AS ").append(FilesTable.KEY_ID_AS_VALUE).append(",")
				.append(FilesTable.KEY_FILE_TYPE).append(",")
				.append(FilesTable.KEY_LOCAL_URL).append(",")
				.append(FilesTable.KEY_PREVIEW).append(",")
				.append(FilesTable.KEY_LOAD_STATUS).append(",")
				.append(FilesTable.KEY_RELATED_MESSAGE_ROW_ID).append(",")
				.append(FilesTable.KEY_SWIFT_PATH)
				.append(" from ")
				.append(MessagesTable.MESSAGES_TABLE)
				.append(" left join ").append(DialogsTable.TABLE_NAME).append(" on ").append(MessagesTable.MESSAGES_TABLE).append(".")
				.append(MessagesTable.KEY_DIALOG_ID).append("=").append(DialogsTable.TABLE_NAME).append(".").append(DialogsTable.Key.DIALOG_ID)
				.append(" left join ").append(UsersTable.USERS_TABLE).append(" on ").append(MessagesTable.MESSAGES_TABLE).append(".").append(MessagesTable.KEY_ORIGINATOR_ID)
				.append("=").append(UsersTable.USERS_TABLE).append(".").append(UsersTable.KEY_ORIGINATOR_ID)
				.append(" left join ").append(FilesTable.FILES_TABLE).append(" on ").append(MessagesTable.MESSAGES_TABLE).append(".").append(MessagesTable.KEY_ID)
				.append("=").append(FilesTable.FILES_TABLE).append(".").append(FilesTable.KEY_RELATED_MESSAGE_ROW_ID);
	}

	@NonNull
	private StringBuilder getBasicMessagesQuery() {
		return new StringBuilder().append("select ").append(MessagesTable.MESSAGES_TABLE).append("._id, serverSequence,convID,text,contentType,type,status,")
				.append(MessagesTable.MESSAGES_TABLE).append(".eventId,").append(MessagesTable.MESSAGES_TABLE).append(".originatorId,timeStamp,encryptVer,")
				.append("description,firstName,lastName,phoneNumber,userType,email,profileImage,coverImage from ").append(MessagesTable.MESSAGES_TABLE)
				.append(" left join ").append(DialogsTable.TABLE_NAME).append(" on ").append(MessagesTable.MESSAGES_TABLE).append(".")
				.append(MessagesTable.KEY_DIALOG_ID).append("=").append(DialogsTable.TABLE_NAME).append(".").append(DialogsTable.Key.DIALOG_ID)
				.append(" left join ").append(UsersTable.USERS_TABLE).append(" on ").append(MessagesTable.MESSAGES_TABLE).append(".").append(MessagesTable.KEY_ORIGINATOR_ID)
				.append("=").append(UsersTable.USERS_TABLE).append(".").append(UsersTable.KEY_ORIGINATOR_ID);
	}

	/**
	 * Add a message, even if the conversation is in pending state
	 */
	public DataBaseCommand<Long> addMessage(final MessagingChatMessage message, final boolean updateUI) {
		return new DataBaseCommand<>(() -> {
			long rowId;
			boolean isEmptyEventId = TextUtils.isEmpty(message.getEventId());
			if (isEmptyEventId) {
				LPLog.INSTANCE.i(TAG, "Received new message without event id, generating new one.. ");
				message.setEventId(UniqueID.createUniqueMessageEventId());
				//in case there is no event id we need to check if such message exists
				//in the db by the conversation id and sequence. is exists - update it,
				//if not, insert this message.
				String whereClause = MessagesTable.KEY_DIALOG_ID + " = ? AND " +
						MessagesTable.KEY_SERVER_SEQUENCE + " = ?";
				rowId = getDB().insertOrUpdate(getContentValuesForMessage(message),
						getContentValuesForMessageUpdate(message),
						whereClause,
						new String[]{message.getDialogId(), String.valueOf(message.getServerSequence())});
				LPLog.INSTANCE.d(TAG, "Insert or Update message: " + LPLog.INSTANCE.mask(message) + " rowId = " + rowId);
			} else {
				//Check if the message exist
				Cursor cursor = getDB().query(null, MessagesTable.KEY_EVENT_ID + " = ?", new String[]{message.getEventId()}, null, null, null);
				if (cursor != null && cursor.getCount() > 0) {
					ContentValues messageValues = getContentValuesForMessageUpdate(message, cursor);

					//If needed update the message in DB
					if (messageValues.size() > 0) {
						rowId = getDB().update(messageValues, MessagesTable.KEY_EVENT_ID + "=?",
								new String[]{String.valueOf(message.getEventId())});
						LPLog.INSTANCE.d(TAG, "Adding message: This message was update with message: " + LPLog.INSTANCE.mask(message) + " rowId = " + rowId);
					} else {
						rowId = -1;
						LPLog.INSTANCE.d(TAG, "Adding message: Skip add\\update this message since its already exist" + LPLog.INSTANCE.mask(message) + " rowId = " + rowId);
					}
				} else {
					// Add the received message to DB
					rowId = getDB().insertWithOnConflict(getContentValuesForMessage(message));
					LPLog.INSTANCE.d(TAG, "Adding message: " + LPLog.INSTANCE.mask(message) + " rowId = " + rowId);
				}
			}

			if (updateUI) {
				if (rowId != -1) {
					//new Item
					getMessagesListener().onNewMessage(createFullMessageRow(rowId, message, -1));
				} else {
					//updated item - need to read again from db cause 'message' might contain different data than db
					//we don;t have the row id on updated message - need to update by event id.
					//updateMessageByRowIdOnDbThread(rowId);
					String eventIdToUpdate = message.getEventId();
					if (isEmptyEventId) {
						LPLog.INSTANCE.d(TAG, "Updating message that originally didn't have event id. ");
						eventIdToUpdate = getEventIdForMessage(message.getDialogId(), message.getServerSequence());
					}
					if (!TextUtils.isEmpty(eventIdToUpdate)) {
						updateMessageByEventId(eventIdToUpdate);
					}
				}
			}

			return rowId;
		});
	}

	public DataBaseCommand<Void> addMultipleMessages(final ArrayList<ContentEventNotification> responseMessages, final String originatorId, final String brandId,
													 final String targetId, final String dialogId, final String conversationId, final long clockDiff, final boolean firstNotification, final boolean mShouldUpdateUI) {

		return new DataBaseCommand<>(new DataBaseCommand.QueryCommand<Void>() {

			public String[] extractLinks(String text) {
				List<String> links = new ArrayList<>();
				String[] l = text.split("\\s+");
				for (int i = 0; i < l.length; i++) {
					if (Patterns.WEB_URL.matcher(l[i]).matches() || Patterns.EMAIL_ADDRESS.matcher(l[i]).matches()) {
						links.add(l[i]);
					}
				}
				return links.toArray(new String[links.size()]);
			}

			private MessagingChatMessage.MessageType checkIfMessageContainsURLandChangeType(MessagingChatMessage.MessageType type, String msg) {
				if (LinkUtils.containsMarkdownHyperlink(msg) && type == MessagingChatMessage.MessageType.AGENT) {
					return MessagingChatMessage.MessageType.AGENT_MARKDOWN_HYPERLINK;
				}
				String[] urls = extractLinks(msg);
				if (urls.length > 0) {
					if (type == MessagingChatMessage.MessageType.CONSUMER) {
						return MessagingChatMessage.MessageType.CONSUMER_URL;
					}
					if (type == MessagingChatMessage.MessageType.CONSUMER_MASKED) {
						return MessagingChatMessage.MessageType.CONSUMER_URL_MASKED;
					}
					if (type == MessagingChatMessage.MessageType.AGENT) {
						return MessagingChatMessage.MessageType.AGENT_URL;
					}
				}
				return type;
			}

			@Override
			public Void query() {

				ArrayList<SQLiteCommand> commands;
				if (responseMessages != null) {
					LPLog.INSTANCE.d(TAG, "Start addMultipleMessages. num of commands = " + responseMessages.size());

					commands = new ArrayList<>(responseMessages.size());

					MessagingChatMessage.MessageState messageState;
					MessagingChatMessage.MessageType messageType;
					int[] sequenceList;
					ContentValues contentValues;
					StringBuilder whereBuilder;
					String[] whereArgs;

					int firstSequence = -1;
					int lastSequence = -1;

					int maxAcceptStatusSequence = -1;
					int maxReadStatusSequence = -1;

					int lastAgentMessageSequence = -1;

					for (ContentEventNotification notification : responseMessages) {

				   		if (notification.event == null){
							LPLog.INSTANCE.e(TAG, ERR_00000078, "received message with empty event! continuing to next message.. " );
							continue;
						}

						switch (notification.event.type) {
							case ContentEvent: {
								BasePublishMessage publishMessage = null;

								if (notification.event.message != null) {
									publishMessage = notification.event.message;
								}

								// If the publish message is a text and the message itself is empty we ignore this
								if ((publishMessage == null)
										|| ((publishMessage.getType() == BasePublishMessage.PublishMessageType.TEXT)
										&& TextUtils.isEmpty(publishMessage.getMessageText()))) {

									LPLog.INSTANCE.e(TAG, ERR_00000079, "Text message received in query messages is empty :| shouldn't happen! " +
											"dialogId = " + dialogId + " , sequence = " + notification.sequence);
									// it's some protocol error, lets ignore it
									continue;
								}

								// Get the content type from the event and set the message type and state
								ContentType contentType = ContentType.fromString(notification.event.contentType);


								if (TextUtils.equals(originatorId, notification.originatorId)) {
									//consumer message - make sure it's not double
									messageType = MessagingChatMessage.MessageType.getMessageContentTypeForConsumer(notification, contentType);
									messageState = MessagingChatMessage.MessageState.SENT;
								} else {
									// If this message is not from myself

									// If message from Controller
									if(notification.originatorMetadata != null && notification.originatorMetadata.mRole == Participants.ParticipantRole.CONTROLLER){
										messageType = MessagingChatMessage.MessageType.CONTROLLER_SYSTEM;
									}
									else { // Message from Agent
										messageType = MessagingChatMessage.MessageType.getMessageContentTypeForAgent(notification,contentType);
										mController.amsConversations.resetEffectiveTTR(targetId);
										mController.amsDialogs.resetEffectiveTTR(targetId);
										//saving the sequence number of the last message sent by an agent
										lastAgentMessageSequence = notification.sequence;
									}
									messageState = MessagingChatMessage.MessageState.RECEIVED;
								}

								// check if we have a link than we need to set the message type to be with URL
								if ((messageType == MessagingChatMessage.MessageType.CONSUMER) ||
										(messageType == MessagingChatMessage.MessageType.CONSUMER_MASKED) ||
										(messageType == MessagingChatMessage.MessageType.AGENT)) {

									messageType = checkIfMessageContainsURLandChangeType(messageType, publishMessage.getMessageText());
								}

								// Create the relevant message and add to DB
								createMessageInDB(commands, messageState, messageType, notification, publishMessage, contentType);

								// Get the QuickReplies JSON from the current notification
								getQuickRepliesFromEvent(brandId, notification, messageType, dialogId);

								if (firstSequence == -1) {
									firstSequence = notification.sequence;
								}
								lastSequence = notification.sequence;
							}
							break;

							case RichContentEvent: {
								BasePublishMessage publishMessage = null;

								if (notification.event.message != null) {
									publishMessage = notification.event.message;
								}

								// If the publish message is a text and the message itself is empty we ignore this
								if ((publishMessage == null)
										|| ((publishMessage.getType() == BasePublishMessage.PublishMessageType.TEXT)
										&& TextUtils.isEmpty(publishMessage.getMessageText()))) {

									LPLog.INSTANCE.e(TAG, ERR_0000007A, "Text message received in query messages is empty :| shouldn't happen! " +
											"dialogId = " + dialogId + " , sequence = " + notification.sequence);
									// it's some protocol error, lets ignore it
									continue;
								}

								// Get the content type from the event and set the message type and state
								ContentType contentType = ContentType.text_structured_content;
								messageType = MessagingChatMessage.MessageType.AGENT_STRUCTURED_CONTENT;


								if (TextUtils.equals(originatorId, notification.originatorId)) {
									//consumer message - make sure it's not double
									messageState = MessagingChatMessage.MessageState.SENT;
								} else {
									//message from agent
									messageState = MessagingChatMessage.MessageState.RECEIVED;
									mController.amsConversations.resetEffectiveTTR(targetId);
                                    mController.amsDialogs.resetEffectiveTTR(targetId);
									//saving the sequence number of the last message sent by an agent
									lastAgentMessageSequence = notification.sequence;
								}

								// Create the relevant message and add to DB
								createMessageInDB(commands, messageState, messageType, notification, publishMessage, contentType);

								// Get the QuickReplies JSON from the current notification
								getQuickRepliesFromEvent(brandId, notification, messageType, dialogId);

								if (firstSequence == -1) {
									firstSequence = notification.sequence;
								}
								lastSequence = notification.sequence;
							}
							break;
							case AcceptStatusEvent:

								messageState = getReceivedMessageState(notification.event.status);
								sequenceList = notification.event.sequenceList;

								if (messageState == null) { //TODO: investigate this line. getReceivedMessageState method marked as NonNull but there are error logs found in Kibana.
									LPLog.INSTANCE.e(TAG, ERR_0000007B, "messageState is null :| shouldn't happen! original status: " + notification.event.status +
											", dialogId = " + dialogId + " , sequence = " + Arrays.toString(sequenceList));

									continue;
								}
								//check who the is the origin of this AcceptStatus Event
								if (!TextUtils.equals(originatorId, notification.originatorId)) {
									//the AcceptStatus is from an Agent
									if(sequenceList[0] == lastAgentMessageSequence){
										LPLog.INSTANCE.d(TAG, "AcceptStatusEvent received from agent for agent message. we ignore this event. lastAgentMsgSequence = " +lastAgentMessageSequence);
										break;
									}
								}

								if ((sequenceList != null) && sequenceList.length > 0) {

									int length = sequenceList.length;
                                    int lastStatusSequence = sequenceList[length - 1];

									//saving max sequence to update all messages at the end.
									if (messageState == MessagingChatMessage.MessageState.READ  &&
											maxReadStatusSequence < lastStatusSequence){

										maxReadStatusSequence = lastStatusSequence;

									}else if (messageState == MessagingChatMessage.MessageState.RECEIVED&&
											maxAcceptStatusSequence < lastStatusSequence){

										maxAcceptStatusSequence = lastStatusSequence;

									}else{
										//for viewed STATUS / SUBMITTED / EXPIRED etc..
										for (int i = 0; i < length; i += (MAX_SQL_VARIABLES - 1)) {
											int[] tempArray;
											int size = (length - i > (MAX_SQL_VARIABLES - 1)) ? (MAX_SQL_VARIABLES - 1) : length - i;
											if (size == length) {
												tempArray = sequenceList;
											} else {
												tempArray = new int[MAX_SQL_VARIABLES];
												System.arraycopy(sequenceList, i, tempArray, 0, size);
											}

											contentValues = new ContentValues();
											whereBuilder = new StringBuilder();
											// Create where args array for the conversation ID and all the sequences from the list
											whereArgs = new String[size + 2];

											createStatementForUpdateMessagesState(dialogId, tempArray, messageState, size, contentValues, whereBuilder, whereArgs);

											commands.add(new UpdateSQLCommand(contentValues, whereBuilder.toString(), whereArgs));
										}
									}

									//for updating UI later..
									if (firstSequence == -1) {
										firstSequence = sequenceList[0];
									}

                                    //Update UI: only if the last sequence is less than the status sequence - update.
                                    //if not, it will be in the range between firstSequence and lastSequence
                                    if (lastSequence < lastStatusSequence) {
                                        lastSequence = lastStatusSequence;
                                    }
                                }

								break;
						}


					}

					LPLog.INSTANCE.d(TAG, "dialogId = " + dialogId + ", responseMessages.size()  = " + responseMessages.size() );

					// Save the last quick replies (if exist and if conversation is not closed)
					if (mQuickRepliesMessageHolder != null && !MessagingFactory.getInstance().getController().isDialogClosed(dialogId)) {
						LPLog.INSTANCE.d(TAG, "QuickReplies exist in the received message, write to SharedPrefs");
						mQuickRepliesMessageHolder.writeToSharedPreferences();
					}

					// First we update all the read than all the Accept to minimize the updates requests!
					if (maxReadStatusSequence > -1){
						LPLog.INSTANCE.d(TAG, "dialogId = " + dialogId + ", maxReadStatusSequence = " + maxReadStatusSequence );
						commands.add(createStatementForUpdateMaxMessagesState(dialogId, MessagingChatMessage.MessageState.READ, maxReadStatusSequence));
					}
					//if accept is less or equals to read, no need to update
					if (maxAcceptStatusSequence > -1 && maxAcceptStatusSequence > maxReadStatusSequence){
						LPLog.INSTANCE.d(TAG, "dialogId = " + dialogId + ", maxAcceptStatusSequence = " + maxAcceptStatusSequence );
						commands.add(createStatementForUpdateMaxMessagesState(dialogId, MessagingChatMessage.MessageState.RECEIVED, maxAcceptStatusSequence));
					}

					getDB().runTransaction(commands);

					if (mShouldUpdateUI) {
						updateMessages(firstNotification, dialogId, firstSequence, lastSequence);
					}

					// Clear pusher's unread count for dialogId.
					sendReadAckToPusherIfRequired(conversationId, dialogId, brandId);
					//sending update on messages status.
					sendReadAckOnMessages(brandId, targetId, originatorId);
				}
				return null;
			}

			@NonNull
			private MessagingChatMessage createMessageInDB(ArrayList<SQLiteCommand> commands, MessagingChatMessage.MessageState messageState, MessagingChatMessage.MessageType messageType, ContentEventNotification notification, BasePublishMessage publishMessage, ContentType contentType) {
				StringBuilder whereBuilder;
				MessagingChatMessage message;
				String eventId = notification.eventId;
				if (TextUtils.isEmpty(eventId)) {
					eventId = UniqueID.createUniqueMessageEventId();

					LPLog.INSTANCE.d(TAG, "no event id for message: " + LPLog.INSTANCE.mask(publishMessage.getMessageText()) + " creating event id: " + eventId);

					//in case there is no event id we need to check if such message exists
					//in the db by the conversation id and sequence. is exists - update it,
					//if not, insert this message.
					whereBuilder = new StringBuilder();
					whereBuilder.append(MessagesTable.KEY_DIALOG_ID).append(" = ? AND ")
							.append(MessagesTable.KEY_SERVER_SEQUENCE).append(" = ?");

					message = createMessage(publishMessage.getMessageText(), messageState, messageType, notification, eventId, contentType.getText());
					InsertOrUpdateSQLCommand insertOrUpdateSQLCommand = new InsertOrUpdateSQLCommand(
							getContentValuesForMessage(message),
							getContentValuesForMessageUpdate(message), whereBuilder.toString(),
							new String[]{dialogId, String.valueOf(notification.sequence)});

					// Add a mMessagesListener on the SQL command in order to add the image file when finished (if required)
					AddSqliteCommandListenerForFile(publishMessage, insertOrUpdateSQLCommand);

					// Add a treatment for form invitation and submission
					AddCommandForForm(publishMessage, message, brandId, commands);

					commands.add(insertOrUpdateSQLCommand);

				} else {

					//we got message with eventId - we insert, if it's exists it will replace.
					message = createMessage(publishMessage.getMessageText(), messageState, messageType, notification, eventId, contentType.getText());

					//Check if the message exist
					Cursor cursor = getDB().query(null, MessagesTable.KEY_EVENT_ID + " = ?", new String[]{message.getEventId()}, null, null, null);
					if (cursor != null && cursor.getCount() > 0) {
						ContentValues messageValues = getContentValuesForMessageUpdate(message, cursor);

						//If needed update the message in DB
						if (messageValues.size() > 0) {
							LPLog.INSTANCE.d(TAG, "Updating message: This message need to be update with message: " + LPLog.INSTANCE.mask(message));
							UpdateSQLCommand updateSQLCommand = new UpdateSQLCommand(messageValues, MessagesTable.KEY_EVENT_ID + "=?",
									new String[]{String.valueOf(message.getEventId())});

							// Add a mMessagesListener on the SQL command in order to add the image file when finished (if required)
							AddSqliteCommandListenerForFile(publishMessage, updateSQLCommand);

							// Add a treatment for form invitation and submission
							AddCommandForForm(publishMessage, message, brandId, commands);

							commands.add(updateSQLCommand);

						} else {
							LPLog.INSTANCE.d(TAG, "Updating message: Skip updating this message since its already exist" + LPLog.INSTANCE.mask(message));
						}
					} else {
						//we got message with eventId - we insert, if it's exists it will replace.
						InsertSQLCommand insertSQLCommand = new InsertSQLCommand(getContentValuesForMessage(message));

						// Add a mMessagesListener on the SQL command in order to add the image file when finished (if required)
						AddSqliteCommandListenerForFile(publishMessage, insertSQLCommand);

						// Add a treatment for form invitation and submission
						AddCommandForForm(publishMessage, message, targetId, commands);

						commands.add(insertSQLCommand);
					}

				}
				return message;
			}

			@NonNull
			private MessagingChatMessage createMessage(String textMessage, MessagingChatMessage.MessageState messageState, MessagingChatMessage.MessageType messageType, ContentEventNotification notification, String eventId, String contentType) {
				MessagingChatMessage message = new MessagingChatMessage(
						notification.originatorId,
						textMessage,
						notification.serverTimestamp + clockDiff,
						dialogId,
						eventId,
						messageType,
						messageState,
						notification.sequence,
						contentType,
						EncryptionVersion.NONE);

                LPLog.INSTANCE.d(TAG, "creating message '" + LPLog.INSTANCE.mask(textMessage) + "', seq: " + notification.sequence + ", at time: " + notification.serverTimestamp
						+ ", dialogId: " + dialogId + ", clock diff: " + clockDiff + " = " + (notification.serverTimestamp + clockDiff));
                return message;
            }

			private void AddCommandForForm(BasePublishMessage publishMessage, MessagingChatMessage msg, String brandId, ArrayList<SQLiteCommand> commands) {
				if (publishMessage.getType() == BasePublishMessage.PublishMessageType.FORM_INVITATION) {
					FormPublishMessage formPublishMessage = (FormPublishMessage) publishMessage;
					LPLog.INSTANCE.d(TAG, "onResult: new form obj to DB getMessage " + LPLog.INSTANCE.mask(formPublishMessage.getMessage()));
					mController.amsMessages.mFormsManager.addForm(formPublishMessage.getInvitationId(),
							new Form(
									msg.getDialogId(),
									conversationId,
									formPublishMessage.getInvitationId(),
									formPublishMessage.getFormId(),
									formPublishMessage.getFormTitle(),
									mController.mAccountsController.getTokenizerUrl(brandId),
									brandId,
									msg.getServerSequence(),
									msg.getEventId()
							));
				}
				if (publishMessage.getType() == BasePublishMessage.PublishMessageType.FORM_SUBMISSION) {
					FormSubmissionPublishMessage formPublishMessage = (FormSubmissionPublishMessage) publishMessage;
					Form form = mController.amsMessages.mFormsManager.getForm(formPublishMessage.getInvitationId());
					if (form != null) {
						mController.amsMessages.mFormsManager.updateForm(formPublishMessage.getInvitationId(), formPublishMessage.getSubmissionId());
						LPLog.INSTANCE.d(TAG, "Updating message: This message need to be update with message: ");
						ContentValues cv = new ContentValues();
						cv.put(MessagesTable.KEY_STATUS, getReceivedMessageState(DeliveryStatus.SUBMITTED).ordinal());
						UpdateSQLCommand updateSQLCommand = new UpdateSQLCommand(cv, MessagesTable.KEY_EVENT_ID + "=?",
								new String[]{String.valueOf(form.getEventId())});
						commands.add(updateSQLCommand);
					}


				}
			}

			/**
			 * Add a mMessagesListener to the given sqLiteCommand to add the image file after the command is complete
			 * @param publishMessage
			 * @param sqLiteCommand
			 */
			private void AddSqliteCommandListenerForFile(BasePublishMessage publishMessage, SQLiteCommand sqLiteCommand) {

				// Validate the given publishMessage is of type file
				if (publishMessage.getType() == BasePublishMessage.PublishMessageType.FILE) {

					final BasePublishMessage finalPublishMessage = publishMessage;
					// Listener
					sqLiteCommand.setListener(rowId -> {
						String tag = "onInsertComplete";

						if (rowId == DBUtilities.ROW_UPDATED) {
							LPLog.INSTANCE.d(TAG, tag + ": message was updated on DB (and not inserted). No need to add the file to DB");
						} else {
							addFileFromPublishMessageToDB(rowId, tag, (FilePublishMessage) finalPublishMessage, targetId);
						}
					});
				}
			}

		});
	}

	/**
	 * Update messages of a conversation once it is re-fetched from INCA server. This will make sure we have
	 * PII data in it's masked format.
	 * @param responseMessages conversation response
	 * @param dialogId re-fetched dialog
	 * @return DataBaseCommand
	 */
	public DataBaseCommand<Void> updateMultipleMessages(final ArrayList<ContentEventNotification> responseMessages, final String dialogId) {

		// end query()
		return new DataBaseCommand<>(() -> {
			if (responseMessages != null) {
				LPLog.INSTANCE.d(TAG, "updateMultipleMessages: Start updating messages for dialogId = " + dialogId);

				DataBaseExecutor.execute(() -> {
					for (ContentEventNotification notification : responseMessages) {
						if (notification.event == null || notification.event.type == Event.Types.AcceptStatusEvent) {
							LPLog.INSTANCE.i(TAG, "updateMultipleMessages: Ignore messages with empty or are of type AcceptStatusEvent");
							continue;
						}

						BasePublishMessage publishMessage = null;

						if (notification.event.message != null) {
							publishMessage = notification.event.message;
						}

						// If the publish message is a text and the message itself is empty we ignore this
						if (publishMessage == null ||
								(publishMessage.getType() == BasePublishMessage.PublishMessageType.TEXT
										&& TextUtils.isEmpty(publishMessage.getMessageText()))) {
							LPLog.INSTANCE.e(TAG, ERR_0000007C, "updateMultipleMessages: Text message received in query messages is empty :| shouldn't happen! " +
									"dialogId = " + dialogId + " , sequence = " + notification.sequence);
							// it's some protocol error, lets ignore it
							continue;
						}

						String sequence = String.valueOf(notification.sequence);
						ContentValues contentValues = new ContentValues();
						final EncryptionVersion messageEncryptionVersion = DBEncryptionService.Companion.getAppEncryptionVersion();
						contentValues.put(MessagesTable.KEY_ENCRYPTION_VERSION, messageEncryptionVersion.ordinal());
						String encryptedMessage = DBEncryptionHelper.encrypt(messageEncryptionVersion, publishMessage.getMessageText());
						// Put message text to be updated
						contentValues.put(MessagesTable.KEY_TEXT, encryptedMessage);

						String whereBuilder = MessagesTable.KEY_DIALOG_ID + " = ? AND " +
								MessagesTable.KEY_SERVER_SEQUENCE + " = ?";
						LPLog.INSTANCE.d(TAG, "updateMultipleMessages: Updating message with sequence: " + sequence);
						getDB().update(contentValues, whereBuilder, new String[]{dialogId, sequence});
					} // end for loop
				});
			}
			return null;
		});
	}

	/**
	 * Get the QuickReplies JSON from the given notification. This only happen for message from the agent
	 * @param brandId
	 * @param notification
	 * @param messageType
	 * @param dialogId
	 */
	private void getQuickRepliesFromEvent(String brandId, ContentEventNotification notification, MessagingChatMessage.MessageType messageType, String dialogId) {

		// If conversation is closed don't add QuickReplies
		if(MessagingFactory.getInstance().getController().isDialogClosed(dialogId)) {
			LPLog.INSTANCE.d(TAG, "getQuickRepliesFromEvent: conversation is closed, not adding QuickReplies message");
			return;
		}

		// Save the message only if from agent
		if ((messageType == MessagingChatMessage.MessageType.AGENT) ||
				(messageType == MessagingChatMessage.MessageType.AGENT_STRUCTURED_CONTENT) ||
				(messageType == MessagingChatMessage.MessageType.AGENT_URL)) {


			QuickRepliesMessageHolder currentQuickRepliesMessageHolder = QuickRepliesMessageHolder.fromContentEventNotification(brandId, notification);

			LPLog.INSTANCE.d(TAG, "getQuickRepliesFromEvent: Message is from agent, try to get QuickReplies string from event");

			// Store the current QR message only if newer
			if (mQuickRepliesMessageHolder == null ||
					((currentQuickRepliesMessageHolder != null) && currentQuickRepliesMessageHolder.newerThan(mQuickRepliesMessageHolder))) {

				LPLog.INSTANCE.d(TAG, FlowTags.QUICK_REPLIES, "QuickReplies message is newer than the current one. New one: " + ((currentQuickRepliesMessageHolder != null) ? currentQuickRepliesMessageHolder.toString() : "null"));

				mQuickRepliesMessageHolder = currentQuickRepliesMessageHolder;
			}
		}
	}

	/**
	 * Get the {@link QuickRepliesMessageHolder} stored in memory. If not exist in memory load from SharedPreferences
	 * @param brandId
	 * @return
	 */
	@Override
	public QuickRepliesMessageHolder getQuickRepliesMessageHolder(String brandId) {

		// If does not exist in memory load from SharedPreferences
		if (mQuickRepliesMessageHolder == null) {
			mQuickRepliesMessageHolder = QuickRepliesMessageHolder.loadFromSharedPreferences(brandId);
		}

		return mQuickRepliesMessageHolder;
	}

	@Override
	public void resetQuickRepliesMessageHolder() {
		LPLog.INSTANCE.d(TAG, "resetQuickRepliesMessageHolder: resetting QuickRepliesMessageHolder");
		mQuickRepliesMessageHolder = null;
	}

	//Check if the message state and server sequence need to be updated in the existing message
	@NonNull
	private ContentValues getContentValuesForMessageUpdate(MessagingChatMessage message, Cursor cursor) {
		MessagingChatMessage existingMessage = getSingleMessageFromCursor(cursor);
		ContentValues messageValues = new ContentValues();

		//Check if we need to update the existing message
		if ((message.getMessageState().ordinal() != existingMessage.getMessageState().ordinal()) &&
				(MessagingChatMessage.MessageState.validChange(existingMessage.getMessageState(), message.getMessageState()))) {
			messageValues.put(MessagesTable.KEY_STATUS, message.getMessageState().ordinal());
		} else {
			LPLog.INSTANCE.d(TAG, "Skip update message state, old val: " + message.getMessageState() + " , new val: " + existingMessage.getMessageState());
		}
		if (message.getServerSequence() != existingMessage.getServerSequence()) {
			messageValues.put(MessagesTable.KEY_SERVER_SEQUENCE, message.getServerSequence());
		} else {
			LPLog.INSTANCE.d(TAG, "Skip update message server sequence, old val: " + message.getServerSequence() + " , new val: " + existingMessage.getServerSequence());
		}
		return messageValues;
	}

    /**
     * MUST BE CALLED FROM DB Thread!
     *  @param messageRowId
     * @param tag
	 * @param finalPublishMessage
	 * @param targetId
	 */
	private void addFileFromPublishMessageToDB(long messageRowId, String tag, FilePublishMessage finalPublishMessage, String targetId) {

		LPLog.INSTANCE.d(TAG, tag + ": MessagingChatMessage was added. row id: " + messageRowId + ". Adding fileMessage to db.");

        // We first save the base64 as bitmap to disk
        String previewImagePath = ImageUtils.saveBase64ToDisk(Infra.instance.getApplicationContext(), finalPublishMessage.getPreview(), targetId);

		LPLog.INSTANCE.d(TAG, tag + ": preview image saved to location: " + previewImagePath);

		if (previewImagePath != null) {
            // Add the file the DB
            FileMessage fileMessage = new FileMessage(previewImagePath, finalPublishMessage.getFileType(), null, finalPublishMessage.getRelativePath(), messageRowId);
            long fileRowId = MessagingFactory.getInstance().getController().amsFiles.addFile(messageRowId, fileMessage).executeSynchronously();
            LPLog.INSTANCE.d(TAG, tag + ": fileMessage was added to db. fileRowId = " + fileRowId);

		}
	}

	@NonNull
	private MessagingChatMessage.MessageState getReceivedMessageState(DeliveryStatus status) {
		MessagingChatMessage.MessageState messageState = null;

		switch (status) {
			case ACCEPT:
				messageState = MessagingChatMessage.MessageState.RECEIVED;
				break;
			case READ:
			case ACTION:
				messageState = MessagingChatMessage.MessageState.READ;
				break;
			case VIEWED:
				messageState = MessagingChatMessage.MessageState.VIEWED;
				break;
			case SUBMITTED:
				messageState = MessagingChatMessage.MessageState.SUBMITTED;
				break;
			case ERROR:
			case ABORTED:
				messageState = MessagingChatMessage.MessageState.ERROR;
				break;
		}
		return messageState;
	}

	/**
	 * Create where statement to update list of sequences (messages) state
	 *
	 * @param dialogId
	 * @param sequenceList
	 * @param messageState
	 * @param length
	 * @param contentValues
	 * @param whereBuilder
	 * @param whereArgs
	 */
	private void createStatementForUpdateMessagesState(String dialogId, int[] sequenceList, MessagingChatMessage.MessageState messageState, int length, ContentValues contentValues, StringBuilder whereBuilder, String[] whereArgs) {
		// The value to be changed
		contentValues.put(MessagesTable.KEY_STATUS, messageState.ordinal());

		// Add the conversation ID as the first argument of the where
		whereArgs[0] = String.valueOf(dialogId);
		whereArgs[1] = String.valueOf(messageState.ordinal());
		whereBuilder.append(MessagesTable.KEY_DIALOG_ID).append(" =? AND ").
				append(MessagesTable.KEY_STATUS).append(" <?  AND ").
				append(MessagesTable.KEY_SERVER_SEQUENCE).append(" in (");

		// Add all sequences to the where args and to the where clause builder
		for (int i = 0; i < length; i++) {
			whereArgs[i + 2] = String.valueOf(sequenceList[i]);
			whereBuilder.append("?");

			// If last item don't add comma
			if (i != length - 1) {
				whereBuilder.append(",");
			}
		}

		whereBuilder.append(")");
	}

	private UpdateSQLCommand createStatementForUpdateMaxMessagesState(String dialogId,MessagingChatMessage.MessageState messageState, int maxSequence) {
		ContentValues contentValues = new ContentValues();
		StringBuilder whereBuilder = new StringBuilder();

		// The value to be changed
		contentValues.put(MessagesTable.KEY_STATUS, messageState.ordinal());

		// Add the conversation ID as the first argument of the where
		String[] whereArgs = new String[]{
				String.valueOf(dialogId),
				String.valueOf(messageState.ordinal()),
				String.valueOf(maxSequence),
				String.valueOf(PENDING_MSG_SEQUENCE_NUMBER)};

		whereBuilder.append(MessagesTable.KEY_DIALOG_ID).append(" =? AND ").
				append(MessagesTable.KEY_STATUS).append(" <? AND ").
				append(MessagesTable.KEY_SERVER_SEQUENCE).append(" <=? AND " ).
				append(MessagesTable.KEY_SERVER_SEQUENCE).append(" >? ");

		return new UpdateSQLCommand(contentValues, whereBuilder.toString(), whereArgs);
	}

	public void updateAllMessagesStateByDialogId(final String dialogId, final MessagingChatMessage.MessageState messageState) {
		DataBaseExecutor.execute(() -> {
			ContentValues contentValues = new ContentValues();
			// The value to be changed
			contentValues.put(MessagesTable.KEY_STATUS, messageState.ordinal());

			int result = getDB().update(contentValues, MessagesTable.KEY_DIALOG_ID + " = ? ", new String[]{dialogId});
			LPLog.INSTANCE.d(TAG, String.format(Locale.ENGLISH, "Updated %d messages on DB with state %s", result, messageState));

			//update list listener
			updateAllMessagesForDialog(dialogId);
		});
	}


	// Change the messages state of all the closed conversation from pending to close
	private void markPendingMessagesAsFailedOnCloseConv(final String brandId) {

		DataBaseExecutor.execute(() -> {

			// Get images' rowId currently in upload progress.
			// This is required since we don't want to mark these messages as failed
			String rowIds = MessagingFactory.getInstance().getController().getInProgressUploadMessageRowIdsString();

			String[] whereArgs = getPendingMessagesQueryParams(brandId, rowIds, String.valueOf(ConversationState.CLOSE.ordinal()));
			String where = getPendingMessagesQuery(rowIds);

			//update list listener - need to query all messages that are going to be effected before setting state - than update UI on each message that something changed.
			Cursor cursor = getDB().query(null, where, whereArgs, null, null, null);
			if (cursor != null) {
				try {
					if (cursor.getCount() == 0) {
						//No pending failed messages to update
						return;
					}
					if (cursor.moveToFirst()) {
						do {
							//there are failed messages that need update.
							MessagingChatMessage message = getSingleMessageFromCursor(cursor);
							message.setMessageState(MessagingChatMessage.MessageState.ERROR);
							long rowId = cursor.getLong(cursor.getColumnIndex(MessagesTable.KEY_ID));
							updateMessageByRowId(rowId, message);
						} while (cursor.moveToNext());
					}
				} finally {
					cursor.close();
				}

				// Update the DB only if we have result for the query
				ContentValues contentValues = new ContentValues();
				// The value to be changed
				contentValues.put(MessagesTable.KEY_STATUS, MessagingChatMessage.MessageState.ERROR.ordinal());

				//m.status = 3 and c.brandId = "qa81253845" and m.convID = c.conversationId
				int result = getDB().update(contentValues, where, whereArgs);
				LPLog.INSTANCE.d(TAG, String.format(Locale.ENGLISH, "Updated %d messages on DB with state %s", result, MessagingChatMessage.MessageState.ERROR));
			}
		});
	}

	public void resendAllPendingMessages(final String brandId) {

		final long resendMessageTimeout = Configuration.getInteger(R.integer.sendingMessageTimeoutInMinutes);

		// Change the messages state of all the closed conversation from pending to close
		markPendingMessagesAsFailedOnCloseConv(brandId);

		DataBaseExecutor.execute(() -> {

			// Get images' rowId currently in upload progress.
			// This is required since we don't want to mark these messages as failed
			String rowIds = MessagingFactory.getInstance().getController().getInProgressUploadMessageRowIdsString();

			String[] whereArgs = getPendingMessagesQueryParams(brandId, rowIds, String.valueOf(ConversationState.OPEN.ordinal()));
			String where = getPendingMessagesQuery(rowIds);

			//update list listener - need to query all messages that are going to be effected before setting state - than update UI on each message that something changed.
			Cursor cursor = getDB().query(null, where, whereArgs, null, null, null);
			if (cursor != null) {
				try {
					if (cursor.getCount() == 0) {
						//No pending messages
						return;
					}
					if (cursor.moveToFirst()) {
						ArrayList<String> eventIdsToErrorUpdate = new ArrayList<>();
						do {
							//Resend all pending messages that didn't pass the timeout
							MessagingChatMessage message = getSingleMessageFromCursor(cursor);
							String dialogId = message.getDialogId();
							if (resendMessageTimeout > 0 && System.currentTimeMillis() < message.getTimeStamp() + TimeUnit.MINUTES.toMillis(resendMessageTimeout)) {
								LPLog.INSTANCE.d(TAG, "Resend message: " + LPLog.INSTANCE.mask(message));
								long rowId = Messaging.NO_FILE_ROW_ID;

								// Get rowId in order to pull the image fileMessage from the DB
								if (MessagingChatMessage.MessageType.isImage(message.getMessageType())) {
									FileMessage fileMessage = mController.amsFiles.getFileByMessageRowId(message.getLocalId());
									if (fileMessage != null) {
										rowId = fileMessage.getFileRowId();
									}
								}
								mController.resendMessage(message.getEventId(), dialogId, rowId, message.getMessageType());
							} else {
								LPLog.INSTANCE.d(TAG, "Resend timeout - Set message to FAILED state,  resendMessageTimeout:" + resendMessageTimeout + ", message: " + LPLog.INSTANCE.mask(message));
								eventIdsToErrorUpdate.add(message.getEventId());
							}
						} while (cursor.moveToNext());

						if (!eventIdsToErrorUpdate.isEmpty()) {
							// Change message state to failed - resend timeout
							updateMessagesState(eventIdsToErrorUpdate, MessagingChatMessage.MessageState.ERROR);
						}
					}
				} finally {
					cursor.close();
				}
			}
		});
	}

	private String getPendingMessagesQuery(String rowIds) {
		/*
		update Messages set status = error where exists for all the rows in this query:
        (select row_id from Messages m, Conversations c where c.state = (m.status = pending or m.status = queue) and c.brandId = "qa81253845" and m.convID = c.conversationId)
         */
		StringBuilder whereBuilder = new StringBuilder().append(MessagesTable.KEY_ID).append(" in (select m.").append(MessagesTable.KEY_ID).append(" from ")
				.append(MessagesTable.MESSAGES_TABLE).append(" m , ")
				.append(DialogsTable.TABLE_NAME).append(" c ")
				.append("where (m.").append(MessagesTable.KEY_STATUS).append("=").append(MessagingChatMessage.MessageState.PENDING.ordinal())
				.append(" or m.").append(MessagesTable.KEY_STATUS).append("=").append(MessagingChatMessage.MessageState.QUEUED.ordinal())
				.append(") and c.").append(DialogsTable.Key.BRAND_ID).append("=?")
				.append(" and c.").append(DialogsTable.Key.STATE).append("=?")
				.append(" and m.").append(MessagesTable.KEY_DIALOG_ID).append("= c.").append(DialogsTable.Key.DIALOG_ID);

		// If there are images in upload progress add them to the where clause
		if (!TextUtils.isEmpty(rowIds)) {
			LPLog.INSTANCE.d(TAG, "resendAllPendingMessages: There is upload images in progress, ignore these messages rowId: " + rowIds);
			whereBuilder.append(" and m.").append(MessagesTable.KEY_ID).append(" not in (?)");
		}

		whereBuilder.append(")");// Close the where clause
		String where = whereBuilder.toString();
		LPLog.INSTANCE.d(TAG, "getPendingMessagesQuery: where clause: " + where);
		return where;
	}

	private String[] getPendingMessagesQueryParams(String brandId, String rowIds, String state) {
		// If there are images in upload progress add them to the where clause
		if (!TextUtils.isEmpty(rowIds)) {
			return new String[]{brandId, state, rowIds};
		} else { // where without image rowId
			return new String[]{brandId, state};
		}
	}

	/**
	 * Update the messageState to a message with the given eventId. The targetId and dialogId params
	 * are required here in order to update the model after the update
	 *
	 * @param eventId
	 * @param messageState
	 */
	public void updateMessageState(final String eventId, final MessagingChatMessage.MessageState messageState) {
		// SQL query: update messages set status = <messageState> where eventId = <eventId>
		DataBaseExecutor.execute(() -> {
			// Update the status of the message
			ContentValues contentValues = new ContentValues();
			// The value to be changed
			contentValues.put(MessagesTable.KEY_STATUS, messageState.ordinal());
			int result = getDB().update(contentValues, MessagesTable.KEY_EVENT_ID + "=?", new String[]{eventId});
			LPLog.INSTANCE.d(TAG, String.format(Locale.ENGLISH, "Updated %d messages on DB with state %s, eventId: %s", result, messageState, eventId));
			//update list listener
			updateMessageByEventId(eventId);
		});
	}

	/**
	 * Update the messageState to array of messages based on given eventId. The targetId and conversationId params
	 * are required here in order to update the model after the update
	 *
	 * @param eventIds
	 * @param messageState
	 */
	public void updateMessagesState(final ArrayList<String> eventIds, final MessagingChatMessage.MessageState messageState) {

		// SQL query: update messages set status = <messageState> where eventId = <eventId>
		DataBaseExecutor.execute(() -> {

			if (eventIds != null && !eventIds.isEmpty()) {
				// Update the status of the message
				ContentValues contentValues = new ContentValues();

				// The value to be changed
				contentValues.put(MessagesTable.KEY_STATUS, messageState.ordinal());

				StringBuilder inOperator = new StringBuilder(" IN (?");
				for (int i = 1; i < eventIds.size(); i++) {
					inOperator.append(",?");
				}
				inOperator.append(")");

				String[] eventIdsArray = new String[eventIds.size()];
				eventIdsArray = eventIds.toArray(eventIdsArray);

				int result = getDB().update(contentValues, MessagesTable.KEY_EVENT_ID + inOperator.toString(), eventIdsArray);

				LPLog.INSTANCE.d(TAG, String.format(Locale.ENGLISH, "Updated %d messages on DB with state %s, eventId: %s", result, messageState, eventIds));

				for (String eventId : eventIdsArray) {
					//update list listener
					updateMessageByEventId(eventId);
				}

			} else {
				LPLog.INSTANCE.d(TAG, "updateMessagesState - Skip updated messages , eventID is empty. messageState = " + messageState);
			}
		});
	}

	public void sendReadAckOnMessages(final String brandId, final String targetId, final String originatorId) {

		DataBaseExecutor.execute(() -> {

			// select serverSequence from messages where conversation state != CLOSED || LOCKED and targetId=<targetId> and messageState=ACCEPT
			Cursor cursor = null;
			if (!TextUtils.isEmpty(targetId)) {

				LPLog.INSTANCE.d(TAG, "Get all unread messages for target " + targetId);

				String queryString = String.format("select m.%s, c.%s, c.%s from %s m, %s c where c.%s>? and c.%s=? and m.%s =? and m.%s=c.%s and m.%s >= '0' and m.%s != ?",
						MessagesTable.KEY_SERVER_SEQUENCE, DialogsTable.Key.DIALOG_ID, DialogsTable.Key.CONVERSATION_ID, MessagesTable.MESSAGES_TABLE, DialogsTable.TABLE_NAME,
						DialogsTable.Key.STATE,DialogsTable.Key.TARGET_ID, MessagesTable.KEY_STATUS, MessagesTable.KEY_DIALOG_ID,
						DialogsTable.Key.DIALOG_ID, MessagesTable.KEY_SERVER_SEQUENCE, MessagesTable.KEY_ORIGINATOR_ID);

				//select m.serverSequence, c.conversationId from messages m, conversations c where c.messageState>? and c.targetId=? and m.status =? and m.convID=c.conversationId and m.serverSequence != '-1' and m.originatorId != ?
				cursor = getDB().rawQuery(queryString, ConversationState.LOCKED.ordinal() , targetId, MessagingChatMessage.MessageState.RECEIVED.ordinal(), originatorId);
			} else if (!TextUtils.isEmpty(brandId)) {
				LPLog.INSTANCE.d(TAG, "Get all unread messages for brand " + brandId );

				String queryString = String.format("select m.%s, c.%s, c.%s  from %s m, %s c where c.%s=? and c.%s>? and m.%s =? and m.%s=c.%s and m.%s >= '0' and m.%s != ?",
						MessagesTable.KEY_SERVER_SEQUENCE, DialogsTable.Key.DIALOG_ID, DialogsTable.Key.CONVERSATION_ID, MessagesTable.MESSAGES_TABLE, DialogsTable.TABLE_NAME,
						DialogsTable.Key.BRAND_ID, DialogsTable.Key.STATE, MessagesTable.KEY_STATUS,
						MessagesTable.KEY_DIALOG_ID, DialogsTable.Key.DIALOG_ID, MessagesTable.KEY_SERVER_SEQUENCE, MessagesTable.KEY_ORIGINATOR_ID);

				//select m.serverSequence, c.conversationId from messages m, conversations c where c.brandId=? and m.status =? and m.convID=c.conversationId and m.serverSequence != '-1' and m.originatorId != ?
				cursor = getDB().rawQuery(queryString, brandId, ConversationState.LOCKED.ordinal() , MessagingChatMessage.MessageState.RECEIVED.ordinal(), originatorId);
			}

			// If no open conversation found, check If there is a record with conversationState = CLOSED and messageState = received
			if (cursor !=null && !cursor.moveToFirst()) {
				String id = null;
				String queryString = null;
				String rawQuery = "select m.%s, c.%s, c.%s  from %s m, %s c where c.%s=? and c.%s=? and m.%s =? and m.%s=c.%s and m.%s >= '0' and m.%s != ?";
				if ((!TextUtils.isEmpty(targetId))) {
					id = targetId;
					queryString = String.format(rawQuery,
							MessagesTable.KEY_SERVER_SEQUENCE, DialogsTable.Key.DIALOG_ID, DialogsTable.Key.CONVERSATION_ID, MessagesTable.MESSAGES_TABLE, DialogsTable.TABLE_NAME,
							DialogsTable.Key.TARGET_ID, DialogsTable.Key.STATE, MessagesTable.KEY_STATUS,
							MessagesTable.KEY_DIALOG_ID, DialogsTable.Key.DIALOG_ID, MessagesTable.KEY_SERVER_SEQUENCE, MessagesTable.KEY_ORIGINATOR_ID);
				} else if (!TextUtils.isEmpty(brandId)) {
					id = brandId;
					queryString = String.format(rawQuery,
							MessagesTable.KEY_SERVER_SEQUENCE, DialogsTable.Key.DIALOG_ID, DialogsTable.Key.CONVERSATION_ID, MessagesTable.MESSAGES_TABLE, DialogsTable.TABLE_NAME,
							DialogsTable.Key.BRAND_ID, DialogsTable.Key.STATE, MessagesTable.KEY_STATUS,
							MessagesTable.KEY_DIALOG_ID, DialogsTable.Key.DIALOG_ID, MessagesTable.KEY_SERVER_SEQUENCE, MessagesTable.KEY_ORIGINATOR_ID);
				}
				cursor = getDB().rawQuery(queryString, id, ConversationState.CLOSE.ordinal() , MessagingChatMessage.MessageState.RECEIVED.ordinal(), originatorId);
			}

			HashMap<String, List<Integer>> sequenceMap = new HashMap<>();
			HashMap<String, String> dialogIdsAndConversationIds = new HashMap<>();

			if (cursor != null) {
				try {
					List<Integer> list;

					if (cursor.moveToFirst()) {
						do {
							String dialogId = cursor.getString(cursor.getColumnIndex(DialogsTable.Key.DIALOG_ID));
							String conversationId = cursor.getString(cursor.getColumnIndex(DialogsTable.Key.CONVERSATION_ID));
							dialogIdsAndConversationIds.put(dialogId, conversationId);

							list = sequenceMap.get(dialogId);
							if (list == null) {
								list = new ArrayList<>();
								sequenceMap.put(dialogId, list);
							}

							int sequence = cursor.getInt(cursor.getColumnIndex(MessagesTable.KEY_SERVER_SEQUENCE));

							list.add(sequence);

						} while (cursor.moveToNext());
					}


				} finally {
					cursor.close();
				}
			} else {
				// Nothing to do
				return;
			}

			try {
				for (String dialogId : sequenceMap.keySet()) {
					LPLog.INSTANCE.d(TAG, "Send a read ack to the server for dialog id " + dialogId + " on the following sequences: " + sequenceMap.get(dialogId));

					// We have the sequence list of the messages we need to ack, run the command for updating the server
					String conversationId = dialogIdsAndConversationIds.get(dialogId);
					String brand = (!TextUtils.isEmpty(targetId) ? targetId : brandId);

					// We send read acknowledgement to pusher first and If it succeeds, send ack to UMS.
					// TODO: See If we can remove sending read to pusher here as we now has a separate function sendReadAckToPusherIfRequired() to do so. This won't cause any problem though.
					new SendReadAcknowledgementCommand(mController, brand, dialogId, conversationId, new ICallback<String, Exception>() {

						@Override
						public void onSuccess(String id) {
							String conId = !TextUtils.isEmpty(id) ? id : conversationId;
							try {
								new DeliveryStatusUpdateCommand(mController.mAccountsController.getConnectionUrl(brandId), brand, dialogId, conId, sequenceMap.get(dialogId)).execute();
							} catch (Exception error) {
								LPLog.INSTANCE.e(TAG, ERR_0000007D, "sendReadAckOnMessages: Failed to send read ack to UMS", error);
							}
						}

						@Override
						public void onError(Exception exception) {
							LPLog.INSTANCE.e(TAG, ERR_0000007E, "sendReadAckOnMessages: Failed to send read ack to pusher. ", exception);
						}
					}).execute();
				}
			} catch (Exception error) {
				LPLog.INSTANCE.e(TAG, ERR_0000007F, "sendReadAckOnMessages: Error while sending read acknowledgement to servers", error);
			}
		});
	}

	/**
	 * Send read acknowledgement to pusher only If pusher has stored positive count for that dialogId.
	 * @param conversationId
	 * @param dialogId
	 * @param brandId
	 */
	private void sendReadAckToPusherIfRequired(String conversationId, String dialogId, String brandId) {
		try {
			new GetUnreadMessagesCountCommand(mController, brandId, null, null, new ICallback<Integer, Exception>() {
				@Override
				public void onSuccess(Integer count) {
					if (count > 0) {
						Map<String, Integer> countMap = GetUnreadMessagesCountCommand.getUnreadCountMapped();
						if (countMap == null || TextUtils.isEmpty(dialogId)) {
							return;
						}
						Integer idCount = countMap.get(dialogId);
						if (idCount != null && idCount > 0) {
							new SendReadAcknowledgementCommand(mController, brandId, dialogId, conversationId, new ICallback<String, Exception>() {
								@Override
								public void onSuccess(String value) {
									LPLog.INSTANCE.d(TAG, "sendReadAckToPusherIfRequired: Cleared Pusher unread count for dialogId: " + dialogId);
								}

								@Override
								public void onError(Exception exception) {
									LPLog.INSTANCE.e(TAG, ERR_00000080, "sendReadAckToPusherIfRequired: Failed to Clear Pusher unread count for dialogId: " + dialogId);
								}
							}).execute();
						}
					}
				}

				@Override
				public void onError(Exception exception) {
					LPLog.INSTANCE.e(TAG, ERR_00000081, "sendReadAckToPusherIfRequired: Failed to fetch Unread Count Mapping. ", exception);
				}
			}).fetchUnreadCountMapping();
		} catch (Exception error) {
			LPLog.INSTANCE.e(TAG, ERR_00000082, "sendReadAckToPusherIfRequired: Failed to send read acknowledgement to pusher. ", error);
		}
	}

	public void setDeliveryStatusUpdateCommand(Form form, DeliveryStatus deliveryStatus) {
		if (form == null) {
			LPLog.INSTANCE.w(TAG, "form not found!");
			return;
		}
		new DeliveryStatusUpdateCommand(mController.mAccountsController.getConnectionUrl(form.getSiteId()), form.getSiteId(), form.getDialogId(), form.getConversationId(), form.getSeqId(), deliveryStatus).execute();
	}

	private ContentValues getContentValuesForMessage(final MessagingChatMessage message) {
		ContentValues messageValues = getContentValuesForMessageUpdate(message);
		messageValues.put(MessagesTable.KEY_EVENT_ID, message.getEventId());
		return messageValues;
	}

	private ContentValues getContentValuesForMessageUpdate(final MessagingChatMessage message) {
		ContentValues messageValues = new ContentValues();
		messageValues.put(MessagesTable.KEY_SERVER_SEQUENCE, message.getServerSequence());
		messageValues.put(MessagesTable.KEY_DIALOG_ID, message.getDialogId());

		// Text is encrypted before saving to DB
		final EncryptionVersion messageEncryptionVersion = DBEncryptionService.Companion.getAppEncryptionVersion();
		messageValues.put(MessagesTable.KEY_ENCRYPTION_VERSION, messageEncryptionVersion.ordinal());

		String encryptedMessage = DBEncryptionHelper.encrypt(messageEncryptionVersion, message.getMessage());
		messageValues.put(MessagesTable.KEY_TEXT, encryptedMessage);
		messageValues.put(MessagesTable.KEY_CONTENT_TYPE, message.getContentType());
		messageValues.put(MessagesTable.KEY_MESSAGE_TYPE, message.getMessageType().ordinal());
		messageValues.put(MessagesTable.KEY_STATUS, message.getMessageState().ordinal());
		messageValues.put(MessagesTable.KEY_TIMESTAMP, message.getTimeStamp());
		messageValues.put(MessagesTable.KEY_ORIGINATOR_ID, message.getOriginatorId());
		messageValues.put(MessagesTable.KEY_METADATA, DBEncryptionHelper.encrypt(messageEncryptionVersion, message.getMetadata()));
		return messageValues;
	}


	static private MessagingChatMessage getSingleMessageFromCursor(Cursor cursor) {
		long msgId = cursor.getLong(cursor.getColumnIndex(MessagesTable.KEY_ID));
		EncryptionVersion encryptionVersion = EncryptionVersion.fromInt(cursor.getInt(cursor.getColumnIndex(MessagesTable.KEY_ENCRYPTION_VERSION)));

		MessagingChatMessage message = new MessagingChatMessage(
				cursor.getString(cursor.getColumnIndex(MessagesTable.KEY_ORIGINATOR_ID)),
				cursor.getString(cursor.getColumnIndex(MessagesTable.KEY_TEXT)),
				cursor.getLong(cursor.getColumnIndex(MessagesTable.KEY_TIMESTAMP)),
				cursor.getString(cursor.getColumnIndex(MessagesTable.KEY_DIALOG_ID)),
				cursor.getString(cursor.getColumnIndex(MessagesTable.KEY_EVENT_ID)),
				MessagingChatMessage.MessageType.values()[cursor.getInt(cursor.getColumnIndex(MessagesTable.KEY_MESSAGE_TYPE))],
				MessagingChatMessage.MessageState.values()[cursor.getInt(cursor.getColumnIndex(MessagesTable.KEY_STATUS))],
				cursor.getInt(cursor.getColumnIndex(MessagesTable.KEY_SERVER_SEQUENCE)),
				cursor.getString(cursor.getColumnIndex(MessagesTable.KEY_CONTENT_TYPE)),
				encryptionVersion);

		message.setMessageId(msgId);
		String metadata = DBEncryptionHelper.decrypt(encryptionVersion, cursor.getString(cursor.getColumnIndex(MessagesTable.KEY_METADATA)));
		message.setMetadata(metadata);
		return message;
	}

	/**
	 * Update the Message Server Sequence & Message Status by event id
	 *
	 * @param eventId
	 * @param serverSequence
	 */
	public void updateOnMessageAck(final String eventId, final long serverSequence) {
		DataBaseExecutor.execute(() -> {
			{
				// Update the message server sequence
				final ContentValues contentValuesSequence = new ContentValues();
				contentValuesSequence.put(MessagesTable.KEY_SERVER_SEQUENCE, serverSequence);
				final String whereClauseSequence = MessagesTable.KEY_EVENT_ID + "=?";
				final String[] whereArgsSequence = {String.valueOf(eventId)};
				final int rowsAffected = getDB().update(contentValuesSequence, whereClauseSequence, whereArgsSequence);
				LPLog.INSTANCE.d(TAG, "Update msg server seq query. Rows affected=" + rowsAffected + " Seq=" + serverSequence);
			}

			{
				// Update the message status if relevant
				final ContentValues contentValuesStatus = new ContentValues();
				contentValuesStatus.put(MessagesTable.KEY_STATUS, MessagingChatMessage.MessageState.SENT.ordinal());
				final String whereClauseStatus = MessagesTable.KEY_EVENT_ID + "=? AND (" + MessagesTable.KEY_STATUS + "=? OR " + MessagesTable.KEY_STATUS + "=?)";
				final String[] whereArgsStatus = {String.valueOf(eventId), String.valueOf(MessagingChatMessage.MessageState.PENDING.ordinal()), String.valueOf(MessagingChatMessage.MessageState.ERROR.ordinal())}; // Update to SENT but only if current status is PENDING or ERROR (avoid lowering the status if messages are not arriving from the server in order)
				final int rowsAffected = getDB().update(contentValuesStatus, whereClauseStatus, whereArgsStatus);
				LPLog.INSTANCE.d(TAG, "Update msg status to SENT. Rows affected=" + rowsAffected);
			}

			//update list listener
			updateMessageByEventId(eventId);
		});
	}

	public DataBaseCommand<Void> loadExistingWelcomeMessage(MessagingChatMessage message) {
		return new DataBaseCommand<>(() -> {
			try (Cursor cursor = getDB().query(null, null, null, null, null, null)) {
				cursor.moveToLast();
				long rowId = cursor.getInt(cursor.getColumnIndex(MessagesTable.KEY_ID));
				mMessagesListener.onNewMessage(createFullMessageRow(rowId, message, -1));
			} catch (Exception e) {
				LPLog.INSTANCE.e(TAG, ERR_00000083, "Exception loading Welcome Message", e);
			}
			return null;
		});
	}

	public DataBaseCommand<Boolean> isLastMessageWelcomeMessage() {
		return new DataBaseCommand<>(() -> {
			boolean isLastMessageWelcomeMessage = false;
			try (Cursor cursor = getDB().query(null, null, null, null, null, null)) {
				cursor.moveToLast();
				if (cursor.getCount() == 0) {
					return false;
				}
				isLastMessageWelcomeMessage = cursor.getInt(cursor.getColumnIndex(MessagesTable.KEY_SERVER_SEQUENCE)) == WELCOME_MSG_SEQUENCE_NUMBER;
			} catch (Exception e) {
				LPLog.INSTANCE.e(TAG, ERR_00000084, "Exception checking if isLastMessageWelcomeMessage", e);
			}
			return isLastMessageWelcomeMessage;
		});
	}

	public DataBaseCommand<Void> updateLastWelcomeMessage(MessagingChatMessage message) {
		return new DataBaseCommand<>(() -> {
			try (Cursor cursor = getDB().query(null, null, null, null, null, null)) {
				cursor.moveToLast();
				int id = cursor.getInt(cursor.getColumnIndex(MessagesTable.KEY_ID));
				String whereString = MessagesTable.KEY_ID + "=?";
				String[] whereArgs = {String.valueOf(id)};
				getDB().update(getContentValuesForMessageUpdate(message), whereString, whereArgs);
			} catch (Exception e) {
				LPLog.INSTANCE.e(TAG, ERR_00000085, "Exception updating last welcome message", e);
			}
			return null;
		});
	}

	/**
	 * Remove welcome message if it's last message and not first message in database.
	 */
	public DataBaseCommand<Void> removeLastWelcomeMessage() {
		return new DataBaseCommand<>(() -> {
			try (Cursor cursor = getDB().query(null, null, null, null, null, null)) {
				cursor.moveToFirst();
				int firstItemId = cursor.getInt(cursor.getColumnIndex(MessagesTable.KEY_ID));
				cursor.moveToLast();
				int lastItemId = cursor.getInt(cursor.getColumnIndex(MessagesTable.KEY_ID));
				if (firstItemId != lastItemId) {
					String whereString = MessagesTable.KEY_ID + "=?";
					String[] whereArgs = {String.valueOf(lastItemId)};
					getDB().removeAll(whereString, whereArgs);
				}
			} catch (Exception e) {
				LPLog.INSTANCE.e(TAG, ERR_00000086, "Exception removing last welcome message", e);
			}
			return null;
		});
	}

	/**
	 * Update messages serverId for messages that are waiting for create dialog response
	 *
	 * @param serverDialogId      - new dialog ID
	 * @return list of the updated messages
	 */
	public DataBaseCommand<Void> updateMessagesDialogServerID(final String serverDialogId) {
		return new DataBaseCommand<>(() -> {
			ContentValues contentValues = new ContentValues();
			contentValues.put(MessagesTable.KEY_DIALOG_ID, serverDialogId);
			int updatedRows = getDB().update(contentValues, MessagesTable.KEY_DIALOG_ID + "=?", new String[]{Dialog.TEMP_DIALOG_ID});

			LPLog.INSTANCE.d(TAG, "updateMessagesConversationServerID , updatedRows = " + updatedRows);
			//update list listeners
			updateAllMessagesForDialog(serverDialogId);
			return null;
		});

	}

	/**
	 * Update messages serverId  and timestamp for message by row id
	 *
	 * @param messageRowId   - specific row to update
	 * @param dialogId - new dialog ID
	 * @return list of the updated messages
	 */
	public DataBaseCommand<Void> updateMessageDialogServerIdAndTime(final long messageRowId, final String dialogId) {

		return new DataBaseCommand<>(() -> {
			ContentValues contentValues = new ContentValues();
			contentValues.put(MessagesTable.KEY_DIALOG_ID, dialogId);
			int updatedRows = getDB().update(contentValues, MessagesTable.KEY_ID + "=? ", new String[]{String.valueOf(messageRowId)});

			LPLog.INSTANCE.d(TAG, "updateMessageDialogServerIdAndTime , rowId to update = " + messageRowId + ", updated = " + updatedRows);
			//update list listeners
			updateMessageByRowIdOnDbThread(messageRowId);
			return null;
		});
	}

	/**
	 * Update messages state for message by row id
	 *
	 * @param messageRowId - specific row to update
	 * @param state        - new state to set
	 * @return list of the updated messages
	 */
	public void updateMessageState(final long messageRowId, final MessagingChatMessage.MessageState state) {
		DataBaseExecutor.execute(() -> {
			ContentValues contentValues = new ContentValues();
			contentValues.put(MessagesTable.KEY_STATUS, state.ordinal());
			int updatedRows = getDB().update(contentValues, MessagesTable.KEY_ID + "=? ", new String[]{String.valueOf(messageRowId)});

			LPLog.INSTANCE.d(TAG, "updateMessageState , rowId to update = " + messageRowId + ", updated = " + updatedRows);
			//update list listeners
			updateMessageByRowIdOnDbThread(messageRowId);
		});
	}

	/**
	 * Update message related to file when file updated
	 *
	 * @param messageRowId - specific row to update
	 */
	public void updateMessageFileChanged(final long messageRowId) {
		if (messageRowId < 0) {
			LPLog.INSTANCE.w(TAG, "updateMessageFileChanged cannot be lower than zero! " + messageRowId);
			return;
		}
		DataBaseExecutor.execute(() -> updateMessageByRowIdOnDbThread(messageRowId));
	}

	/***************************
	 * UPDATE UI LISTENER
	 ******************************/

	private MessagesListener getMessagesListener() {
		if (mMessagesListener != null){
			return mMessagesListener;
		}
		return mNullMessagesListener;
	}

	//region AmsMessagesLoaderProvider
    /**
     * Adding MessagesListener to get updates for changes in messages
     *
     * @param messagesListener     listener to notify on updates
     * @param messagesSortedByType way to sort messages by brand or target id
     * @param typeValue            value for the selected messagesSortedByType - the target id or brand id
     */
    @Override
    public void addOnUpdateListener(MessagesListener messagesListener, final MessagesSortedBy messagesSortedByType, final String typeValue) {
        mMessagesListener = messagesListener;
        DataBaseExecutor.execute(() -> {
			// Query the DB
			ArrayList<FullMessageRow> searchedMessageList = loadMessagesOnDbThread(messagesSortedByType, typeValue, 100, -1, -1);
			getMessagesListener().initMessages(searchedMessageList);
		});

    }

    @Override
    public void removeOnUpdateListener() {
        mMessagesListener = null;
    }

    @Override
    public boolean hasListener() {
        return mMessagesListener != null;
    }

	/**
	 * Loading messages by specific rules:
	 *
	 * @param messagesSortedByType way to sort messages by brand or target id
	 * @param typeValue            value for the selected messagesSortedByType - the target id or brand id
	 * @param limit                max num of messages required.  0 or negative value for no limit
	 * @param olderThanTimestamp   negative value if there is no need to use this value
	 * @param newerThanTimestamp   negative value if there is no need to use this value
	 * @return
	 */
	@Override
	public DataBaseCommand<ArrayList<FullMessageRow>> loadMessages(final MessagesSortedBy messagesSortedByType, final String typeValue, final int limit, final long olderThanTimestamp, final long newerThanTimestamp) {
		return new DataBaseCommand<>(() -> {
			// Query the DB
			return loadMessagesOnDbThread(messagesSortedByType, typeValue, limit, olderThanTimestamp, newerThanTimestamp);
		});

	}

	@Nullable
	private ArrayList<FullMessageRow> loadMessagesOnDbThread(MessagesSortedBy messagesSortedByType, String typeValue, int limit, long olderThanTimestamp, long newerThanTimestamp) {
		Cursor cursor = null;
		switch (messagesSortedByType) {
			case TargetId:
				cursor = messagesByTarget(typeValue, limit, olderThanTimestamp, newerThanTimestamp);
				break;
			case ConversationId:
				cursor = messagesByConversationID(typeValue, limit);
				break;
		}

		ArrayList<FullMessageRow> searchedMessageList = null;
		if (cursor != null) {
			try {
				searchedMessageList = new ArrayList<>(cursor.getCount());
				if (cursor.moveToFirst()) {
					do {
						searchedMessageList.add(new FullMessageRow(cursor));
					} while (cursor.moveToNext());
				}
			} finally {
				cursor.close();
			}
		}
		return searchedMessageList;
	}

	@Override
	public String getMyUserId(String targetId) {
		return mController.getOriginatorId(targetId);
	}

	@Override
	public MessagingUserProfile loadMessagingUserProfile(String originatorId) {
		return mController.amsUsers.getUserById(originatorId).executeSynchronously();
	}
	//endregion

	@NonNull
	private FullMessageRow createFullMessageRow(long messageRowId, @NonNull MessagingChatMessage message, long fileRowId) {
		MessagingUserProfile messagingUserProfile = mController.amsUsers.getUserById(message.getOriginatorId()).executeSynchronously();
		String avatarUrl = messagingUserProfile == null ? "" : messagingUserProfile.getAvatarUrl();
		String agentNickname = messagingUserProfile == null ? "" : messagingUserProfile.getNickname();
		FileMessage fileMessage;
		if (fileRowId != -1) {
			fileMessage = mController.amsFiles.getFileByFileRowIdOnDbThread(fileRowId);
		} else {
			fileMessage = mController.amsFiles.getFileByMessageRowId(messageRowId);
		}
		FullMessageRow fullMessageRow = new FullMessageRow(message, avatarUrl, fileMessage);
		fullMessageRow.setAgentNickName(agentNickname);
		return fullMessageRow;
	}

	/**
	 * Clear all closed conversation messages of the given targetId
	 *
	 * @param targetId
	 * @return the number of messages removed
	 */
	public DataBaseCommand<Integer> clearMessagesOfClosedConversations(final String targetId) {

		return new DataBaseCommand<>(() -> {
			//"where messages._id in (select m._id from messages m, dialogs d where d.targetId='21337028' and d.state=0 and d.dialog_id=m.dialogId or m.dialogId=KEY_WELCOME_DIALOG_ID)"
			String whereString = MessagesTable.KEY_ID + " in (select m." + MessagesTable.KEY_ID + " from " + MessagesTable.MESSAGES_TABLE + " m, "
					+ DialogsTable.TABLE_NAME + " d where d." + DialogsTable.Key.TARGET_ID + "=? and d." + DialogsTable.Key.STATE
					+ "=? and d." + DialogsTable.Key.DIALOG_ID+ "=m." + MessagesTable.KEY_DIALOG_ID + " or m." + MessagesTable.KEY_DIALOG_ID + "=?)";
			String[] whereArgs = {targetId, String.valueOf(DialogState.CLOSE.ordinal()), AmsDialogs.KEY_WELCOME_DIALOG_ID}; // Closed conversations for targetId


			int removed = getDB().removeAll(whereString, whereArgs);
			LPLog.INSTANCE.d(TAG, "clearMessagesOfClosedConversations: removed: " + removed + " where: " + whereString + ", whereArgs: " + Arrays.toString(whereArgs));

			//update list listeners
			getMessagesListener().removeAllClosedConversations(targetId);
			return removed;
		});
	}

	/**
	 * Clear all closed conversation messages of the given targetId. This will be triggered when upgrade the SDK version or background app for unauth account.
	 *
	 * @param targetId
	 * @return the number of messages removed
	 */
	public DataBaseCommand<Integer> clearAllMessages(final String targetId) {

		return new DataBaseCommand<>(() -> {

			int removed = getDB().removeAll(null, null);
			LPLog.INSTANCE.d(TAG, "clearAllMessages from messages table");

			//update list listeners
			getMessagesListener().clearAllMessages(targetId);
			return removed;
		});
	}

    /**
     * RUN ONLY ON DB THREAD!!
     *
	 * @param firstNotification
	 * @param dialogId
	 * @param firstSequence
	 * @param lastSequence
	 */
    private void updateMessages(boolean firstNotification, String dialogId, int firstSequence, int lastSequence) {
        long firstTimestamp = getTimestampMessage(dialogId, firstSequence);
        long lastTimestamp = getTimestampMessage(dialogId, lastSequence);
        if (firstNotification){
			LPLog.INSTANCE.d(TAG, "updateMessages first notification event. onQueryMessagesResult ");
	        getMessagesListener().onQueryMessagesResult(firstTimestamp, lastTimestamp);
		}else{
			LPLog.INSTANCE.d(TAG, "updateMessages NOT first notification event. onUpdateMessages ");
			getMessagesListener().onUpdateMessages(firstTimestamp, lastTimestamp);
		}
    }

    /**
     * RUN ONLY ON DB THREAD!!
     *
     * @param eventId
     */
    private void updateMessageByEventId(String eventId) {
        Cursor cursor = getDB().query(null, MessagesTable.KEY_EVENT_ID + " = ?", new String[]{eventId}, null, null, null);
        if (cursor != null) {
            try {
                if (cursor.moveToFirst()) {
                    MessagingChatMessage message = getSingleMessageFromCursor(cursor);
                    int rowId = cursor.getInt(cursor.getColumnIndex(MessagesTable.KEY_ID));
                    getMessagesListener().onUpdateMessage(createFullMessageRow(rowId, message, -1));
                }
            } finally {
                cursor.close();
            }
        }

	}

	/**
	 * RUN ONLY ON DB THREAD!!
	 *
	 * @param dialogId
	 * @param sequence
	 */
	private String getEventIdForMessage(String dialogId, int sequence) {
		String eventId = null;
		Cursor cursor = getDB().rawQuery("SELECT " + MessagesTable.KEY_EVENT_ID + " FROM " + MessagesTable.MESSAGES_TABLE + " WHERE " + MessagesTable.KEY_DIALOG_ID
				+ " =? AND " + MessagesTable.KEY_SERVER_SEQUENCE + " =? ", dialogId, sequence);

		if (cursor != null) {
			try {
				if (cursor.moveToFirst()) {
					eventId = cursor.getString(cursor.getColumnIndex(MessagesTable.KEY_EVENT_ID));
				}
			} finally {
				cursor.close();
			}
		}
		return eventId;
	}

	public DataBaseCommand<MessagingChatMessage> getMessageByEventId(final String eventId) {
		if (TextUtils.isEmpty(eventId)) {
			LPLog.INSTANCE.w(TAG, "getMessageByEventId - eventId is empty");
			return null;
		}
		return new DataBaseCommand<>(() -> {
			try (Cursor cursor = getDB().query(null, MessagesTable.KEY_EVENT_ID + " = ?", new String[]{eventId}, null, null, null)) {
				if (cursor != null) {
					return getSingleMessageFromCursor(cursor);
				}
			}
			return null;
		});
	}

	/**
	 * @return If resolve message is already exists for the dialog.
	 */
	public DataBaseCommand<Boolean> isResolveMessageForDialogAdded(final String dialogId) {
		if (TextUtils.isEmpty(dialogId)) {
			LPLog.INSTANCE.w(TAG, "getResolveMessageForDialog - dialogId is empty");
			return null;
		}
		return new DataBaseCommand<>(() -> {
			// SELECT dialogId FROM messages WHERE dialogId={dialogId} AND serverSequence=-2
			String queryString = "SELECT " + MessagesTable.KEY_DIALOG_ID + " FROM " + MessagesTable.MESSAGES_TABLE + " WHERE " + MessagesTable.KEY_DIALOG_ID
					+ "=? AND " + MessagesTable.KEY_SERVER_SEQUENCE + "=?";
			try (Cursor cursor = getDB().rawQuery(queryString, dialogId, AmsMessages.RESOLVE_MSG_SEQUENCE_NUMBER)) {
				if (cursor != null) {
					cursor.moveToFirst();
					String dbDialogId = cursor.getString(cursor.getColumnIndex(MessagesTable.KEY_DIALOG_ID));
					return dbDialogId.equals(dialogId);
				}
			} catch (Exception e) {
				LPLog.INSTANCE.w(TAG, e.getMessage(), e);
			}
			return false;
		});
	}

	public DataBaseCommand<Long> getRowIdByEventId(final String eventId) {
		if (TextUtils.isEmpty(eventId)) {
			LPLog.INSTANCE.w(TAG, "getRowIdByEventId - eventId is empty");
			return null;
		}
		return new DataBaseCommand<>(() -> {
			try (Cursor cursor = getDB().query(new String[]{MessagesTable.KEY_ID}, MessagesTable.KEY_EVENT_ID + " = ?", new String[]{eventId}, null, null, null)) {
				if (cursor != null) {
					return cursor.getLong(cursor.getColumnIndex(MessagesTable.KEY_ID));
				}
			} catch (Exception e) {
				LPLog.INSTANCE.e(TAG, ERR_00000087, "Exception while getting a rowId by eventId", e);
			}
			return -1L;
		});
	}

    /**
     * RUN ONLY ON DB THREAD!!
     *
     * @param messageRowId
     */
    private void updateMessageByRowIdOnDbThread(long messageRowId) {
        MessagingChatMessage message = getMessageByRowIdOnDbThread(messageRowId);
        if (message != null){
			getMessagesListener().onUpdateMessage(createFullMessageRow(messageRowId, message, -1));
		} else {
			LPLog.INSTANCE.e(TAG, ERR_00000088, "updateMessageByRowIdOnDbThread - message does not exist");
		}
    }

	public DataBaseCommand<Void> updateFileMessageByRowId(final long messageRowId, final long fileRowId) {

		return new DataBaseCommand<>(() -> {
			MessagingChatMessage message = getMessageByRowIdOnDbThread(messageRowId);
			getMessagesListener().onUpdateMessage(createFullMessageRow(messageRowId, message, fileRowId));
			return null;
		});
    }

    @Nullable
	private MessagingChatMessage getMessageByRowIdOnDbThread(long messageRowId) {
		Cursor cursor = getDB().query(null, MessagesTable.KEY_ID + " = ?", new String[]{String.valueOf(messageRowId)}, null, null, null);
		if (cursor != null) {
			try {
				if (cursor.moveToFirst()) {
					return getSingleMessageFromCursor(cursor);
				}
			} finally {
				cursor.close();
			}
		}
		return null;
	}

    /**
     * RUN ONLY ON DB THREAD!!
     *
     * @param rowId
     * @param message
     */
    private void updateMessageByRowId(long rowId, MessagingChatMessage message) {
        getMessagesListener().onUpdateMessage(createFullMessageRow(rowId, message, -1));
    }

    /**
     * RUN ONLY ON DB THREAD!!
     *
     * @param dialogId
     */
    private void updateAllMessagesForDialog(String dialogId) {
        Cursor cursor = getDB().query(new String[]{"MIN(" + MessagesTable.KEY_TIMESTAMP + ")", "MAX(" + MessagesTable.KEY_TIMESTAMP + ")"},
                MessagesTable.KEY_DIALOG_ID + " = ?", new String[]{dialogId}, null, null, null);
        if (cursor != null) {
            try {
                if (cursor.moveToFirst()) {
                    long firstMessageTimestampForConversation = cursor.getLong(0);
                    long lastMessageTimestampForConversation = cursor.getLong(1);
                    getMessagesListener().onUpdateMessages(firstMessageTimestampForConversation, lastMessageTimestampForConversation);

				}
			} finally {
				cursor.close();
			}
		}
	}


	/**
	 * updating messages after fetch history.
	 */
	public void updateFetchHistoryEnded(final boolean updated) {
		DataBaseExecutor.execute(() -> {
			if (updated) {
				getMessagesListener().onHistoryFetched();
			} else {
				getMessagesListener().onHistoryFetchedFailed();
			}
		});
	}

	/**
	 * Updating messages when agent details relevant for conversation was updated.
	 *
	 * @param dialogId
	 */
	public void updateAgentDetailsUpdated(String dialogId) {
		updateAllMessagesForDialog(dialogId);
	}

    public void updateHandledExConversation(boolean emptyNotification) {
        getMessagesListener().onExConversationHandled(emptyNotification);
    }

	/**
	 * RUN ONLY ON DB THREAD!!
	 *
	 * @param dialogId
	 * @param sequence
	 */
	private long getTimestampMessage(String dialogId, int sequence) {
		long timestamp = 0;
		Cursor cursor = getDB().rawQuery("SELECT " + MessagesTable.KEY_TIMESTAMP + " FROM " + MessagesTable.MESSAGES_TABLE + " WHERE " + MessagesTable.KEY_DIALOG_ID
				+ " =? AND " + MessagesTable.KEY_SERVER_SEQUENCE + " =? ", dialogId, sequence);

		if (cursor != null) {
			try {
				if (cursor.moveToFirst()) {
					timestamp = cursor.getLong(cursor.getColumnIndex(MessagesTable.KEY_TIMESTAMP));
				}
			} finally {
				cursor.close();
			}
		}
		return timestamp;
	}

	@Override
	public void shutDown() {
		mMessageTimeoutQueue.removeAll();
	}

	@Override
	public void clear() {

	}

	public void removeAllMessages(String brandId) {
		getMessagesListener().clearAllMessages(brandId);
	}
}
