package com.liveperson.messaging.commands;

import com.liveperson.api.response.model.DeliveryStatusUpdateInfo;
import com.liveperson.api.response.types.DeliveryStatus;
import com.liveperson.infra.Command;
import com.liveperson.infra.ForegroundService;
import com.liveperson.infra.LPConversationsHistoryStateToDisplay;
import com.liveperson.infra.log.LPLog;
import com.liveperson.infra.network.socket.SocketManager;
import com.liveperson.messaging.MessagingFactory;
import com.liveperson.messaging.network.socket.requests.DeliveryStatusUpdateRequest;

import java.util.Arrays;
import java.util.List;

/**
 * Created by nirni on 11/23/15.
 * <p/>
 * Command for sending delivery status update to the server/agent (on the current conversation)
 * The command is registered on the request callback, and on success updates the state in the messages model
 */
public class DeliveryStatusUpdateCommand implements Command {
    private static final String TAG = "DeliveryStatusUpdateCommand";

    private String mTargetId;
    private String mSocketUrl;
    // The message sequence list to update the state on
    private List<Integer> mSequenceList;
    private String mDialogId;
    private String mConversationId;
    private DeliveryStatus mStatusToSend;
    private DeliveryStatusUpdateInfo mInfo;

    public DeliveryStatusUpdateCommand(String socketUrl, String targetId, String dialogId, String conversationId, List<Integer> sequenceList) {
        mSocketUrl = socketUrl;
        mTargetId = targetId;
        mDialogId = dialogId;
        mConversationId = conversationId;
        mSequenceList = sequenceList;
    }

	public DeliveryStatusUpdateCommand(String socketUrl, String targetId, String dialogId, String conversationId, int sequence , DeliveryStatus deliveryStatus) {
		this(socketUrl, targetId, dialogId, conversationId, Arrays.asList(sequence));
		mStatusToSend = deliveryStatus;
	}

	public DeliveryStatusUpdateCommand(String socketUrl, String targetId, String dialogId, String conversationId, int sequence , DeliveryStatus deliveryStatus, DeliveryStatusUpdateInfo info) {
		this(socketUrl, targetId, dialogId, conversationId, sequence, deliveryStatus);
		mInfo = info;
	}

	@Override
    public void execute() {

        if (MessagingFactory.getInstance().getController().getConversationViewParams().getHistoryConversationsStateToDisplay() == LPConversationsHistoryStateToDisplay.CLOSE) {
            return;
        }

        //if there was no demand to send a specific delivery status we calculate it according to the brand's foreground status.
        if (mStatusToSend == null){
            // If the current brand's conversation is in active (in the UI) we send a read ack on this message, otherwise we send only accept ack
            if (mTargetId != null && ForegroundService.getInstance().isTargetForeground(mTargetId)) {
                mStatusToSend = DeliveryStatus.READ;

            }else{
                mStatusToSend = DeliveryStatus.ACCEPT;
                // TODO: 3/8/18  LE-83854 [Android][SDK] - Fix endless AcceptStatus (accept) when receiving message in Background
                //need to fix it in Ams messages before sending accept status to check if we are in FG or BG and send only the relevant messages accordingly.
                return;
            }
        }

        // Create the request
        DeliveryStatusUpdateRequest statusUpdateRequest = new DeliveryStatusUpdateRequest(mSocketUrl, mDialogId, mConversationId, mStatusToSend, mSequenceList);

        statusUpdateRequest.setInfo(mInfo);

        LPLog.INSTANCE.d(TAG, String.format("Sending status update request to the agent. Sequence: %s, Status: %s", mSequenceList, mStatusToSend));

        // Send the request
        SocketManager.getInstance().send(statusUpdateRequest);
    }
}
