package com.liveperson.messaging.model;

import android.content.ContentValues;
import android.content.Context;
import android.database.Cursor;
import android.os.Bundle;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import android.text.TextUtils;
import android.text.format.DateUtils;

import com.liveperson.api.response.events.ContentEventNotification;
import com.liveperson.api.response.model.DialogData;
import com.liveperson.api.response.model.MultiDialog;
import com.liveperson.api.response.types.CSAT;
import com.liveperson.api.response.types.CloseReason;
import com.liveperson.api.response.types.DialogState;
import com.liveperson.api.response.types.DialogType;
import com.liveperson.api.response.types.TTRType;
import com.liveperson.infra.Clearable;
import com.liveperson.infra.ConversationViewParams;
import com.liveperson.infra.ICallback;
import com.liveperson.infra.Infra;
import com.liveperson.infra.LPConversationsHistoryStateToDisplay;
import com.liveperson.infra.configuration.Configuration;
import com.liveperson.infra.database.BaseDBRepository;
import com.liveperson.infra.database.DataBaseCommand;
import com.liveperson.infra.database.DataBaseExecutor;
import com.liveperson.infra.database.tables.DialogsTable;
import com.liveperson.infra.log.FlowTags;
import com.liveperson.infra.log.LPLog;
import com.liveperson.infra.messaging.R;
import com.liveperson.infra.sdkstatemachine.shutdown.ShutDown;
import com.liveperson.infra.utils.CollectionsUtil;
import com.liveperson.infra.utils.LocalBroadcast;
import com.liveperson.messaging.Messaging;
import com.liveperson.messaging.commands.tasks.FetchConversationManager;
import com.liveperson.infra.utils.ClockUtils;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Locale;
import java.util.Map;
import java.util.concurrent.TimeUnit;

import static com.liveperson.infra.errors.ErrorCode.ERR_00000089;
import static com.liveperson.infra.errors.ErrorCode.ERR_0000008A;
import static com.liveperson.infra.errors.ErrorCode.ERR_0000008B;
import static com.liveperson.infra.errors.ErrorCode.ERR_0000008C;
import static com.liveperson.infra.errors.ErrorCode.ERR_0000008D;

/**
 * Created by Perry on 22/04/2018.
 */
public class AmsDialogs extends BaseDBRepository implements ShutDown, Clearable {

	private static final String TAG = "AmsDialogs";

    public class BROADCASTS {
        public static final String UPDATE_DIALOG = "BROADCAST_UPDATE_DIALOG";
        public static final String UPDATE_CSAT_DIALOG = "BROADCAST_UPDATE_CSAT_DIALOG";
        public static final String UPDATE_DIALOG_CLOSED = "BROADCAST_UPDATE_DIALOG_CLOSED";
        public static final String UPDATE_NEW_DIALOG_MSG = "BROADCAST_UPDATE_NEW_DIALOG_MSG";
        public static final String UPDATE_UNREAD_MSG = "BROADCAST_UPDATE_DIALOG_UNREAD_MSG";
    }

    public static final String KEY_DIALOG_ID = "DIALOG_ID";
    public static final String KEY_DIALOG_TARGET_ID = "KEY_DIALOG_TARGET_ID";
    public static final String KEY_DIALOG_STATE = "KEY_DIALOG_STATE";
    public static final String KEY_DIALOG_SHOWED_CSAT = "KEY_DIALOG_SHOWED_CSAT";
    public static final String KEY_DIALOG_ASSIGNED_AGENT = "KEY_DIALOG_ASSIGNED_AGENT";

    public static final String KEY_WELCOME_DIALOG_ID = "KEY_WELCOME_DIALOG_ID";

    protected final Messaging mController;
    @Nullable
    private Dialog mActiveDialog;
    private Map<String, Dialog> mDialogsByDialogId = new HashMap<>();
    private HashSet<Integer> acceptedSequenceEvents = new HashSet<>();

    /**
     * Creates new AmsConversations
     *
     * @param controller
     */
    public AmsDialogs(Messaging controller) {
        super(DialogsTable.TABLE_NAME);
        mController = controller;
    }

    /**
     * Queries the DB / the cache and returns the active dialog if found, otherwise it returns null.
     * @param brandId
     * @return The active dialog or null of not found.
     */
    public DataBaseCommand<Dialog> queryActiveDialog(final String brandId) {
        return new DataBaseCommand<>(() -> {
            Dialog dialog = mActiveDialog;
            if (dialog != null && dialog.getBrandId().equals(brandId)) {
                if (dialog.getState() == DialogState.OPEN || dialog.getState() == DialogState.PENDING) {
                    return dialog;
                }
	            LPLog.INSTANCE.e(TAG, ERR_00000089, "queryActiveDialog: Impossible case! The active dialog is not open");
            }

            Cursor cursor = getDB().query(
                    null,
                    DialogsTable.Key.TARGET_ID + "=? and " + DialogsTable.Key.STATE + " in (?, ?) " +
                            "order by _id desc",
                    new String[]{brandId, String.valueOf(DialogState.OPEN.ordinal()), String.valueOf(DialogState.PENDING.ordinal())}, null, null, null);

            ArrayList<Dialog> parsedDialogs = readDialogs(cursor);
            if (parsedDialogs.size() == 0) {
	            LPLog.INSTANCE.d(TAG, "queryActiveDialog: Active dialog not found in DB");
                return null;
            } else if (parsedDialogs.size() > 1) {
	            LPLog.INSTANCE.w(TAG, "queryActiveDialog: More than 1 open dialogs found in DB");
            }

            mActiveDialog = parsedDialogs.get(0);
            return mActiveDialog;
        });
    }

    /**
     * Set active conversation
     *
     * @param dialog
     */
    private void addDialog(String brandId, Dialog dialog) {
        if (getDialogById(brandId) == null && dialog == null) {
            return;
        }
        addDialogToMaps(dialog);
        sendUpdateStateIntent(dialog);
    }

    /**
     * Load the last active dialog
     *
     * @param targetId
     */
    public void loadOpenDialogsForBrand(final String targetId) {
        DataBaseExecutor.execute(() -> {
            Cursor cursor = getDB().query(
                    null,
                    DialogsTable.Key.TARGET_ID + "=? and " + DialogsTable.Key.STATE + " in (?, ?) " +
                            "order by _id desc limit 1",
                    new String[]{targetId, String.valueOf(DialogState.OPEN.ordinal()), String.valueOf(DialogState.PENDING.ordinal())}, null, null, null);

            ArrayList<Dialog> dialogs = readDialogs(cursor);

            for (Dialog dialog : dialogs) {
	            LPLog.INSTANCE.d(TAG, "Setting current dialog for " + targetId + ". dialog id = " + dialog.getDialogId());
                addDialog(targetId, dialog);
            }
        });
    }

    private Dialog readDialog(Cursor cursor) {
        Dialog result = null;
        ArrayList<Dialog> dialogs = readDialogs(cursor);
        if (dialogs.size() == 1) {
            result = dialogs.get(0);
        }

        return result;
    }

    @NonNull
    private static ArrayList<Dialog> readDialogs(Cursor cursor) {
        ArrayList<Dialog> dialogs = new ArrayList<>();
        if (cursor != null) {
            try {
                if (cursor.moveToFirst()) {
                    do {
                        dialogs.add(new Dialog(cursor));
                    } while (cursor.moveToNext());
                }
            } catch (Exception e) {
                LPLog.INSTANCE.e(TAG, ERR_0000008A, "Exception while reading Dialogs.", e);
            } finally {
                cursor.close();
            }
        }
        return dialogs;
    }

    public static HashMap<String, Dialog> extractDialogsToMap(ConversationData data) {
        ArrayList<Dialog> dialogs = extractDialogs(data);
        HashMap<String, Dialog> dialogHashMap = new HashMap<>(dialogs.size());
        for (Dialog dialog : dialogs) {
            dialogHashMap.put(dialog.getDialogId(), dialog);
        }
        return dialogHashMap;
    }

    /**
     * Creates and returns an instance of dialogs list sorted by creation timestamp
     * @return A new list of dialogs sorted by start timestamp
     */
    public static ArrayList<Dialog> extractDialogs(ConversationData data) {
        ArrayList<Dialog> result = new ArrayList<>();
        if (data.dialogs == null || data.dialogs.length == 0) {
            Dialog dialog = new Dialog(data);
            result.add(dialog);
        } else {
            for (DialogData dialogData : data.dialogs) {
                result.add(new Dialog(dialogData, data));
            }
        }

        if (result.size() > 1) {
	        LPLog.INSTANCE.v(TAG, "Before sort: " + result);
            Collections.sort(result, (dialog1, dialog2) -> {
                long timeDiff = dialog1.getStartTimestamp() - dialog2.getStartTimestamp();
                if (timeDiff > 0) return 1;
                if (timeDiff < 0) return -1;
                return 0;
            });
            LPLog.INSTANCE.v(TAG, "After sort: " + result);
        }

        return result;
    }

    @Nullable
    public static Dialog getOpenDialog(ArrayList<Dialog> dialogs) {
        if (dialogs.size() == 0) {
            LPLog.INSTANCE.e(TAG, ERR_0000008B, "getOpenDialog: Got an empty dialogs list");
            return null;
        }

        ArrayList<Dialog> filtered = CollectionsUtil.filter(dialogs, Dialog::isOpenOrPending);

        if (filtered.size() == 0) {
            LPLog.INSTANCE.e(TAG, ERR_0000008C, "getOpenDialog: Missing open dialog in conversation");
            return null;
        } else if (filtered.size() > 1) {
            LPLog.INSTANCE.e(TAG, ERR_0000008D, "getOpenDialog: Too many simultaneous open dialogs found in conversation: ");
        }

        return filtered.get(0);
    }

    public void updateDialogs(ConversationData data) {
        ArrayList<Dialog> dialogs = extractDialogs(data);
        for (Dialog dialog : dialogs) {
            updateDialogDatabase(dialog, data);
        }
    }

    /**
     * Update the dialog record in database according to conversation's data
     *
     * @param updatedDialog An updated state of the dialog
     */
    private void updateDialogDatabase(Dialog updatedDialog, ConversationData data) {
        final Dialog dialog = getDialogById(updatedDialog.getDialogId());
        if (dialog == null) return;

        final ContentValues dialogValues = new ContentValues();

        DialogState dialogState = updatedDialog.getState();
        if (dialog.getState() != dialogState) {
            dialog.setState(dialogState);
            dialogValues.put(DialogsTable.Key.STATE, (dialogState != null ? dialogState.ordinal() : -1));
        }

        if (dialog.getConversationTTRType() != updatedDialog.getConversationTTRType()) {
            dialog.setConversationTTRType(updatedDialog.getConversationTTRType());
            switch (updatedDialog.getConversationTTRType()) {
                case URGENT:
                    mController.mEventsProxy.onConversationMarkedAsUrgent();
                    break;
                case NORMAL:
                    mController.mEventsProxy.onConversationMarkedAsNormal();
                    break;
            }
            dialogValues.put(DialogsTable.Key.TTR_TYPE, updatedDialog.getConversationTTRType().ordinal());
        }

        if (dialog.getRequestId() != data.requestId) {
            dialog.setRequestId(data.requestId);
            dialogValues.put(DialogsTable.Key.REQUEST_ID, data.requestId);
        }

        if (!TextUtils.equals(dialog.getConversationId(), dialog.getConversationId())) {
            dialog.setConversationId(updatedDialog.getConversationId());
            dialogValues.put(DialogsTable.Key.CONVERSATION_ID, updatedDialog.getConversationId());
        }

        String assignedAgentId = updatedDialog.getAssignedAgentId();
        if (!TextUtils.equals(dialog.getAssignedAgentId(), assignedAgentId)) {
            dialog.setAssignedAgentId(assignedAgentId);
            dialogValues.put(DialogsTable.Key.ASSIGNED_AGENT_ID, assignedAgentId);
        }

        if (dialog.getUnreadMessages() != updatedDialog.getUnreadMessages()) {
            dialog.setUnreadMessages(updatedDialog.getUnreadMessages());
            dialogValues.put(DialogsTable.Key.UNREAD_MESSAGES_COUNT, updatedDialog.getUnreadMessages());
            sendUpdateUnreadMsgIntent(dialog);
        }
        if (dialog.getStartTimestamp() != updatedDialog.getStartTimestamp()) {
            dialog.setStartTimestamp(updatedDialog.getStartTimestamp());
            dialogValues.put(DialogsTable.Key.START_TIMESTAMP, updatedDialog.getStartTimestamp());
        }

        //LastServerSequence should be updated only by updateLastServerSequence(int)

        if (dialogValues.size() > 0) {
            final String dialogId = dialog.getDialogId();
            dialogValues.put(DialogsTable.Key.DIALOG_ID, dialogId);
            dialogValues.put(DialogsTable.Key.DIALOG_TYPE, dialog.getDialogType().toString());

            DataBaseExecutor.execute(() -> {
                getDB().update(dialogValues, DialogsTable.Key.DIALOG_ID + "=?",
                        new String[]{String.valueOf(dialogId)});
                sendUpdateStateIntent(dialog);
	            LPLog.INSTANCE.d(TAG, FlowTags.DIALOGS, "Updated new dialog's data in DB. Dialog: " + dialogId);
            });
        }
    }

    public DataBaseCommand<Void> deleteDialogById() {
        return new DataBaseCommand<>(() -> {
            getDB().removeAll(DialogsTable.Key.DIALOG_ID + "=?", new String[]{Dialog.TEMP_DIALOG_ID});
	        LPLog.INSTANCE.d(TAG, "Finished removing dialog");
            return null;
        });
    }

    /**
     * Creates a Pending dialog
     *
     * @param targetId
     * @param brandId
     * @param requestId
     */
    public void createPendingDialog(final String targetId, final String brandId, final long requestId) {
        createDialog(targetId, brandId, DialogState.PENDING, requestId);
    }

    /**
     * Creates a Queued dialog
     *
     * @param targetId
     * @param brandId
     * @param requestId
     */
    public void createQueuedDialog(final String targetId, final String brandId, final long requestId) {
        createDialog(targetId, brandId, DialogState.QUEUED, requestId);
    }

    private void createDialog(final String targetId, final String brandId, final DialogState dialogState, final long requestId){

        final Dialog dialog = new Dialog(targetId, brandId);
        dialog.setConversationId(Conversation.TEMP_CONVERSATION_ID);
        dialog.setDialogId(Dialog.TEMP_DIALOG_ID);
        dialog.setChannelType(MultiDialog.ChannelType.MESSAGING);
        dialog.setDialogType(DialogType.MAIN);
        dialog.setState(dialogState);
        dialog.setConversationTTRType(TTRType.NORMAL);
        dialog.setRequestId(requestId);

        addDialogToMaps(dialog);

        final ContentValues dialogValues = new ContentValues();
        dialogValues.put(DialogsTable.Key.DIALOG_TYPE, dialog.getDialogType().toString());

        DataBaseExecutor.execute(() -> {
            dialogValues.put(DialogsTable.Key.DIALOG_ID, dialog.getDialogId());
            dialogValues.put(DialogsTable.Key.CONVERSATION_ID, dialog.getConversationId());
            dialogValues.put(DialogsTable.Key.BRAND_ID, dialog.getBrandId());
            dialogValues.put(DialogsTable.Key.TARGET_ID, dialog.getTargetId());
            dialogValues.put(DialogsTable.Key.STATE, dialog.getState().ordinal());
            dialogValues.put(DialogsTable.Key.TTR_TYPE, dialog.getConversationTTRType().ordinal());
            dialogValues.put(DialogsTable.Key.ASSIGNED_AGENT_ID, "");
            dialogValues.put(DialogsTable.Key.REQUEST_ID, dialog.getRequestId());
            dialogValues.put(DialogsTable.Key.LAST_SERVER_SEQUENCE, 0L);
            dialogValues.put(DialogsTable.Key.UNREAD_MESSAGES_COUNT, -1);
            dialogValues.put(DialogsTable.Key.START_TIMESTAMP, System.currentTimeMillis());
            getDB().insert(dialogValues);
	        LPLog.INSTANCE.d(TAG, "create New Pending Dialog - temp ID = " + dialog.getDialogId());
            sendUpdateStateIntent(dialog);
        });
    }

    /**
     * update the lastServerSequence param of the conversation by conversationId
     *
     * @param dialogId conversation server id
     * @param serverSequence
     */
    public void updateLastServerSequenceByDialogId(String dialogId, int serverSequence) {
        final Dialog dialog = getDialogById(dialogId);
        if (dialog != null && dialog.getLastServerSequence() != serverSequence) {
            boolean updated = false;

            long lastServerSequence = dialog.getLastServerSequence();
            if (lastServerSequence + 1 == serverSequence) {
                //it's the next sequence we are waiting for.
                dialog.setLastServerSequence(serverSequence);
                LPLog.INSTANCE.d(TAG, "Got an expected sequence!! last sequence = " + lastServerSequence + " (1) , new current sequence : " + serverSequence);
                updated = true;
            } else {
                LPLog.INSTANCE.d(TAG, "Failed! Got an unexpected sequence!! last sequence = " + lastServerSequence + " (1),  new unexpected sequence : " + serverSequence);
                acceptedSequenceEvents.add(serverSequence);
            }

            //check if we got next sequences waiting to be updated...
            updated = updateNextWaitingSequences(dialog, updated);

            if (updated) {
                LPLog.INSTANCE.d(TAG, "Running update last server sequence in db command. new last sequence = " + dialog.getLastServerSequence());
                updateSequenceInDb(dialog);
            }

        }
    }

    /**
     * Update current sequence if newer sequences are available.
     *
     * @param dialog
     * @param updated
     * @return
     */
    private boolean updateNextWaitingSequences(Dialog dialog, boolean updated) {
        int nextSequence = dialog.getLastServerSequence() + 1;
        HashSet<Integer> sequencesToRemove = new HashSet<>();
        while (acceptedSequenceEvents.contains(nextSequence)) {
            LPLog.INSTANCE.d(TAG, "Waited sequence " + nextSequence + " this is the new last server sequence!");
            sequencesToRemove.add(nextSequence);
            dialog.setLastServerSequence(nextSequence);
            nextSequence++;
            updated = true;
        }
        if (!sequencesToRemove.isEmpty()) {
            acceptedSequenceEvents.removeAll(sequencesToRemove);
        }
        return updated;
    }

    private void removeOldWaitingSequences(int serverSequence) {
        HashSet<Integer> sequencesToRemove = new HashSet<>();
        for (Integer sequence : acceptedSequenceEvents) {
            if (sequence <= serverSequence) {
                LPLog.INSTANCE.d(TAG, "Waited sequence " + sequence + " deleted after query messages!");
                sequencesToRemove.add(sequence);
            }
        }
        if (!sequencesToRemove.isEmpty()) {
            acceptedSequenceEvents.removeAll(sequencesToRemove);
        }
    }

    private void updateSequenceInDb(final Dialog dialog) {
        DataBaseExecutor.execute(() -> {
            ContentValues dialogValues = new ContentValues();
            int serverSequence = dialog.getLastServerSequence();
            String dialogId = dialog.getDialogId();

            dialogValues.put(DialogsTable.Key.LAST_SERVER_SEQUENCE, serverSequence);
            final String whereClause = DialogsTable.Key.DIALOG_ID + "=?";
            final String[] whereArgs = {dialogId};
            int updatedRows = getDB().update(dialogValues, whereClause, whereArgs);
	        LPLog.INSTANCE.d(TAG, "update sequence " + serverSequence + " for " + dialogId + " without gap. updated rows = " + updatedRows);
            sendUpdateStateIntent(dialog);
        });
    }

    private DataBaseCommand<Boolean> updateLastServerSequenceByDialogId(String dialogId, int serverSequence, int sequenceGap, boolean updateUI) {
        final Dialog dialog = getDialogById(dialogId);
        if (dialog != null) {
            long lastServerSequence = dialog.getLastServerSequence();
            if (lastServerSequence + sequenceGap == serverSequence) {
                //it's the next sequence we are waiting for.
                dialog.setLastServerSequence(serverSequence);
                LPLog.INSTANCE.d(TAG, "Got an expected sequence!! last sequence = " + lastServerSequence + " gap = " + sequenceGap + ", new current sequence : " + serverSequence);

                removeOldWaitingSequences(serverSequence);

            } else {
                LPLog.INSTANCE.d(TAG, "Failed! Got an unexpected sequence!! last sequence = " + lastServerSequence + " gap = " + sequenceGap + ", new unexpected sequence : " + serverSequence);
                return null;
            }
        }
        return getUpdateSequenceInDb(dialogId, serverSequence, sequenceGap, updateUI, dialog);
    }

    @NonNull
    private DataBaseCommand<Boolean> getUpdateSequenceInDb(final String dialogId, final int serverSequence, final int sequenceGap, final boolean updateUI, final Dialog dialog) {
        return new DataBaseCommand<>(() -> {
            ContentValues dialogValues = new ContentValues();//getContentValues(dialog);
            dialogValues.put(DialogsTable.Key.LAST_SERVER_SEQUENCE, serverSequence);

            final String whereClause = DialogsTable.Key.DIALOG_ID + "=? and " +
                    DialogsTable.Key.LAST_SERVER_SEQUENCE + "+" + sequenceGap + " = " +
                    serverSequence;
            final String[] whereArgs = {dialogId};
            int updatedRows = getDB().update(dialogValues, whereClause, whereArgs);
	        LPLog.INSTANCE.d(TAG, "update sequence " + serverSequence + " for " + dialogId + ". gap = " + sequenceGap + ". updated rows = " + updatedRows);
            if (dialog != null && updateUI) {
                sendUpdateStateIntent(dialog);
            }
            if (updatedRows > 0) {
                return true;
            } else {
                Cursor c = getDB().rawQuery("select " + DialogsTable.Key.LAST_SERVER_SEQUENCE + " from " + DialogsTable.TABLE_NAME
                        + " where " + DialogsTable.Key.DIALOG_ID + " =?", whereArgs);
                if (c.moveToFirst()) {
                    int lastServerSequenceInDb = c.getInt(c.getColumnIndex(DialogsTable.Key.LAST_SERVER_SEQUENCE));
	                LPLog.INSTANCE.d(TAG, "No rows updated! last sequence stored in db : " + lastServerSequenceInDb);
                }
                c.close();
            }
            return false;
        });
    }

    public DataBaseCommand<Dialog> updateCurrentDialogServerId(final ConversationData data) {
        final Dialog dialog = getDialogById(Dialog.TEMP_DIALOG_ID);
        removeTempDialog(data.targetId);
        dialog.setState(DialogState.parse(data.state));
        dialog.setStartTimestamp(data.startTs);
        dialog.setDialogId(DialogData.extractDialogId(data));
        dialog.setConversationId(data.conversationId);
        dialog.setConversationTTRType(data.conversationTTRType);
        dialog.setRequestId(data.requestId);
        dialog.setAssignedAgentId(data.getAssignedAgentId());
        addDialogToMaps(dialog);
        //LastServerSequence should be updated only by updateLastServerSequence(int)
        return new DataBaseCommand<>(() -> {
            getDB().insert(getContentValues(dialog));

            sendUpdateStateIntent(dialog);
	        LPLog.INSTANCE.d(TAG, "Finished updating dialog with server id");
            return dialog;
        });
    }

    public DataBaseCommand<Dialog> updateClosedDialog(ConversationData conversationData, boolean shouldUpdateUI) {
        ArrayList<Dialog> dialogs = extractDialogs(conversationData);
        Dialog dialog = dialogs.get(0); // Must exist!
        return updateClosedDialog(conversationData, dialog, shouldUpdateUI);
    }

    public DataBaseCommand<Dialog> updateClosedDialog(ConversationData conversationData, final Dialog dialog, boolean shouldUpdateUI) {
        LPLog.INSTANCE.d(TAG, FlowTags.DIALOGS, "updateClosedDialog: Closing dialog: " + dialog.getDialogId());
        dialog.setState(DialogState.CLOSE);
        final String brandId = dialog.getTargetId();

        shouldUpdateUI = shouldNotifyUIDialogClosed(dialog.getCloseReason(), brandId, dialog.getEndTimestamp(), shouldUpdateUI);
        final CSAT.CSAT_SHOW_STATUS csatShowStatus = AmsConversations.calculateShowedCsat(conversationData, shouldUpdateUI);

        Dialog dialogToClose = mDialogsByDialogId.get(dialog.getDialogId());
        if (dialogToClose != null) {
            //updating current dialog
            if (dialogToClose.getState() != DialogState.CLOSE) {
                LPLog.INSTANCE.d(TAG, FlowTags.DIALOGS, "Closing dialog " + dialogToClose.getDialogId() +
                        ", close reason:" + dialog.getCloseReason() + ", close ts:" + dialog.getEndTimestamp());
                dialogToClose.getTTRManager().cancelAll();
                dialogToClose.setCloseReason(dialog.getCloseReason());
                dialogToClose.setEndTimestamp(dialog.getEndTimestamp());
                dialogToClose.setAssignedAgentId(dialog.getAssignedAgentId());
                dialogToClose.setState(DialogState.CLOSE);
            }
            dialogToClose.setShowedCSAT(csatShowStatus);
        }

        if (mActiveDialog != null && mActiveDialog.getDialogId().equals(dialog.getDialogId())) {
            LPLog.INSTANCE.d(TAG, FlowTags.DIALOGS, "updateClosedDialog: Cleaning active dialog: " + mActiveDialog.getDialogId());
            mActiveDialog = null;
        }

        final boolean finalUpdateUI = shouldUpdateUI;

        return new DataBaseCommand<>(() -> {
            Cursor cursor = getDB().query(null, DialogsTable.Key.DIALOG_ID + "=?", new String[]{dialog.getDialogId()}, null, null, null);
            Dialog result = readDialog(cursor);

            if (result == null) {
	            LPLog.INSTANCE.i(TAG, "Old dialog " + dialog.getDialogId() + " does not exist in DB. creating new one closed conversation"
                        + ", close reason:" + dialog.getCloseReason() + ", close ts:" + dialog.getEndTimestamp());
                getDB().insert(getContentValues(dialog));
            } else {
                if (result.getState() != DialogState.CLOSE) {
	                LPLog.INSTANCE.d(TAG, "Closing current dialog.. ");
                    result.setState(DialogState.CLOSE);
                    result.setCloseReason(dialog.getCloseReason());
                    result.setEndTimestamp(dialog.getEndTimestamp());
                    result.setAssignedAgentId(dialog.getAssignedAgentId());
                    result.setShowedCSAT(csatShowStatus);
                    getDB().update(getContentValues(dialog), DialogsTable.Key.DIALOG_ID + "=?",
                            new String[]{String.valueOf(dialog.getDialogId())});

                    if (finalUpdateUI) {
                        sendUpdateCSATDialogIntent(dialog);
                    }
                    // Notify about closing dialog
                    sendDialogClosedIntent(dialog, dialog.getAssignedAgentId());
                } else {
                    if (dialog.isShowedCSAT() != csatShowStatus) {
                        dialog.setShowedCSAT(csatShowStatus);
                        getDB().update(getContentValues(dialog), DialogsTable.Key.DIALOG_ID + "=?",
                                new String[]{String.valueOf(dialog.getDialogId())});
                    }
                    // If dialog already exists & closed - return null
                    if (finalUpdateUI) {
                        sendUpdateCSATDialogIntent(dialog);
                    }
                    sendDialogClosedIntent(dialog, dialog.getAssignedAgentId());
                    return null;
                }

            }

            return dialog;
        });
    }

    private boolean shouldNotifyUIDialogClosed(CloseReason closeReason, String brandId, long closeTS, boolean updateUI) {
        if (updateUI){
            //if close reason is by System (auto close) no need to update ui.
            if (isAutoClose(closeReason)){
                LPLog.INSTANCE.d(TAG, "Updating closed dialog. Close Reason = System. do not update UI.");
                updateUI = false;
            }else{
                // Check expiry :
                // if we got to a decision we need to update ui -
                // we need to check if the closed dialog didn't expired.
                updateUI = checkExpiry(brandId, closeTS, updateUI);
            }
        }

        return updateUI;
    }

    private boolean isAutoClose(CloseReason closeReason) {
        return closeReason == CloseReason.TIMEOUT || closeReason == CloseReason.SYSTEM;
    }

    private boolean checkExpiry(String brandId, long closeTimestamp, boolean updateUI) {
        int expirationInMinutes = Configuration.getInteger(R.integer.csatSurveyExpirationInMinutes);
        if (expirationInMinutes != 0){ // 0 means no expiration
            final long clockDiff = mController.mConnectionController.getClockDiff(brandId);
            long now = System.currentTimeMillis();
            long endTime = clockDiff + closeTimestamp;
            long expirationInMillis = TimeUnit.MINUTES.toMillis(expirationInMinutes);
            if (now - endTime > expirationInMillis){
                LPLog.INSTANCE.d(TAG, "Closing dialog - time expired for CSAT. endTime = " + endTime + " expirationInMinutes = " + expirationInMinutes);
                updateUI = false;
            }

        }
        return updateUI;
    }

    private static void sendUpdateStateIntent(Dialog dialog) {
        Bundle extras = new Bundle();
        extras.putString(KEY_DIALOG_TARGET_ID, dialog.getTargetId());
        extras.putString(KEY_DIALOG_ID, dialog.getDialogId());
        extras.putInt(KEY_DIALOG_STATE, dialog.getState().ordinal());
        extras.putString(KEY_DIALOG_ASSIGNED_AGENT, dialog.getAssignedAgentId());
        LPLog.INSTANCE.d(TAG, "Sending dialog update with : " + extras);
        LocalBroadcast.sendBroadcast(BROADCASTS.UPDATE_DIALOG, extras);
    }

    private static void sendDialogClosedIntent(Dialog dialog, String assignedAgentId) {
        Bundle extras = new Bundle();
        extras.putString(KEY_DIALOG_TARGET_ID, dialog.getTargetId());
        extras.putString(KEY_DIALOG_ID, dialog.getDialogId());
        extras.putString(KEY_DIALOG_ASSIGNED_AGENT, assignedAgentId);
        LPLog.INSTANCE.d(TAG, "Sending dialog autoClosed update with : " + extras);
        LocalBroadcast.sendBroadcast(BROADCASTS.UPDATE_DIALOG_CLOSED, extras);
    }

    private static void sendUpdateCSATDialogIntent(Dialog dialog) {
        Bundle extras = new Bundle();
        extras.putString(KEY_DIALOG_TARGET_ID, dialog.getTargetId());
        extras.putString(KEY_DIALOG_ID, dialog.getDialogId());
        extras.putInt(KEY_DIALOG_STATE, dialog.getState().ordinal());
        extras.putString(KEY_DIALOG_ASSIGNED_AGENT, dialog.getAssignedAgentId());
        extras.putInt(KEY_DIALOG_SHOWED_CSAT, dialog.isShowedCSAT().getValue());
        LPLog.INSTANCE.d(TAG, "Sending dialog CSAT update with : " + extras);
        LocalBroadcast.sendBroadcast(BROADCASTS.UPDATE_CSAT_DIALOG, extras);
    }

    private static void sendUpdateNewDialogIntent(Dialog dialog) {
        Bundle extras = new Bundle();
        extras.putString(KEY_DIALOG_TARGET_ID, dialog.getTargetId());
        extras.putString(KEY_DIALOG_ID, dialog.getDialogId());
        LPLog.INSTANCE.d(TAG, "Sending dialog update with : " + extras);
        LocalBroadcast.sendBroadcast(BROADCASTS.UPDATE_NEW_DIALOG_MSG, extras);
    }

    private static void sendUpdateUnreadMsgIntent(Dialog dialog) {
        Bundle extras = new Bundle();
        extras.putString(KEY_DIALOG_TARGET_ID, dialog.getTargetId());
        LPLog.INSTANCE.d(TAG, "Sending dialog update with : " + extras);
        LocalBroadcast.sendBroadcast(BROADCASTS.UPDATE_UNREAD_MSG, extras);
    }

    private static ContentValues getContentValues(Dialog dialog) {
        ContentValues dialogValues = new ContentValues();

        dialogValues.put(DialogsTable.Key.CONVERSATION_ID, dialog.getConversationId());
        dialogValues.put(DialogsTable.Key.DIALOG_ID, dialog.getDialogId());
        dialogValues.put(DialogsTable.Key.DIALOG_TYPE, dialog.getDialogType().toString());
        dialogValues.put(DialogsTable.Key.BRAND_ID, dialog.getBrandId());
        dialogValues.put(DialogsTable.Key.TARGET_ID, dialog.getTargetId());
        dialogValues.put(DialogsTable.Key.STATE, (dialog.getState() != null ? dialog.getState().ordinal() : -1));
        dialogValues.put(DialogsTable.Key.TTR_TYPE, (dialog.getConversationTTRType() != null ?dialog.getConversationTTRType().ordinal() : -1));
        dialogValues.put(DialogsTable.Key.ASSIGNED_AGENT_ID, dialog.getAssignedAgentId());
        dialogValues.put(DialogsTable.Key.REQUEST_ID, dialog.getRequestId());
        dialogValues.put(DialogsTable.Key.CLOSE_REASON, (dialog.getCloseReason() != null ? dialog.getCloseReason().ordinal() : -1));
        dialogValues.put(DialogsTable.Key.START_TIMESTAMP, dialog.getStartTimestamp());
        dialogValues.put(DialogsTable.Key.END_TIMESTAMP, dialog.getEndTimestamp());
        dialogValues.put(DialogsTable.Key.LAST_SERVER_SEQUENCE, dialog.getLastServerSequence());
        dialogValues.put(DialogsTable.Key.CSAT_STATUS, dialog.isShowedCSAT().getValue());
        dialogValues.put(DialogsTable.Key.UNREAD_MESSAGES_COUNT, dialog.getUnreadMessages());

        return dialogValues;
    }

	public void resetEffectiveTTR(final String targetId) {
		final Dialog dialog = getDialogById(targetId);
		if (dialog != null) {
			dialog.getTTRManager().resetEffectiveTTR();
		}
	}

    public DataBaseCommand<Dialog> queryDialogById(final String dialogId) {
        return new DataBaseCommand<>(() -> {
            Cursor c = getDB().rawQuery("select * from " + DialogsTable.TABLE_NAME + " where " + DialogsTable.Key.DIALOG_ID + " = ?", dialogId);
            Dialog dialog = null;
            if (c != null) {
                // Cursor will be closed here:
                dialog = readDialog(c);
            }
            return dialog;
        });
    }

    private void addDialogToMaps(Dialog dialog) {
        if (!dialog.isClosed()) {
            setActiveDialog(dialog);
        }

        cacheDialog(dialog);

        // Store the dialog by dialog id
        LPLog.INSTANCE.d(TAG, "Putting dialog in dialogs ,ap. Dialog ID: " + dialog.getDialogId() + " targetId: " + dialog.getTargetId());
    }

    private void removeTempDialog(String targetId) {
	    LPLog.INSTANCE.d(TAG, "Removing temp dialog Id: " + Dialog.TEMP_DIALOG_ID + " target Id: " + targetId);

        Dialog dialog = mActiveDialog;
        if (dialog != null && dialog.getDialogId().equals(Dialog.TEMP_DIALOG_ID)) {
	        LPLog.INSTANCE.d(TAG, FlowTags.DIALOGS, "removeDialog: Cleaning active dialog: " + mActiveDialog.getDialogId());
            mActiveDialog = null;
        }
        mDialogsByDialogId.remove(Dialog.TEMP_DIALOG_ID);
    }

    /**
     * Remove all dialogs of the given targetId from the internal maps
     *
     * @param targetId
     */
    private void removeAllDialogsFromMaps(String targetId) {
        // Don't remove if there is an active dialog
        if (mActiveDialog != null && mActiveDialog.isOpenOrPending()) { // TODO: Clear history?
            LPLog.INSTANCE.w(TAG, "removeAllDialogsFromMaps: current dialog from brand " + targetId + " is active. Did not remove");
            return;
        }

        mDialogsByDialogId.clear();
        mActiveDialog = null;
        LPLog.INSTANCE.d(TAG, "removeAllDialogsFromMaps: Removing all dialogs");
    }

    /**
     * Clear all dialogs of the given targetId
     *
     * @param targetId
     * @return the number of messages removed
     */
    public DataBaseCommand<Integer> clearClosedDialogs(final String targetId) {

        // Remove dialog from internal maps
        removeAllDialogsFromMaps(targetId);

        // Remove dialog from DB
        return new DataBaseCommand<>(new DataBaseCommand.QueryCommand<Integer>() {

            //where targetId='xyz' and state=0;
            String whereString = DialogsTable.Key.TARGET_ID + "=? and " + DialogsTable.Key.STATE + "=?";
            String[] whereArgs = {targetId, String.valueOf(DialogState.CLOSE.ordinal())}; // Closed dialogs for targetId

            @Override
            public Integer query() {
                return getDB().removeAll(whereString, whereArgs);
            }
        });

    }

    public DataBaseCommand<Integer> clearAllDialogs(final String targetId) {

        // Remove dialog from internal maps
        removeAllDialogsFromMaps(targetId);

        // Remove dialog from DB
        return new DataBaseCommand<>(new DataBaseCommand.QueryCommand<Integer>() {

            //where targetId='xyz' and state=0;
            String whereString = DialogsTable.Key.TARGET_ID + "=?";
            String[] whereArgs = {targetId}; // Delete all dialogs for targetId regardless of dialog state

            @Override
            public Integer query() {
                return getDB().removeAll(whereString, whereArgs);
            }
        });

    }

    /**
     * Gets the current active dialog unless there's no active open / pending dialog.
     * @return The current active dialog as long as it's active (open / pending), otherwise null
     */
    @Nullable
    public Dialog getActiveDialog() {
        Dialog dialog = mActiveDialog;

        if (dialog != null && dialog.isClosed()) {
            dialog = null;
        }

        return dialog;
    }

    @Nullable
    public Dialog getDialogById(final String dialogId) {
        return mDialogsByDialogId.get(dialogId);
    }

    public void getDialogByIdAsync(final String dialogId, final ICallback<Dialog, Exception> callback) {

        Dialog dialog = mDialogsByDialogId.get(dialogId);

        if (dialog != null) {
            callback.onSuccess(dialog);
        } else {
            queryDialogById(dialogId).setPostQueryOnBackground(dialog1 -> {
                if (dialog1 != null) {
                    callback.onSuccess(dialog1);
                } else {
                    callback.onError(new Exception("Dialog not found"));
                }
            }).execute();
        }
    }

    public ArrayList<Dialog> getDialogsByConversationId(String conversationId) {
        ArrayList<Dialog> dialogs = new ArrayList<>();
        if (conversationId == null) return dialogs;

        for (Map.Entry<String, Dialog> dialogEntry : mDialogsByDialogId.entrySet()) {
            final Dialog dialogEntryValue = dialogEntry.getValue();
            final String dialogEntryConversationId = dialogEntryValue.getConversationId();
            if (conversationId.equals(dialogEntryConversationId)) {
                dialogs.add(dialogEntryValue);
            }
        }

        return dialogs;
    }

    public DataBaseCommand<Dialog> fetchPreviousDialog(final String targetId) {
        return new DataBaseCommand<>(() -> {
            Cursor c = getNextDialogToFetch(targetId);
            Dialog dialog = null;
            if (c != null && c.moveToFirst()) {
                // The cursor will closed inside readDialog(c)
                dialog = readDialog(c);
                cacheDialog(dialog);
                FetchConversationManager conversationManager = new FetchConversationManager(mController);
                Conversation conversation = mController.amsConversations.queryConversationById(dialog.getConversationId()).executeSynchronously();
                ArrayList<Dialog> dialogs = mController.amsDialogs.queryDialogsByConversationId(conversation.getConversationId()).executeSynchronously();
                conversationManager.fetchConversation(conversation, dialogs);
            }
            return dialog;
        });
    }

    private void cacheDialog(Dialog dialog) {
        mDialogsByDialogId.put(dialog.getDialogId(), dialog);
    }

    private Cursor getNextDialogToFetch(String targetId) {
        // Returns a record of one dialog for specific target id, when we know they have at least 1 message (by the time of the last message)
        // but we don't have those messages cause the server sequence equals -1
        return getDB().rawQuery(
                "select * from " + DialogsTable.TABLE_NAME +
                        " where " + DialogsTable.Key.TARGET_ID + " = ? and " +
                        DialogsTable.Key.LAST_SERVER_SEQUENCE + " = -1 " +
                        "order by " + DialogsTable.Key.START_TIMESTAMP + " DESC limit 1"
                , targetId);
    }

    /**
     * Increasing updating process (query messages/ agent details) that currently updating specific dialog.
     * If we'll be interrupted we can recover by looking on this field.
     *
     * @param dialogId
     */
    public void addUpdateRequestInProgress(final String dialogId) {
        final Dialog dialog = getDialogById(dialogId);

        if (dialog != null) {
            int mUpdateInProgress = dialog.getUpdateInProgress() + 1;
            LPLog.INSTANCE.d(TAG, "adding update request in progress for dialog: " + dialogId + ", requests in progress: " + mUpdateInProgress);
            dialog.setUpdateInProgress(mUpdateInProgress);
        }
        updateRequestsInProgress(dialogId, 1);
    }

    /**
     * Decreasing updating process (query messages / agent details) that currently updating specific dialog.
     * If we'll be interrupted we can recover by looking on this field.
     *
     * @param dialogId
     */
    public void removeUpdateRequestInProgress(final String dialogId) {
        final Dialog dialog = getDialogById(dialogId);

        if (dialog != null) {
            int mUpdateInProgress = dialog.getUpdateInProgress() - 1;
            LPLog.INSTANCE.d(TAG, "removing update request for dialog ID: " + dialogId + ", requests in progress: " + mUpdateInProgress);
            dialog.setUpdateInProgress(mUpdateInProgress);
        }

        updateRequestsInProgress(dialogId, -1);
    }

    /**
     * update the number of updating process in db
     *
     * @param dialogId
     * @param addValue       new value
     */
    private void updateRequestsInProgress(final String dialogId, final int addValue) {
        DataBaseExecutor.execute(() -> {
            try (Cursor c = getDB().query(new String[]{DialogsTable.Key.CONCURRENT_REQUESTS_COUNTER},
                    DialogsTable.Key.DIALOG_ID + " = ? ", new String[]{dialogId}, null, null, null)) {
                int updateInProgress = 0;
                if (c != null && c.moveToFirst()) {
                    updateInProgress = c.getInt(c.getColumnIndex(DialogsTable.Key.CONCURRENT_REQUESTS_COUNTER));
                }

	            LPLog.INSTANCE.d(TAG, "update request for dialog in DB: " + dialogId + ", requests in progress: " + updateInProgress + " added value = " + addValue);

                ContentValues dialogValues = new ContentValues();
                dialogValues.put(DialogsTable.Key.CONCURRENT_REQUESTS_COUNTER, updateInProgress + addValue);
                getDB().update(dialogValues, DialogsTable.Key.CONVERSATION_ID + "=?", new String[]{String.valueOf(dialogId)});
            }
        });
    }

	public void updateDialogState(final String dialogId, final DialogState state) {

		final Dialog dialog = getActiveDialog();

		if (dialog != null) {
			LPLog.INSTANCE.d(TAG, "update dialog state, new state = " + state);
			dialog.setState(state);
		}

        DataBaseExecutor.execute(() -> {
	        LPLog.INSTANCE.d(TAG, "update new state for dialog in DB: " + dialogId + ", state: " + state);
            ContentValues dialogValues = new ContentValues();
            dialogValues.put(DialogsTable.Key.STATE, state.ordinal());
            getDB().update(dialogValues, DialogsTable.Key.DIALOG_ID + "=?", new String[]{String.valueOf(dialogId)});
        });
	}

	@Override
    public void clear() {
		// Clear all TTR stored data
		for (Dialog dialog : mDialogsByDialogId.values()) {
			dialog.getTTRManager().clear();
		}

        // Clear accepted sequence events list
        acceptedSequenceEvents.clear();

        // Clearing all dialog maps
        mDialogsByDialogId.clear();
        mActiveDialog = null;
	}

    @Override
    public void shutDown() {
        // Removing all waiting ttr events
        for (Dialog dialog : mDialogsByDialogId.values()) {
            dialog.getTTRManager().shutDown();
        }
    }

    /**
     * Create new current dialog
     *
     * @param dialog The dialog to update
     */
    public void createNewCurrentDialog(final Dialog dialog) {
        DataBaseExecutor.execute(() -> {
            getDB().insert(getContentValues(dialog));
            LPLog.INSTANCE.i(TAG, "Created new current dialog with ID: " + dialog.getDialogId());

            sendUpdateStateIntent(dialog);
            sendUpdateNewDialogIntent(dialog);
        });
    }

    /**
     * Creating a dialog that will be saved only locally in device's DB.
     * This dialog will not be synced with the server's transcript.
     *
     * @param brandId
     * @param dummyConversationId
     * @param dummyDialogId
     * @param lastMessageSequence
     * @param startTime
     */
    public void createDummyDialogForWelcomeMessage(final String brandId, final String dummyConversationId, final String dummyDialogId, final long lastMessageSequence, final long startTime) {
        DataBaseExecutor.execute(() -> {
            ContentValues dialogValues = new ContentValues();
            dialogValues.put(DialogsTable.Key.BRAND_ID, brandId);
            dialogValues.put(DialogsTable.Key.TARGET_ID, brandId);
            dialogValues.put(DialogsTable.Key.DIALOG_ID, dummyDialogId);
            dialogValues.put(DialogsTable.Key.CONVERSATION_ID, dummyConversationId);
            dialogValues.put(DialogsTable.Key.STATE, DialogState.LOCKED.ordinal());
            dialogValues.put(DialogsTable.Key.TTR_TYPE, TTRType.NORMAL.ordinal());
            dialogValues.put(DialogsTable.Key.ASSIGNED_AGENT_ID, "");
            dialogValues.put(DialogsTable.Key.REQUEST_ID, -1);
            dialogValues.put(DialogsTable.Key.LAST_SERVER_SEQUENCE, lastMessageSequence);
            dialogValues.put(DialogsTable.Key.UNREAD_MESSAGES_COUNT, -1);
            dialogValues.put(DialogsTable.Key.START_TIMESTAMP, startTime);
            getDB().insert(dialogValues);
            LPLog.INSTANCE.d(TAG, "created dummy dialog for first message- startTime = " + startTime);
        });
    }

    public DataBaseCommand<ArrayList<Dialog>> queryOpenDialogsOfConversation(final String conversationId) {
        return new DataBaseCommand<>(() -> {
            // SQL command: "select * from dialogs where conversation_id = " + conversationId + " and state in (OPEN, PENDING)"
            Cursor cursor = getDB().query(
                    null,
                    DialogsTable.Key.CONVERSATION_ID + "=? and " + DialogsTable.Key.STATE + " in (?, ?) " +
                            "order by _id desc",
                    new String[]{conversationId, String.valueOf(DialogState.OPEN.ordinal()), String.valueOf(DialogState.PENDING.ordinal())}, null, null, null);

            // Cursor will be closed in readDialogs(...)
            return readDialogs(cursor);
        });
    }

    public DataBaseCommand<ArrayList<Dialog>> queryDialogsByConversationId(final String conversationId) {
        return new DataBaseCommand<>(() -> {
            Cursor c = getDB().rawQuery("select * from " + DialogsTable.TABLE_NAME + " where " + DialogsTable.Key.CONVERSATION_ID + " = ?", conversationId);
            ArrayList<Dialog> dialogs = null;
            if (c != null) {
                // Cursor will be closed in readDialogs(...)
                dialogs = readDialogs(c);
            }
            return dialogs != null ? dialogs : new ArrayList<>();
        });
    }

    /**
     * Check If database dialog table has welcome message dialog present
     * @return true/false
     */
    public DataBaseCommand<Boolean> hasWelcomeMessageDialog() {
        return new DataBaseCommand<>(() -> {
            try (Cursor cursor = getDB().rawQuery("select * from " + DialogsTable.TABLE_NAME + " where " + DialogsTable.Key.DIALOG_ID + " = ?", KEY_WELCOME_DIALOG_ID)) {
                if (cursor != null && cursor.moveToFirst()) {
                    return true;
                }
            }
            return false;
        });
    }

    public void setActiveDialog(Dialog activeDialog) {
        if (activeDialog == null) {
            return;
        }

        if (mActiveDialog != null) {
            if (!mActiveDialog.getDialogId().equals(activeDialog.getDialogId())) {
                LPLog.INSTANCE.d(TAG, FlowTags.DIALOGS, "Setting a new active dialog: " + activeDialog.getDialogId());
            }
        }

        mDialogsByDialogId.put(activeDialog.getDialogId(), activeDialog);

        mActiveDialog = activeDialog;
    }

    public void closeActiveDialog() {
        if (mActiveDialog != null) {
            Dialog activeDialog = mDialogsByDialogId.get(mActiveDialog.getDialogId());
            if (activeDialog != null) {
                activeDialog.setState(DialogState.CLOSE);
            }
        }

        mActiveDialog = null;
    }

    public void markPendingDialogsAsFailed(final String brandId) {
        final Dialog dialog = mActiveDialog;
        if (dialog != null && (dialog.getState() == DialogState.PENDING || dialog.getState() == DialogState.QUEUED)){
            dialog.setState(DialogState.CLOSE);
            dialog.setEndTimestamp(ClockUtils.getSyncedTimestamp());
        }
        DataBaseExecutor.execute(() -> {
            String where = DialogsTable.Key.STATE + " =? AND "+ DialogsTable.Key.BRAND_ID + " = ?";
            ContentValues contentValues = new ContentValues();
            // The value to be changed
            contentValues.put(DialogsTable.Key.STATE, DialogState.CLOSE.ordinal());

            int result = getDB().update(contentValues, where, new String[]{String.valueOf(DialogState.PENDING.ordinal()), brandId});
	        LPLog.INSTANCE.d(TAG, String.format(Locale.ENGLISH, "Updated %d pending dialog as Closed on DB", result));
        });
    }

    /**
     * Must run of DB Thread!
     * @param targetId
     * @return
     */
    public boolean areMoreMessagesAvailableToFetch(String targetId) {
        // LE-79838 [Android] History Control API :
        boolean shouldFetch = false;
        ConversationViewParams conversationViewParams = mController.getConversationViewParams();
        if (conversationViewParams.getHistoryConversationsStateToDisplay() == LPConversationsHistoryStateToDisplay.OPEN) {
            //since more conversation to load are closed, and we asked to show only open conversation, we return false.
            return shouldFetch;
        }
        // end of LE-79838

        Cursor c = getNextDialogToFetch(targetId);
        if (c != null) {
            if (c.getCount() == 1 && c.moveToFirst()) {
                // The cursor will closed inside readConversation(c)
                Dialog dialog = readDialog(c);
                if (dialog.getDialogId().equals(Dialog.TEMP_DIALOG_ID)) {
                    return false;
                }
                shouldFetch = true;

                if (conversationViewParams.isFilterOn()) {

                    long historyConversationsMaxDays = conversationViewParams.getHistoryConversationsMaxDays() * DateUtils.DAY_IN_MILLIS;

                    switch (conversationViewParams.getHistoryConversationMaxDaysType()) {
                        case endConversationDate:
                            if (dialog.getEndTimestamp() < (System.currentTimeMillis() - historyConversationsMaxDays)) {
                                shouldFetch = false;
                            }
                        case startConversationDate:
                            if (dialog.getStartTimestamp() < (System.currentTimeMillis() - historyConversationsMaxDays)) {
                                shouldFetch = false;
                            }
                    }
                }

            } else {
                c.close();
            }
        }


        return shouldFetch;
    }

    public void saveMessagesResult(@Nullable String dialogId, @NonNull final ArrayList<ContentEventNotification> messagesResult, final boolean firstNotification, final boolean shouldUpdateUI, @NonNull final ICallback<Dialog, Exception> callback) {
        Dialog currentDialog = getDialogById(dialogId);

        // If the received event is not on the current conversation an error is raised
        if (currentDialog == null) {
            queryDialogById(dialogId).setPostQueryOnBackground(dialog -> {
                if (dialog != null) {
                    saveMessages(dialog, messagesResult, firstNotification, shouldUpdateUI, callback);
                }
            }).execute();
        } else {
            saveMessages(currentDialog, messagesResult, firstNotification, shouldUpdateUI, callback);
        }

    }

    private void saveMessages(final Dialog dialog, final ArrayList<ContentEventNotification> messagesResult, final boolean firstNotification, final boolean shouldUpdateUI, final ICallback<Dialog, Exception> callback) {
        if (messagesResult.size() == 0) {
            LPLog.INSTANCE.d(TAG, "No messages in query response.");
            //updating last server sequence on conversation
            //empty conversation. need to update last sequence in DB to 0.
            int lastServerSequence = dialog.getLastServerSequence() == -1 ? 0 : dialog.getLastServerSequence();
            DataBaseCommand<Boolean> command = updateLastServerSequenceByDialogId(dialog.getDialogId(), lastServerSequence, 1, shouldUpdateUI);
            if (command != null) {
                command.execute();
            }
            callback.onError(new Exception("Empty results - no messages"));
            return;
        }

        //get last message from response for its sequence.
        int lastContentSequence = -1;
        for (int i = messagesResult.size() - 1; i >= 0; i--) {
            if (messagesResult.get(i).sequence > -1) {
                lastContentSequence = messagesResult.get(i).sequence;
                break;
            }

        }
        LPLog.INSTANCE.d(TAG, dialog.getDialogId() + " - Last sequence event received in query messages response: " + lastContentSequence);


        if (lastContentSequence == -1 || lastContentSequence < dialog.getLastServerSequence()) {
            LPLog.INSTANCE.d(TAG, dialog.getDialogId() + " - didn't receive any new sequence " + lastContentSequence);
            // We will be updating existing messages in database for existing conversation which may
            // have been created from UMS and now missing PII/masked data.
            if (dialog.isClosed()) {
                mController.amsMessages.updateMultipleMessages(messagesResult, dialog.getDialogId()).setPostQueryOnBackground(data -> {
                    LPLog.INSTANCE.d(TAG, "Finished updating " + messagesResult.size() + " messages for dialog id " + dialog.getDialogId());
                    callback.onSuccess(dialog);
                }).execute();
            } else {
                callback.onSuccess(dialog);
            }
            return;
        }

        //updating last server sequence on conversation
        DataBaseCommand<Boolean> command = updateLastServerSequenceByDialogId(dialog.getDialogId(),
                lastContentSequence,
                lastContentSequence - dialog.getLastServerSequence(),
                shouldUpdateUI);
        command.setPostQueryOnBackground(isEventExpected -> {
            //adding messages only if sequence are matching
            String originatorId = mController.getOriginatorId(dialog.getTargetId());
            long clockDiff = mController.mConnectionController.getClockDiff(dialog.getBrandId());

            mController.amsMessages.addMultipleMessages(messagesResult,
                    originatorId, dialog.getBrandId(), dialog.getTargetId(), dialog.getDialogId(), dialog.getConversationId(), clockDiff, firstNotification, shouldUpdateUI).setPostQueryOnBackground(data -> {
                        callback.onSuccess(dialog);
	                    LPLog.INSTANCE.d(TAG, "Finished saving " + messagesResult.size() + " messages for dialog id " + dialog.getDialogId());
                    }).execute();
        });
        command.execute();
    }

    public static boolean isUmsSupportingDialogs() {
        return UmsCompatibilityManager.isSupportingDialogs();
    }

    public static void setIsUmsSupportingDialogs(boolean isSupportingDialogs) {
        UmsCompatibilityManager.setIsSupportingDialogs(isSupportingDialogs);
    }

    private static class UmsCompatibilityManager {
        private static Boolean _isSupportingDialogs;

        private static boolean isSupportingDialogs() {
            if (_isSupportingDialogs == null) {
                _isSupportingDialogs = Infra.instance.getApplicationContext().getSharedPreferences(FileNames.TEMP_COMPATIBILITY_PERSISTENCE, Context.MODE_PRIVATE).getBoolean(Keys.IS_SUPPORTING_DIALOGS, false);
                if (_isSupportingDialogs) {
                    LPLog.INSTANCE.d(TAG, FlowTags.DIALOGS, "Communicating with a new UMS, multi-dialog is supported!");
                } else {
                    LPLog.INSTANCE.d(TAG, FlowTags.DIALOGS, "Communicating with a legacy UMS, multi-dialog is not supported!");
                }
            }

            return _isSupportingDialogs;
        }

        private static void setIsSupportingDialogs(Boolean isSupportingDialogs) {
            if (isSupportingDialogs() == isSupportingDialogs) return;

            if (isSupportingDialogs) {
                LPLog.INSTANCE.d(TAG, FlowTags.DIALOGS, "Communicating with UMS that has multi-dialog support!");
            } else {
                LPLog.INSTANCE.d(TAG, FlowTags.DIALOGS, "Communicating with UMS that HAS NO multi-dialog support!");
            }

            _isSupportingDialogs = isSupportingDialogs;
            Infra.instance.getApplicationContext().getSharedPreferences(FileNames.TEMP_COMPATIBILITY_PERSISTENCE, Context.MODE_PRIVATE).edit().putBoolean(Keys.IS_SUPPORTING_DIALOGS, isSupportingDialogs).apply();
        }

        private static class Keys {
            static final String IS_SUPPORTING_DIALOGS = "IS_SUPPORTING_DIALOGS";
        }

        private static class FileNames {
            static final String TEMP_COMPATIBILITY_PERSISTENCE = "LP_COMPATIBILITY";
        }
    }
}
