package com.liveperson.messaging.commands.pusher

import android.text.TextUtils
import com.liveperson.infra.Command
import com.liveperson.infra.ICallback
import com.liveperson.infra.Infra
import com.liveperson.infra.InternetConnectionService
import com.liveperson.infra.auth.LPAuthenticationParams
import com.liveperson.infra.errors.ErrorCode.ERR_00000054
import com.liveperson.infra.errors.ErrorCode.ERR_00000055
import com.liveperson.infra.log.LPLog.e
import com.liveperson.infra.managers.PreferenceManager
import com.liveperson.infra.network.http.HttpHandler
import com.liveperson.infra.network.http.request.HttpGetRequest
import com.liveperson.messaging.LpError
import com.liveperson.messaging.Messaging
import com.liveperson.messaging.TaskType
import com.liveperson.messaging.model.SynchronizedAuthenticationCompletedCallback
import com.liveperson.messaging.model.SynchronizedInternetConnectionCallback
import org.json.JSONObject
import javax.net.ssl.SSLPeerUnverifiedException

class GetIsPusherRegisteredCommand(private val controller: Messaging,
                                   private val brandId: String,
                                   private val deviceToken: String,
                                   private val appId: String,
                                   private val authenticationParams: LPAuthenticationParams?,
                                   private val callback: ICallback<Boolean, Exception>) : Command {

    private var pusherHelper: PusherHelper? = null
    companion object {
        private const val TAG = "GetIsPusherRegisteredCommand"
        private const val URL = "https://%s/api/account/%s/device/isRegistered?appId=%s&consumerId=%s"
    }

    override fun execute() {
        if (TextUtils.isEmpty(appId)) {
            callback.onError(Exception("Unable to make request. Error: Missing appID"))
            notifyError(Exception("Unable to make request. Error: Missing appID"))
            return
        }
        pusherHelper = PusherHelper(brandId, controller)
        validateNetworkState()
    }

    private fun validateNetworkState() {
        if (InternetConnectionService.isNetworkAvailable()) {
            authorizeAndSendRequest()
        } else {
            SynchronizedInternetConnectionCallback { authorizeAndSendRequest() }.execute()
        }
    }

    private fun authorizeAndSendRequest() {
        val isExecuting = SynchronizedAuthenticationCompletedCallback(controller.mAccountsController, brandId, object : ICallback<Void, Exception> {
            override fun onSuccess(value: Void?) {
                try {
                    val pusherDomain = pusherHelper?.getPusherDomain()
                    if (TextUtils.isEmpty(pusherDomain)) {
                        notifyError(Exception("Error: Missing Domain"))
                        return
                    }

                    val consumerId = pusherHelper?.getConsumerId()
                    if (TextUtils.isEmpty(consumerId)) {
                        notifyError(Exception("Error: Missing consumerID"))
                        return
                    }

                    val url = String.format(URL, pusherDomain, brandId, appId, consumerId)
                    val token = pusherHelper?.getAuthToken()

                    sendRequest(url, token)

                } catch (error: Exception) {
                    e(TAG, ERR_00000054, "sendRequest: Failed to obtain domain/consumerId/token to make request ", error)
                }
            }

            override fun onError(exception: Exception?) {
                exception?.let {
                    e(TAG, ERR_00000055, "authorizeAndSendRequest: Failed to authorize ", it)
                    notifyError(it)
                }
            }

        }).executeWithReturnValue()

        if (!isExecuting) {
            controller.connect(brandId, authenticationParams, null, true)
        }
    }


    private fun sendRequest(requestURL: String, token: String?) {
        val certificates = pusherHelper?.getCertificatePinningKeys()
        val request = HttpGetRequest(requestURL)
        request.addHeader("authorization", "bearer $token")
        request.certificatePinningKeys = certificates
        request.callback = object : ICallback<String, Exception?> {
            override fun onSuccess(response: String) {
                val jsonResponse = JSONObject(response)
                val isRegistered = jsonResponse.getBoolean("isRegistered")
                val registeredDeviceToken = jsonResponse.getString("deviceToken")

                if (isRegistered && deviceToken == registeredDeviceToken) {
                    PreferenceManager.getInstance().setBooleanValue(PreferenceManager.IS_PUSHER_REGISTERED_PREFERENCE_KEY, brandId, true)
                    callback.onSuccess(true)
                } else {
                    PreferenceManager.getInstance().setBooleanValue(PreferenceManager.IS_PUSHER_REGISTERED_PREFERENCE_KEY, brandId, false)
                    callback.onSuccess(false)
                }
            }

            override fun onError(exception: Exception?) {
                callback.onError(exception)
            }
        }
        HttpHandler.execute(request)
    }

    private fun notifyError(exception: Exception) {
        Infra.instance.postOnMainThread {
            if (exception is SSLPeerUnverifiedException) {
                controller.mEventsProxy.onError(TaskType.INVALID_CERTIFICATE, exception.message)
                controller.mEventsProxy.onError(LpError.INVALID_CERTIFICATE, exception.message)
            }
            callback.onError(exception)
        }
    }
}
