package com.liveperson.messaging.commands.tasks;

import com.liveperson.infra.configuration.ConfigurableResource;
import com.liveperson.infra.configuration.Configuration;
import com.liveperson.infra.log.LPLog;
import com.liveperson.infra.managers.PreferenceManager;
import com.liveperson.infra.messaging.R;
import com.liveperson.messaging.model.AmsConnectionAnalytics;

import java.util.HashMap;
import java.util.Map;
import java.util.Set;

/**
 * Created by MaayanK on 4/5/18.
 * A task to remove older images from storage
 */

public class CollectSDKConfigurationDataTask extends BaseAmsAccountConnectionTask {

	private static final String TAG = "CollectConfigurationDataTask";

	private static final long MIN_DAYS_BEFORE_RESENDING_DATA = 30L * 24 * 60 * 60 * 1000;

	private boolean isShouldExecute() {

		long currentTimeInMillis = System.currentTimeMillis();
		long mLastDatePublish =  com.liveperson.infra.managers.PreferenceManager.getInstance().getLongValue(
				PreferenceManager.CONFIGURATION_DATA_PUBLISH_DATE_PREFERENCE_KEY,
				com.liveperson.infra.managers.PreferenceManager.APP_LEVEL_PREFERENCES,
				-1);

		LPLog.INSTANCE.d(TAG, "Android configuration currentTimeInMillis: " + currentTimeInMillis);
		LPLog.INSTANCE.d(TAG, "Android configuration mLastDatePublish: " + mLastDatePublish);

		// in case this is the first time or its been more than 30 days since the last task, send the data again
		return (mLastDatePublish == -1 || mLastDatePublish - currentTimeInMillis >= MIN_DAYS_BEFORE_RESENDING_DATA);

	}

	@Override
	public void execute() {

		if (!isShouldExecute()) {
			LPLog.INSTANCE.d(TAG, "Android configuration should not execute");
			mCallback.onTaskSuccess();
			return;
		}

		AmsConnectionAnalytics.collectSDKConfigurationDataTaskStart();

		// The following operations are dealing with reflection, so we'll do it on a different thread.
		new Thread(() -> {
			HashMap<Integer, String> booleans = Configuration.scanXmlIdsAndNames(R.bool.class);
			HashMap<Integer, String> colors = Configuration.scanXmlIdsAndNames(R.color.class);
			HashMap<Integer, String> integers = Configuration.scanXmlIdsAndNames(R.integer.class);
			HashMap<Integer, String> dimensions = Configuration.scanXmlIdsAndNames(R.dimen.class);

			String sb = "SDKConfiguration: " +
					printList(booleans, ConfigurableResource.ResourceType.Boolean) +
					printList(integers, ConfigurableResource.ResourceType.Integer) +
					printList(dimensions, ConfigurableResource.ResourceType.Dimension);
			LPLog.INSTANCE.d(TAG, "Android configuration is: " + sb);

			AmsConnectionAnalytics.collectSDKConfigurationDataTaskEnd();

			PreferenceManager.getInstance().setLongValue(
					PreferenceManager.CONFIGURATION_DATA_PUBLISH_DATE_PREFERENCE_KEY,
					PreferenceManager.APP_LEVEL_PREFERENCES,
					System.currentTimeMillis());

			LPLog.INSTANCE.d(TAG, "Android configuration setLongValue: " + System.currentTimeMillis());

			mCallback.onTaskSuccess();
		}).start();

	}


	/**
	 * print a list of ConfigurableResource objects to represent the resources on a list.
	 * @param resources A hash map of resource IDs: integer address and name.
	 * @param resourceType the type of the resources.
	 */
	private StringBuilder printList(HashMap<Integer,String> resources, ConfigurableResource.ResourceType resourceType) {
		StringBuilder sb = new StringBuilder();
		Set<Map.Entry<Integer,String>> resourceEntries = resources.entrySet();
		for (Map.Entry<Integer,String> resourceEntry: resourceEntries) {

			String resourceName = resourceEntry.getValue();
			String value;

			switch (resourceType) {
				case Integer:
					value = String.valueOf(Configuration.getInteger(resourceEntry.getKey()));
					break;
				case Dimension:
					value = String.valueOf(Configuration.getDimension(resourceEntry.getKey()));
					break;
				case Boolean:
					value = String.valueOf(Configuration.getBoolean(resourceEntry.getKey()));
					break;
				default:
					value = "unknown value";
					break;

			}
			sb.append(" configurationId= ").append(resourceName).append(" value= ").append(value).append("\n");
		}
		return sb;
	}

	@Override
	public String getName() {
		return TAG;
	}
}
