package com.liveperson.api.response.model;

import android.text.TextUtils;

import com.liveperson.api.exception.BadConversationException;
import com.liveperson.api.response.types.CSAT;
import com.liveperson.api.response.types.CloseReason;
import com.liveperson.api.response.types.ConversationState;
import com.liveperson.api.response.types.DialogState;
import com.liveperson.infra.log.LPLog;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.util.Arrays;

import static com.liveperson.infra.errors.ErrorCode.ERR_00000069;
import static com.liveperson.infra.errors.ErrorCode.ERR_0000006A;
import static com.liveperson.infra.errors.ErrorCode.ERR_0000006B;

/**
 * Created by shiranr on 1/14/18.
 */

abstract public class ConversationHistoryDetails {
    private static final String TAG = "ConversationHistoryDetails";

    public Participants participants;
    @Deprecated
    private ConversationState state; // In case the MAIN dialog is closed the state will be also closed
    protected ConversationState stage; // In case all dialogs is closed the stage will be also closed
    public long startTs;
    public long endTs;
    public CSAT csat;
    public CloseReason closeReason;
    public DialogData[] dialogs;
    public String conversationId;

    //{"convId":"145e7d23-ecd8-4ead-a3c1-e6a0b0d999ae",
    // "participants":[{"id":"a44b8fb5e565dea916e1f3bd50cd0e5fc3cd992f38cf77da0a3966d399a5d11b","role":"CONSUMER"}
    // ,{"id":"86aece85-c4af-545c-8793-3b40a973ae84","role":"ASSIGNED_AGENT"}],
    //
    // "state":"CLOSE",
    // "startTs":1515401555951,
    // "endTs":1515402857709,
    // "csat":{"csatRate":4,"csatResolutionConfirmation":false,"status":"PARTIALLY_FILLED"}}
    public ConversationHistoryDetails(JSONObject jsonObject, String conversationId) throws JSONException, BadConversationException {
        endTs = jsonObject.optLong("endTs", -1);
        startTs = jsonObject.optLong("startTs", -1);
        if (conversationId == null) {
            conversationId = jsonObject.optString("convId");
        }
        this.conversationId = conversationId;

        if (startTs == 0 || endTs == 0) {
            throw new BadConversationException("Bad startTs/endTs " + startTs + " / " + endTs);
        }

        parseStage(jsonObject);

        csat = new CSAT(jsonObject);

        //check what format we got the participants data array or object. (INCA or UMS)
        JSONArray listOfParticipants = jsonObject.optJSONArray("participants");
        if (listOfParticipants != null) {
            //V3
            parseParticipantsId(listOfParticipants);
        }

        parseCloseReason(jsonObject);

        JSONArray dialogArray = jsonObject.optJSONArray("dialogs");
        if (dialogArray == null) {
            dialogArray = new JSONArray();
        }

        if (TextUtils.isEmpty(conversationId)) {
            throw new BadConversationException("Empty conversation id");
        }

        dialogs = new DialogData[dialogArray.length()];
        for (int i = 0; i < dialogArray.length(); i++) {
            DialogData dialog = MultiDialog.create(dialogArray.getJSONObject(i), conversationId);
            if (dialog.closeReason == null) {
                dialog.closeReason = closeReason;
            }
            if (dialog.state == null) {
                dialog.setState(DialogState.parse(getState()));
            }

            dialogs[i] = dialog;
        }
        // Fixing timestamps, bug: LE-92631
//        fixTimestamps(dialogs);
// Comment out the line above to avoid an issue with voice & video call.
// If there is a call in the conversation, this method will make the conversation resolve divider always before the first call invite.
    }

    private static void fixTimestamps(DialogData[] dialogs) {
        if (dialogs.length == 0 || dialogs.length == 1) return;

        //LPMobileLog.d(TAG, "Testing: " + Arrays.toString(dialogs));

        Arrays.sort(dialogs, (d1, d2) -> Long.compare(d1.creationTs, d2.creationTs));

        for (int i = 1; i < dialogs.length; i++) {
            long diff = dialogs[i].creationTs - dialogs[i - 1].endTs;
            if (diff < 1) {
                // Fine tuning the previous dialog end timestamp to on millisecond BEFORE the following dialog's creation timestamp
                dialogs[i - 1].endTs += diff - 1;
            }
        }

        //LPMobileLog.d(TAG, "Testing: " + Arrays.toString(dialogs));
    }

    private void parseCloseReason(JSONObject jsonObject) {
        closeReason = CloseReason.CONSUMER;
        String closeReasonStr = jsonObject.optString("closeReason");
        if (closeReasonStr != null && !TextUtils.isEmpty(closeReasonStr) && !closeReasonStr.equals("null")) {
            closeReason = CloseReason.valueOf(closeReasonStr);
        }
    }

    // "participants":[{"id":"a44b8fb5e565dea916e1f3bd50cd0e5fc3cd992f38cf77da0a3966d399a5d11b","role":"CONSUMER"}
    // ,{"id":"86aece85-c4af-545c-8793-3b40a973ae84","role":"ASSIGNED_AGENT"}],
    //
    private void parseParticipantsId(JSONArray listOfParticipants) {
        participants = new Participants();

        for (int i = 0; i < listOfParticipants.length(); i++) {
            try {
                JSONObject par = (JSONObject) listOfParticipants.get(i);
                String id = par.optString("id");
                String stringRole = par.optString("role");
                Participants.ParticipantRole role = Participants.ParticipantRole.getParticipantRole(stringRole);

                if (role != null && id != null) {
                    participants.add(new String[]{id}, role);
                }
            } catch(Exception e) {
                LPLog.INSTANCE.e(TAG, ERR_00000069, "Failed to parse participants list.", e);
            }
        }
    }

    private void parseStage(JSONObject jsonObject) {
        if (jsonObject.has("stage")) {
            stage = ConversationState.parse(jsonObject.optString("stage"));
            LPLog.INSTANCE.d(TAG, "This JSON has stage field! JSON: " + jsonObject);
        }
        else {
			LPLog.INSTANCE.e(TAG, ERR_0000006A, "This JSON has NO stage field!");
		}

        // deprecated...
        String stateStr = jsonObject.optString("state");
        if (!TextUtils.isEmpty(stateStr)) {
            state = ConversationState.valueOf(stateStr);
        }
        else {
			LPLog.INSTANCE.e(TAG, ERR_0000006B, "parseStage: JSON does not have 'state' value.");
			if (stage != null) {
				state = stage;
				LPLog.INSTANCE.d(TAG, "parseStage: setting state to stage value (" + stage + ")");
			}
			else {
				LPLog.INSTANCE.w(TAG, "parseStage: setting state CLOSE");
				state = ConversationState.CLOSE;
			}
		}
        // ... end of deprecated.
    }

    public ConversationState getState() {
        // To support server's backward compatibility, we first check that the stage exists, if not, return the old state.
        return stage == null ? state : stage;
    }
}
