package com.liveperson.messaging.commands;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.liveperson.api.request.BaseAMSSocketRequest;
import com.liveperson.api.response.model.ContentType;
import com.liveperson.api.response.model.DeliveryStatusUpdateInfo;
import com.liveperson.api.response.types.ConversationState;
import com.liveperson.api.response.types.DialogState;
import com.liveperson.infra.Command;
import com.liveperson.infra.ICallback;
import com.liveperson.infra.log.LPLog;
import com.liveperson.infra.network.socket.SocketManager;
import com.liveperson.infra.utils.EncryptionVersion;
import com.liveperson.infra.utils.LinkUtils;
import com.liveperson.infra.utils.MaskedMessage;
import com.liveperson.infra.utils.UniqueID;
import com.liveperson.messaging.Messaging;
import com.liveperson.messaging.model.AmsMessages;
import com.liveperson.messaging.model.Conversation;
import com.liveperson.messaging.model.Dialog;
import com.liveperson.messaging.model.MessagingChatMessage;
import com.liveperson.messaging.network.MessageTimeoutQueue;
import com.liveperson.messaging.network.socket.requests.NewConversationRequest;
import com.liveperson.messaging.network.socket.requests.SendMessageRequest;

import static com.liveperson.infra.errors.ErrorCode.ERR_000000A7;
import static com.liveperson.infra.errors.ErrorCode.ERR_000000A8;
import static com.liveperson.infra.errors.ErrorCode.ERR_000000A9;

/**
 * Created by Ilya Gazman on 11/11/2015.
 * <p/>
 * A command for sending message
 */
public class SendMessageCommand implements Command {

    private static final String TAG = "SendMessageCommand";

    protected final Messaging mController;

    protected String mBrandId;
    protected String mTargetId;
    protected MaskedMessage mMessage;
    protected String mConsumerId;
    protected String mEventId;
	protected DeliveryStatusUpdateInfo mInfo;

    /**
     * Creates a command for sending messages
     *
     * @param controller
     * @param message
     */
    public SendMessageCommand(Messaging controller, String targetId, String brandId, MaskedMessage message) {
        mController = controller;
        mBrandId = brandId;
        mTargetId = targetId;
        mMessage = message;
    }

    /**
     * Creates a command for sending messages
     *
     * @param controller
     * @param message
	 * @param info a {@link DeliveryStatusUpdateInfo} object with metadata to be sent with the publish
     */
    public SendMessageCommand(Messaging controller, String targetId, String brandId, MaskedMessage message, @Nullable DeliveryStatusUpdateInfo info) {
		this(controller, targetId, brandId, message);
        mInfo = info;
    }

    @Override
    public void execute() {
        Dialog dialog = mController.amsDialogs.getActiveDialog();
        Conversation conversation = mController.amsConversations.getConversationFromTargetIdMap(mTargetId);
        boolean isConversationOpen = (conversation != null && conversation.getState() == ConversationState.OPEN);

        mConsumerId = mController.amsUsers.getConsumerId(mTargetId);

        if (!isConversationOpen) {
            // Create and persist a temp conversation + temp dialog in DB
            sendOnNewConversation();
        } else if (dialog == null) {
            LPLog.INSTANCE.w(TAG, "'Zombie Conversation' state occurred (One conversation is open without open dialogs) while the consumer is trying to send a message! Closing this conversation and open a new one....");
            ResolveConversationCommand resolveConversationCommand = new ResolveConversationCommand(mController.amsConversations, mTargetId, mController.mAccountsController.getConnectionUrl(mBrandId));
            resolveConversationCommand.setCallback(new ICallback<String, Throwable>() {
                @Override
                public void onSuccess(String value) {
                    // Create and persist a temp conversation + temp dialog in DB
                    sendOnNewConversation();
                }

                @Override
                public void onError(Throwable error) {
                    LPLog.INSTANCE.e(TAG, ERR_000000A7, "Failed to resolve conversation before creating a new one, continuing with creating anyway", error);
                    // Create and persist a temp conversation + temp dialog in DB
                    sendOnNewConversation();
                }
            });
            resolveConversationCommand.execute();
        } else {
            LPLog.INSTANCE.d(TAG, "Send message - dialog = " + dialog.getDialogId() + ", " + dialog.getState());
            switch (dialog.getState()) {
                case CLOSE:
                    // Create and persist a temp conversation + temp dialog in DB
                    sendOnNewConversation();
                    break;
                case LOCKED:
                    // _TODO: 11/12/2015 What is this state?
                    // _TODO: 15/05/2018 It's a state from the UMS but it's not in use... for 3 years
                    break;
                case OPEN:
                case PENDING:
                case QUEUED:
                    addMessageToDBAndSend(dialog.getDialogId(), mMessage);
                    break;
            }
        }
    }

    /**
     * Creates and persists a temp conversation + temp dialog in DB, adds a message to DB and sends a create-conversation request.
     */
    private void sendOnNewConversation() {
        BaseAMSSocketRequest newConversationRequest = createPendingConversationAndDialog();
        sendCreateConversationRequest(newConversationRequest);
        // This must come AFTER creating conversation request!
        addMessageToDBAndSend(Dialog.TEMP_DIALOG_ID, mMessage);
	    LPLog.INSTANCE.d(TAG, "sendOnNewConversation: " + LPLog.INSTANCE.mask(mMessage));
    }

    /**
     * Creates pending conversation
     */
    protected BaseAMSSocketRequest createPendingConversationAndDialog() {
        final NewConversationRequest newConversationRequest = createNewConversationRequest();
        mController.amsConversations.createPendingConversation(mTargetId, mBrandId, newConversationRequest.getRequestId());
        mController.amsDialogs.createPendingDialog(mTargetId, mBrandId, newConversationRequest.getRequestId());
        return newConversationRequest;
    }

    private NewConversationRequest createNewConversationRequest() {
        return createNewConversationRequest(Conversation.TEMP_CONVERSATION_ID, Dialog.TEMP_DIALOG_ID, null);
    }

    protected NewConversationRequest createNewConversationRequest(String tempConversationId, String tempDialogId, Long requestId) {
        final NewConversationRequest newConversationRequest = new NewConversationRequest(mController, mTargetId, mBrandId, tempConversationId, tempDialogId);
        if (requestId != null) {
            newConversationRequest.setRequestId(requestId);
        }

        mController.amsConversations.enqueuePendingConversationRequestId(newConversationRequest.getRequestId());

        return newConversationRequest;
    }

    public void sendCreateConversationRequest(BaseAMSSocketRequest newConversationRequest) {
        SocketManager.getInstance().send(newConversationRequest);
    }

    /**
     * Sends message request
     */
    protected void sendMessage(String conversationId, String dialogId, String message) {
	    SendMessageRequest sendMessageRequest = new SendMessageRequest(mController, mEventId, mTargetId, mBrandId, conversationId, dialogId, message);

	    sendMessage(dialogId, sendMessageRequest);
    }

	/**
	 * Sends message request
	 */
	protected void sendMessage(String dialogId, SendMessageRequest sendMessageRequest) {

		SocketManager.getInstance().send(sendMessageRequest);
		// Add the message to the message queue in order to track if we got ack on it
		mController.amsMessages.mMessageTimeoutQueue.add(MessageTimeoutQueue.MessageType.PUBLISH, (int) sendMessageRequest.getRequestId(),
				mBrandId, dialogId, mEventId);
	}

	protected void sendMessageIfDialogIsOpen() {
		Dialog dialog = mController.amsDialogs.getActiveDialog();

		if (dialog != null) {
			SendMessageRequest sendMessageRequest;
			LPLog.INSTANCE.d(TAG, "sendMessageIfDialogIsOpen: " + dialog.getState());
			if (dialog.getState() == DialogState.OPEN) {
				sendMessageRequest = createMessageRequest(mController, mEventId, mTargetId, mBrandId, dialog.getDialogId(), dialog.getConversationId());
				sendMessage(dialog.getDialogId(), sendMessageRequest);
			} else if (dialog.getState() == DialogState.PENDING) {
				sendMessageRequest = createMessageRequest(mController, mEventId, mTargetId, mBrandId, dialog.getDialogId(), dialog.getConversationId());
				dialog.getPendingData().addToPendingRequests(sendMessageRequest);
			} else {
				LPLog.INSTANCE.e(TAG, ERR_000000A8, "sendMessageIfDialogIsOpen: unhandled dialog state:" + dialog.getState());
			}
		} else {
			LPLog.INSTANCE.e(TAG, ERR_000000A9, "sendMessageIfDialogIsOpen: Failed to find an active dialog!");
		}
	}

    /**
     * This method is used to add a message to DB. If regexPattern and maskCharacter are provided, the message is checked
     * and masked if the regex pattern exist in the message. Otherwise it is sent as is.
     *
     * @param dialogId
     * @param message
     */
    protected void addMessageToDBAndSend(String dialogId, MaskedMessage message) {
        mEventId = UniqueID.createUniqueMessageEventId();
        LPLog.INSTANCE.i(TAG, "addMessageToDBAndSend: mEventId = "+ mEventId + " dialog ID = " + dialogId);

        final MessagingChatMessage chatMessage = createNewChatMessage(dialogId, message);

        if (mInfo != null && mInfo.getMetadata() != null) {
            chatMessage.setMetadata(mInfo.getMetadata().toString());
        }

        // Add to DB and send
        mController.amsMessages.addMessage(chatMessage, true)
                .setPostQueryOnBackground((Long data) -> {
                    LPLog.INSTANCE.i(TAG, "Send message, time: " + chatMessage.getTimeStamp());
                    sendMessageIfDialogIsOpen();
                }).execute();

        // If messages are different it means the message was masked. We add a system message for the user
        if (mMessage.isMasked()) {
            String maskedEventId = UniqueID.createUniqueMessageEventId();

            MessagingChatMessage warning = new MessagingChatMessage(chatMessage.getOriginatorId(),
                    message.getMaskedSystemMessage(),
                    chatMessage.getTimeStamp() + 1, chatMessage.getDialogId(), maskedEventId,
                    MessagingChatMessage.MessageType.SYSTEM_MASKED, MessagingChatMessage.MessageState.RECEIVED,
                    AmsMessages.MASKED_CC_MSG_SEQUENCE_NUMBER, ContentType.text_plain.getText(), EncryptionVersion.NONE);

            mController.amsMessages.addMessage(warning, true).execute();
        }
    }

    @NonNull
    protected MessagingChatMessage createNewChatMessage(String dialogId, MaskedMessage message) {
        MessagingChatMessage.MessageType messageType;
        // If a message contains URL, mark it as a Consumer URL
        if (message.isMasked()) {
            if (LinkUtils.containsWebUrls(message.getDbMessage())) {
                messageType = MessagingChatMessage.MessageType.CONSUMER_URL_MASKED;
            } else {
                messageType = MessagingChatMessage.MessageType.CONSUMER_MASKED;
            }
        } else {
            if (LinkUtils.containsWebUrls(message.getDbMessage())) {
                messageType = MessagingChatMessage.MessageType.CONSUMER_URL;
            } else {
                messageType = MessagingChatMessage.MessageType.CONSUMER;
            }
        }
        return new MessagingChatMessage(mConsumerId,
                message.getDbMessage(), System.currentTimeMillis(),
                dialogId,
				mEventId,
                messageType,
                MessagingChatMessage.MessageState.PENDING,
                EncryptionVersion.NONE);
    }

	@NonNull
	protected SendMessageRequest createMessageRequest(Messaging mController, String mEventId, String mTargetId, String mBrandId, String dialogId, String conversationId) {
		SendMessageRequest sendMessageRequest = new SendMessageRequest(mController, mEventId, mTargetId, mBrandId, dialogId, conversationId);
		sendMessageRequest.setMessageContent(mMessage.getServerMessage());
		sendMessageRequest.setInfo(mInfo);
		return sendMessageRequest;
	}


    public String getEventId() {
        return mEventId;
    }
}
