package com.liveperson.messaging;

import android.app.Notification;
import android.app.PendingIntent;
import android.content.Context;
import android.content.Intent;
import android.os.Bundle;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import android.text.TextUtils;
import android.webkit.MimeTypeMap;
import android.widget.Toast;

import com.liveperson.api.LivePersonCallback;
import com.liveperson.api.request.GeneratedUploadTokenField;
import com.liveperson.api.response.model.DeliveryStatusUpdateInfo;
import com.liveperson.api.response.types.ConversationState;
import com.liveperson.api.response.types.DeliveryStatus;
import com.liveperson.api.response.types.DialogState;
import com.liveperson.api.response.types.DialogType;
import com.liveperson.api.response.types.TTRType;
import com.liveperson.infra.Clearable;
import com.liveperson.infra.ConversationViewParams;
import com.liveperson.infra.ICallback;
import com.liveperson.infra.Infra;
import com.liveperson.infra.PushType;
import com.liveperson.infra.PushUnregisterType;
import com.liveperson.infra.auth.LPAuthenticationParams;
import com.liveperson.infra.auth.LPAuthenticationType;
import com.liveperson.infra.callbacks.InitLivePersonCallBack;
import com.liveperson.infra.callbacks.LogoutLivePersonCallBack;
import com.liveperson.infra.configuration.Configuration;
import com.liveperson.infra.eventmanager.EventManagerService;
import com.liveperson.infra.log.FlowTags;
import com.liveperson.infra.log.LPLog;
import com.liveperson.infra.managers.PreferenceManager;
import com.liveperson.infra.messaging.R;
import com.liveperson.infra.model.LPWelcomeMessage;
import com.liveperson.infra.model.types.ChatState;
import com.liveperson.infra.network.socket.SocketManager;
import com.liveperson.infra.preferences.PushMessagePreferences;
import com.liveperson.infra.sdkstatemachine.logout.PreLogoutCompletionListener;
import com.liveperson.infra.sdkstatemachine.shutdown.ShutDownAsync;
import com.liveperson.infra.sdkstatemachine.shutdown.ShutDownCompletionListener;
import com.liveperson.infra.statemachine.InitProcess;
import com.liveperson.infra.statemachine.LogoutProcess;
import com.liveperson.infra.statemachine.ShutDownProcess;
import com.liveperson.infra.utils.FileUtils;
import com.liveperson.infra.utils.LPAudioUtils;
import com.liveperson.infra.utils.LocalBroadcast;
import com.liveperson.infra.utils.MaskedMessage;
import com.liveperson.infra.utils.MessageValidator;
import com.liveperson.messaging.background.BackgroundActionsService;
import com.liveperson.messaging.background.FileSharingManager;
import com.liveperson.messaging.background.filesharing.FileExtensionTypes;
import com.liveperson.messaging.background.filesharing.FileSharingType;
import com.liveperson.messaging.commands.ChangeChatStateCommand;
import com.liveperson.messaging.commands.ChangeConversationTTRCommand;
import com.liveperson.messaging.commands.CloseDialogCommand;
import com.liveperson.messaging.commands.DeliveryStatusUpdateCommand;
import com.liveperson.messaging.commands.ResendFormSubmissionMessageCommand;
import com.liveperson.messaging.commands.ResendMessageCommand;
import com.liveperson.messaging.commands.ResendURLMessageCommand;
import com.liveperson.messaging.commands.ResolveConversationCommand;
import com.liveperson.messaging.commands.SendCsatCommand;
import com.liveperson.messaging.commands.SendFormSubmissionMessageCommand;
import com.liveperson.messaging.commands.SendGenerateUploadTokenCommand;
import com.liveperson.messaging.commands.SendMessageCommand;
import com.liveperson.messaging.commands.SendMessageWithURLCommand;
import com.liveperson.messaging.commands.SendSetUserProfileCommand;
import com.liveperson.messaging.commands.pusher.GetIsPusherRegisteredCommand;
import com.liveperson.messaging.commands.pusher.GetUnreadMessagesCountCommand;
import com.liveperson.messaging.commands.pusher.RegisterPusherCommand;
import com.liveperson.messaging.commands.pusher.SendReadAcknowledgementCommand;
import com.liveperson.messaging.commands.pusher.UnregisterPusherCommand;
import com.liveperson.messaging.commands.tasks.MessagingEventSubscriptionManager;
import com.liveperson.messaging.controller.AccountsController;
import com.liveperson.messaging.controller.AmsReadController;
import com.liveperson.messaging.controller.ClientProperties;
import com.liveperson.messaging.controller.ConnectionsController;
import com.liveperson.messaging.model.AgentData;
import com.liveperson.messaging.model.AmsAccount;
import com.liveperson.messaging.model.AmsConversations;
import com.liveperson.messaging.model.AmsDialogs;
import com.liveperson.messaging.model.AmsFiles;
import com.liveperson.messaging.model.AmsMessages;
import com.liveperson.messaging.model.AmsUsers;
import com.liveperson.messaging.model.Conversation;
import com.liveperson.messaging.model.ConversationUtils;
import com.liveperson.messaging.model.Dialog;
import com.liveperson.messaging.model.Form;
import com.liveperson.messaging.model.MessagingChatMessage;
import com.liveperson.messaging.model.MessagingUserProfile;
import com.liveperson.messaging.model.QuickRepliesMessageHolder;
import com.liveperson.messaging.model.UserProfile;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.util.ArrayList;

import static com.liveperson.infra.errors.ErrorCode.ERR_0000006C;
import static com.liveperson.infra.errors.ErrorCode.ERR_0000006D;
import static com.liveperson.infra.errors.ErrorCode.ERR_0000006E;
import static com.liveperson.infra.errors.ErrorCode.ERR_0000006F;
import static com.liveperson.infra.errors.ErrorCode.ERR_00000070;
import static com.liveperson.infra.errors.ErrorCode.ERR_00000071;
import static com.liveperson.infra.errors.ErrorCode.ERR_00000072;
import static com.liveperson.infra.errors.ErrorCode.ERR_00000073;
import static com.liveperson.infra.errors.ErrorCode.ERR_00000074;
import static com.liveperson.infra.managers.PreferenceManager.KEY_TYPED_TEXT;

/**
 * Created by Ilya Gazman on 11/10/2015.
 * <p/>
 * A singleton holder, it helps with intelligence
 */
public class Messaging implements IMessaging, ShutDownAsync, Clearable {

	private static final String TAG = "Messaging";
	public static final int NO_FILE_ROW_ID = -1;
	private static final String SUBMISSION_ID = "submissionId";
	private static final String INVITATION_ID = "invitationId";
	private static final String FORM_TITLE = "formTitle";

	public ConnectionsController mConnectionController;
	public AccountsController mAccountsController;

	public AmsMessages amsMessages;
	public AmsConversations amsConversations;
	public AmsDialogs amsDialogs;
	public AmsUsers amsUsers;
	public AmsFiles amsFiles;

	// Flag indicating if still busy fetching history.
	private boolean isStillBusyFetching;

	// Indicate if we're using a service to upload/download images
	private boolean mUploadUsingService;

	private ClientProperties mClientProperties;

	// This controller does not have access from outside. It is used to listen on connection and foreground
	// changes and act on them
	public AmsReadController amsReadController;

	public LivePersonEventsProxy mEventsProxy;

	private MessageValidator mMessageValidator;

	private FileSharingManager mFileSharingManager;
	/**
	 * The PendingIntent to use on the foreground service notification
	 */
	private PendingIntent mImageServicePendingIntent = null;
	/**
	 * The notification builder for the image upload/download foreground service
	 */
	private Notification.Builder mImageForegroundServiceUploadNotificationBuilder = null;
	private Notification.Builder mImageForegroundServiceDownloadNotificationBuilder = null;

	/**
	 * Indication whether the structured content feature is on in branding
	 */
	private boolean mEnableStructuredContent;

	private MessagingEventSubscriptionManager mMessagingEventSubscriptionManager;

	private ConversationViewParams mConversationViewParams;

	private LPAudioUtils mAudioUtils;

	// Max number of files to keep in storage
	private int mMaxNumberOfStoredImageFiles;
	private int mMaxNumberOfStoredVoiceFiles;
	private int mMaxNumberOfStoredDocumentFiles;

	private String clientOnlySystemMessage;
	private String realTimeSystemMessage;

	/**
	 * Use SharedPreferences to:
	 *  - Set flag value after cleared history
	 *  - Remove key when consumer opened conversation screen
	 */
	private final String KEY_DID_CLEAR_HISTORY = "KEY_DID_CLEAR_HISTORY";

	public Messaging() {
		mEventsProxy = new LivePersonEventsProxy();
	}

	private void initMembers(Context context) {
		mConnectionController = new ConnectionsController(this);
		mAccountsController = new AccountsController(mClientProperties);
		amsMessages = new AmsMessages(this);
		amsConversations = new AmsConversations(this);
		amsDialogs = new AmsDialogs(this);
		amsUsers = new AmsUsers();
		amsFiles = new AmsFiles();
		amsReadController = new AmsReadController(this);
		mUploadUsingService = Configuration.getBoolean(R.bool.upload_photo_using_service);
		mFileSharingManager = new FileSharingManager(this, context);
		mEnableStructuredContent = Configuration.getBoolean(R.bool.enable_structured_content);
		mMessagingEventSubscriptionManager = new MessagingEventSubscriptionManager();
		// Get limit from resources
		mMaxNumberOfStoredImageFiles = Configuration.getInteger(R.integer.max_number_stored_images);
		mMaxNumberOfStoredVoiceFiles = Configuration.getInteger(R.integer.max_number_stored_voice_files);
        mMaxNumberOfStoredDocumentFiles = Configuration.getInteger(R.integer.max_number_stored_documents);
		mAudioUtils = new LPAudioUtils();
	}

	/**
	 * Init Messaging
	 * @param context  - app context
	 * @param initData - the sdk version.
	 * @param callBack - InitLivePersonCallBack callback that will notify if init succeeded or failed.
	 */
	@Override
	public void init(final Context context, final MessagingInitData initData, final InitLivePersonCallBack callBack) {
		init(context, initData, new InitProcess() {
			@Override
			public void init() {
				initMessaging(context, initData);
			}

			@Override
			public InitLivePersonCallBack getInitCallback() {
				return callBack;
			}
		});
	}

	/**
	 * Init Messaging under upper module (MessagingUI) with it's own InitProcess.
	 * If Messaging is being used without upper module of UI, use {@link #init(Context, MessagingInitData, InitLivePersonCallBack)}
	 * @param context     - app context
	 * @param initData    - the sdk version and app id.
	 * @param initProcess - init process of upper module (MessagingUI)
	 */
	@Override
	public void init(final Context context, final MessagingInitData initData, final InitProcess initProcess) {

		Infra.instance.init(context, initData, new InitProcess() {
			@Override
			public InitLivePersonCallBack getInitCallback() {
				return initProcess.getInitCallback();
			}

			@Override
			public void init() {
				initMessaging(context, initData);
				initProcess.init();
			}

		});
	}

	/**
	 * Initialize SDK analytics service.
	 * @param brandId brand's account id
	 * @param appId host app id
	 */
	public void initAnalyticsService(Context applicationContext, String brandId, String appId) {
		Infra.instance.initAnalyticsService(applicationContext, brandId, appId);
	}

	/**
	 * Init the masked message text in app language.
	 *
	 * @param context Current context. Not app context.
	 */
	public void initMaskedMessage(Context context) {
		realTimeSystemMessage = context.getString(com.liveperson.infra.R.string.lp_system_message_real_time_masked);
		clientOnlySystemMessage = context.getString(com.liveperson.infra.R.string.lp_system_message_client_only_masked);
	}

	MessageValidator getMessageValidator(String brandId) {
		if (mMessageValidator == null) {
			//todo maayan remove this
			AmsAccount account = mAccountsController.getAccount(brandId);
			if (account == null){
				LPLog.INSTANCE.i(TAG, "Missing account for a consumer. SDK may not be initialized for brandId: " + brandId);
				return null;
			}
			boolean isAuthenticated = account.isAuthenticated();
			mMessageValidator = new MessageValidator(isAuthenticated, realTimeSystemMessage, clientOnlySystemMessage);
		}
		return mMessageValidator;
	}

	private void initMessaging(Context context, MessagingInitData initData) {
		LPLog.INSTANCE.d(TAG, "Initializing...");

		updateClientProperties(initData);

		// Get message masking information
		initMembers(context);
		mConnectionController.initConnectionReceiver();
		bootstrapRegistration();
	}

	/**
	 * saving important data of the device, sdk and the host app.
	 * @param initData
	 */
	private void updateClientProperties(MessagingInitData initData) {
		if (initData != null) {
			//any new init data will overwrite the old client properties.
			mClientProperties = new ClientProperties(initData.getAppId(), initData.getSdkVersion());
		} else if (mClientProperties == null) {
			//in case init data is null, we need to restore client properties from shared preferences.
			mClientProperties = new ClientProperties();
		}
	}

	public void updateWelcomeMessage(String brandId) {
		if (isConversationEmptyOrClose(brandId) && !QuickRepliesMessageHolder.containsQuickRepliesFromAgent(brandId)) {
			LPWelcomeMessage welcomeMessage = mConversationViewParams.getLpWelcomeMessage();
			ConversationUtils conversationUtils = new ConversationUtils(MessagingFactory.getInstance().getController());
			conversationUtils.updateWelcomeMessage(brandId, welcomeMessage);
			amsMessages.resetQuickRepliesMessageHolder();
			QuickRepliesMessageHolder.updateQuickReplies(brandId, welcomeMessage.getQuickReplies(false));
		}
	}

	public boolean isConversationEmptyOrClose(String brandId) {
		Conversation conversation = amsConversations.getConversationFromTargetIdMap(brandId);
		return  (conversation == null || conversation.getState() == ConversationState.CLOSE);
	}

	/**
	 * Maps response handlers for server responses
	 */
	private void bootstrapRegistration() {
		// Register a handler for content event notifications
		SocketManager.getInstance().putGeneralHandlerMap(new GeneralMessagingResponseHandler(this));
	}

	/**
	 * Start connecting to the AMS server
	 */
	@Override
	public void connect(String brandId, LPAuthenticationParams lpAuthenticationParams, @Nullable ConversationViewParams conversationViewParams, boolean connectInBackground) {
		initBrand(brandId, lpAuthenticationParams, conversationViewParams);
		LPLog.INSTANCE.d(TAG, "Connecting to brand " + brandId);
		mConnectionController.connect(brandId, connectInBackground);
	}
	/**
	 * Start connecting to the AMS server
	 */
	@Override
	public void connect(String brandId, LPAuthenticationParams lpAuthenticationParams, @Nullable ConversationViewParams conversationViewParams) {
		connect(brandId, lpAuthenticationParams, conversationViewParams, false);
	}

	@Override
	public void reconnect(String brandId, LPAuthenticationParams lpAuthenticationParams) {
		LPLog.INSTANCE.d(TAG, "reconnect: set a new authentication key for brand with lpAuthenticationParams of type " + lpAuthenticationParams.getAuthType());
		connect(brandId, lpAuthenticationParams, null, false);
	}

	/**
	 * Indicate move to background
	 * @param brandId
	 * @param timeout the time to leave the socket open
	 */
	@Override
	public void moveToBackground(String brandId, long timeout) {

		// Stop all playing voice
		MessagingFactory.getInstance().getController().getAudioUtils().getPlayingAudioManager().stopAllCurrentlyPlaying();
		try {
			SendReadAcknowledgementCommand.clearAcknowledgedConversations();
			GetUnreadMessagesCountCommand.clearMappedUnreadCount();
			Infra.instance.getAnalyticsService().sendAnalyticsDataToLoggos(true);
//			Infra.instance.getConsumerManager().resetAuthState();
			PushMessagePreferences.INSTANCE.cleanUp(brandId);
			if (!mAccountsController.isInUnAuthMode(brandId)) {
				amsMessages.removeLastOutboundMessage().execute();
			}
		} catch (Exception error) {
			LPLog.INSTANCE.e(TAG, ERR_0000006C, "moveToBackground: Failed to clear acknowledged conversations", error);
		}

		mConnectionController.moveToBackground(brandId, timeout);
	}

	/**
	 * Indicate move to foreground
	 * @param brandId
	 * @param lpAuthenticationParams
	 */
	@Override
	public void moveToForeground(String brandId, LPAuthenticationParams lpAuthenticationParams, @Nullable ConversationViewParams conversationViewParams) {
		LPLog.INSTANCE.d(TAG, "moveToForeground: brandId = " + brandId);

		initBrand(brandId, lpAuthenticationParams, conversationViewParams);
		mConnectionController.moveToForeground(brandId);
		// See If we need to add outbound campaign's welcome message
		if (isConversationEmptyOrClose(brandId) && lpAuthenticationParams.getAuthType() == LPAuthenticationType.AUTH) {
			ConversationUtils conversationUtils = new ConversationUtils(MessagingFactory.getInstance().getController());
			conversationUtils.updateOutboundCampaignMessage(brandId);
		}
	}

	@Override
	public void onConversationDestroyed(String brandId) {
		Infra.instance.getConsumerManager().resetAuthState();
	}

	/**
	 * Indicate the upload image service has started
	 * @param brandId
	 */
	@Override
	public void serviceStarted(String brandId) {
		LPLog.INSTANCE.d(TAG, "serviceStarted: brandId = " + brandId);
		mConnectionController.serviceStarted(brandId);
	}

	/**
	 * Indicate service was stopped
	 * @param brandId
	 */
	@Override
	public void serviceStopped(String brandId) {
		LPLog.INSTANCE.d(TAG, "serviceStopped: brandId = " + brandId);
		mConnectionController.serviceStopped(brandId);
	}

	/**
	 * A root method for init the Messaging
	 *
	 * @param brandId
	 */
	private void initBrand(String brandId, LPAuthenticationParams lpAuthenticationParams, @Nullable ConversationViewParams conversationViewParams) {
		LPLog.INSTANCE.d(TAG, "Init brand " + brandId);
		mAccountsController.addNewAccount(brandId);
		mAccountsController.setLPAuthenticationParams(brandId, lpAuthenticationParams);
		if (conversationViewParams != null) {
			setConversationViewParams(conversationViewParams);
		}
		mConnectionController.addNewConnection(brandId);
	}

	/**
	 * Message timer expired, will try reconnect is necessary
	 *
	 * @param brandId
	 */
	public void onMessageTimeout(String brandId) {
		//reconnect automatically if we are in foreground.
		mConnectionController.connect(brandId);
	}

	/**
	 * Send message that contains at least one URL to the AMS server
	 *
	 * @param message
	 */
	@Override
	public void sendMessageWithURL(String targetId, String brandId, String message, String urlToParse, String title, String description, String imageURL, String siteName) {
		// masking message if needed
		MaskedMessage maskedMessage = getMessageValidator(brandId).maskMessage(message, true);
		new SendMessageWithURLCommand(this, targetId, brandId, maskedMessage, urlToParse, title, description, imageURL, siteName).execute();
	}

	/**
	 * Send message to the AMS server
	 *
	 * @param message
	 * @param info a {@link DeliveryStatusUpdateInfo} object with metadata to be sent
	 */
	@Override
	public void sendMessage(String targetId, String brandId, String message, @Nullable DeliveryStatusUpdateInfo info) {
		// masking message if needed
		MaskedMessage maskedMessage = getMessageValidator(brandId).maskMessage(message, true);

		if (maskedMessage != null && !TextUtils.isEmpty(maskedMessage.getServerMessage())){
			new SendMessageCommand(this, targetId, brandId, maskedMessage, info).execute();
		} else {
			LPLog.INSTANCE.i(TAG, "cannot send empty message");
		}
	}

	@Override
	public MaskedMessage getMaskedMessage(String brandId, String message) {
		MessageValidator messageValidator = getMessageValidator(brandId);
		return (messageValidator == null) ? new MaskedMessage(message, message, false, null) : messageValidator.maskMessage(message, true);
	}

	/**
	 * Resend a message
	 *
	 * @param eventId
	 * @param dialogId
	 * @param messageType
	 */
	public int resendMessage(final String eventId, String dialogId, final MessagingChatMessage.MessageType messageType) {
		return resendMessage(eventId, dialogId, NO_FILE_ROW_ID, messageType);
	}
	/**
	 * Resend a message
	 *
	 * @param eventId
	 * @param dialogId
	 */
	@Override
	public int resendMessage(final String eventId, final String dialogId, final long fileRowId, final MessagingChatMessage.MessageType messageType) {
		// Get the brandId
		if (isDialogClosed(dialogId) && !Dialog.TEMP_DIALOG_ID.equals(dialogId)) {
			LPLog.INSTANCE.i(TAG, "Resend message- conversation does not exists or closed.");
			return R.string.lp_resend_failed_conversation_closed;
		}

		if (MessagingChatMessage.MessageType.isConsumerMaskedMessage(messageType)) {
			LPLog.INSTANCE.i(TAG, "Resend message- message is masked, resend is not available.");
			return R.string.lp_resend_failed_masked_message;
		}

		amsMessages.getMessageByEventId(eventId).setPostQueryOnBackground(data -> {
			//if conversation is null, means it was on temporary conversation id - we will resend it..
			Dialog dialog = amsDialogs.queryDialogById(dialogId).executeSynchronously();
			if (data != null) {
				String message = data.getMessage();
				MaskedMessage maskedMessage;
				switch (messageType) {
					case CONSUMER_IMAGE:
					case CONSUMER_DOCUMENT:
						String extension = MimeTypeMap.getSingleton().getExtensionFromMimeType(data.getContentType());
						if (extension != null) {
							reSendFileMessage(FileSharingType.getFileTypeFromExtension(extension), dialog.getBrandId(), dialog.getTargetId(), eventId, message, data.getTimeStamp(), fileRowId);
						}
						break;
					case CONSUMER_VOICE:
						reSendFileMessage(FileSharingType.VOICE, dialog.getBrandId(), dialog.getTargetId(), eventId, message, data.getTimeStamp(), fileRowId);
						break;
					case CONSUMER_URL:
						maskedMessage = getMessageValidator(dialog.getBrandId()).maskMessage(message, true);
						new ResendURLMessageCommand(Messaging.this, eventId, dialog.getTargetId(), dialog.getBrandId(), maskedMessage).execute();
						break;
					case CONSUMER_FORM:
						resendFormSubmissionMessageCommand(eventId, message);
						break;
					default:
						maskedMessage = getMessageValidator(dialog.getBrandId()).maskMessage(message, true);
						ResendMessageCommand resendMessageCommand;
						if (TextUtils.isEmpty(data.getMetadata())) {
							resendMessageCommand = new ResendMessageCommand(Messaging.this, eventId, dialog.getTargetId(), dialog.getBrandId(), maskedMessage);
						} else {
							try {
								JSONArray metadata = new JSONArray(data.getMetadata());
								resendMessageCommand = new ResendMessageCommand(Messaging.this, eventId, dialog.getTargetId(), dialog.getBrandId(), maskedMessage, new DeliveryStatusUpdateInfo(metadata));
							} catch (JSONException e) {
								LPLog.INSTANCE.e(TAG, ERR_0000006D, "Failed to parse JSON", e);
								resendMessageCommand = new ResendMessageCommand(Messaging.this, eventId, dialog.getTargetId(), dialog.getBrandId(), maskedMessage);
							}
						}
						resendMessageCommand.execute();
				}
			}
		}).execute();
		return -1;
	}

	public boolean isDialogClosed(String dialogId) {
		boolean isDialogClosed = false;
		Dialog dialog = amsDialogs.getDialogById(dialogId);
		if (dialog == null || dialog.getState() == DialogState.CLOSE) {
			LPLog.INSTANCE.d(TAG, "isDialogClosed - dialog (dialogId = " + dialogId + ") does not exists or closed. (dialog = " + (dialog == null ? "null" : dialog.getState()) + ")");
			isDialogClosed = true;
		}

		return isDialogClosed;
	}

	private void reSendFileMessage(FileSharingType fileSharingType, String brandId, String targetId, String eventId, String message, long originalMessageTime, long fileRowId) {
		Context applicationContext = Infra.instance.getApplicationContext();
		if (!mUploadUsingService) {
			LPLog.INSTANCE.d(TAG, "reSendImageMessage: re-uploading photo without a service");

			mFileSharingManager.reUploadFile(fileSharingType, brandId, targetId, message, eventId, originalMessageTime,
					fileRowId, new FileSharingManager.FileUploadProgressListener() {
						@Override
						public void onDoneUpload() {
							LPLog.INSTANCE.d(TAG, "onDoneUpload!");
						}
						@Override
						public void onFailedUpload(Throwable exception) {
							LPLog.INSTANCE.d(TAG, "onFailedUpload! ", exception);
						}
					});

		} else {
			LPLog.INSTANCE.d(TAG, "reSendImageMessage: re-uploading photo using a service");

			Intent intent = new Intent(applicationContext, BackgroundActionsService.class);
			intent.putExtra(BackgroundActionsService.EXTRA_ACTION_TYPE, BackgroundActionsService.EXTRA_TYPE_ACTION_REUPLOAD);
			intent.putExtra(BackgroundActionsService.EXTRA_FILE_TYPE, fileSharingType.ordinal());
			intent.putExtra(FileSharingManager.SERVICE_EXTRA_BRAND_ID, brandId);
			intent.putExtra(FileSharingManager.SERVICE_EXTRA_TARGET_ID, targetId);
			intent.putExtra(FileSharingManager.SERVICE_EXTRA_MESSAGE, message);
			intent.putExtra(FileSharingManager.SERVICE_EXTRA_EVENT_ID, eventId);
			intent.putExtra(FileSharingManager.SERVICE_EXTRA_ORIGINAL_MESSAGE_TIME, originalMessageTime);
			intent.putExtra(FileSharingManager.SERVICE_EXTRA_FILE_ROW_ID, fileRowId);
			applicationContext.startService(intent);
		}

	}


	@Override
	public void sendFileMessage(FileSharingType fileSharingType, String brandId, String targetId, String imageUriString, String message, boolean imageFromCamera) {
		Context applicationContext = Infra.instance.getApplicationContext();
		if (mUploadUsingService) {

			LPLog.INSTANCE.d(TAG, "startUploadPhoto: uploading photo using a service");

			Intent intent = new Intent(applicationContext, BackgroundActionsService.class);
			intent.putExtra(BackgroundActionsService.EXTRA_ACTION_TYPE, BackgroundActionsService.EXTRA_ACTION_TYPE_UPLOAD);
			intent.putExtra(BackgroundActionsService.EXTRA_FILE_TYPE, fileSharingType.ordinal());
			intent.putExtra(FileSharingManager.SERVICE_EXTRA_BRAND_ID, brandId);
			intent.putExtra(FileSharingManager.SERVICE_EXTRA_TARGET_ID, targetId);
                /*  for(Uri imageUri : mArrayUri){
                intent.putExtra(BackgroundActionsService.SERVICE_EXTRA_FILE_URI, imageUri.toString());
                getActivity().startService(intent);
                }*/
			intent.putExtra(FileSharingManager.SERVICE_EXTRA_FILE_URI, imageUriString);
			intent.putExtra(FileSharingManager.SERVICE_EXTRA_FILE_CAPTION, message);
			intent.putExtra(FileSharingManager.SERVICE_EXTRA_IMAGE_FROM_CAMERA, imageFromCamera);
			applicationContext.startService(intent);
		} else {
			LPLog.INSTANCE.d(TAG, "startUploadPhoto: uploading photo without a service");

			mFileSharingManager.uploadFile(fileSharingType, brandId, targetId, imageUriString, message, imageFromCamera, new FileSharingManager.FileUploadProgressListener() {
				@Override
				public void onDoneUpload() {
					LPLog.INSTANCE.d(TAG, "onDoneUpload!");
				}

				@Override
				public void onFailedUpload(Throwable exception) {
					LPLog.INSTANCE.d(TAG, "onFailedUpload! ", exception);
					Infra.instance.getApplicationHandler().post( () ->
						Toast.makeText(Infra.instance.getApplicationContext(), R.string.lp_failed_upload_toast_message, Toast.LENGTH_LONG).show());
				}
			});
		}

	}

	@Override
	public void downloadFile(FileSharingType fileSharingType, String brandId, String targetId, String imageSwiftPath, long messageRowId, long fileRowId, String conversationId) {
		Context applicationContext = Infra.instance.getApplicationContext();
		if (mUploadUsingService) {

			LPLog.INSTANCE.d(TAG, "startUploadPhoto: uploading photo using a service");

			Intent intent = new Intent(applicationContext, BackgroundActionsService.class);
			intent.putExtra(BackgroundActionsService.EXTRA_ACTION_TYPE, BackgroundActionsService.EXTRA_TYPE_ACTION_DOWNLOAD);
			intent.putExtra(BackgroundActionsService.EXTRA_FILE_TYPE, fileSharingType.ordinal());
			intent.putExtra(FileSharingManager.SERVICE_EXTRA_BRAND_ID, brandId);
			intent.putExtra(FileSharingManager.SERVICE_EXTRA_TARGET_ID, targetId);
			intent.putExtra(FileSharingManager.SERVICE_EXTRA_FILE_URI, imageSwiftPath);
			intent.putExtra(FileSharingManager.SERVICE_EXTRA_FILE_ROW_ID, fileRowId);
			intent.putExtra(FileSharingManager.SERVICE_EXTRA_MESSAGE_ROW_ID, messageRowId);
			intent.putExtra(FileSharingManager.SERVICE_EXTRA_CONVERSATION_ID, conversationId);
			applicationContext.startService(intent);

		} else {

			mFileSharingManager.downloadFile(fileSharingType, brandId, targetId, imageSwiftPath, messageRowId, fileRowId, conversationId, new FileSharingManager.FileDownloadProgressListener() {

				@Override
				public void onDoneDownload() {
					LPLog.INSTANCE.d(TAG, "onDoneDownload!");
				}

				@Override
				public void onFailedDownload(Throwable exception) {
					LPLog.INSTANCE.d(TAG, "onFailedDownload! ", exception);
					Infra.instance.getApplicationHandler().post(() ->
						Toast.makeText(Infra.instance.getApplicationContext(), R.string.lp_failed_download_toast_message, Toast.LENGTH_LONG).show());
				}
			});
		}
	}


	/**
	 * A wrapper around SendGenerateUploadTokenCommand
	 *
	 */
	@Override
	public void generateUploadToken(String formId, String brandId, final String invitationId) {
		Dialog activeDialog = amsDialogs.getActiveDialog();
		if (activeDialog == null) {
			LPLog.INSTANCE.e(TAG, ERR_0000006E, "Failed to generate upload token, there's no active dialog!");
			MessagingFactory.getInstance().getController().amsMessages.mFormsManager.getForm(invitationId).setFormStatus(Form.FormStatus.ERROR);
			MessagingFactory.getInstance().getController().amsMessages
					.setDeliveryStatusUpdateCommand(MessagingFactory.getInstance()
					.getController().amsMessages
					.mFormsManager.getForm(invitationId),
					DeliveryStatus.ERROR);
		} else {
			new SendGenerateUploadTokenCommand(mAccountsController.getConnectionUrl(brandId),
					formId,
					activeDialog.getDialogId(),
					invitationId,
					new ICallback<Object, Throwable>() {
						@Override
						public void onSuccess(Object value) {

							Messaging.this.amsMessages.mFormsManager.updateForm(invitationId,
									((GeneratedUploadTokenField.Response.Body) value).readOtk,
									((GeneratedUploadTokenField.Response.Body) value).writeOtk);
							Form currentForm = Messaging.this.amsMessages.mFormsManager.getForm(invitationId);

							if (currentForm == null) {
								LPLog.INSTANCE.d(TAG, "no form was found ");
								return;
							}
							String url = currentForm.getOpenFormURL();
							LPLog.INSTANCE.d(TAG, "url = " + url);
							Bundle bundle = new Bundle();
							bundle.putString("url", url);
							bundle.putString("invitation_id", invitationId);
							bundle.putString("form_title", currentForm.getFormTitle());
							LPLog.INSTANCE.d(TAG, "Sending PCI update invitationId = " +
									LPLog.INSTANCE.mask(invitationId) +
									" form title : " +
									LPLog.INSTANCE.mask(currentForm.getFormTitle()));
							LocalBroadcast.sendBroadcast(AmsConversations.BROADCAST_UPDATE_FORM_URL, bundle);
						}

						@Override
						public void onError(Throwable exception) {
							LPLog.INSTANCE.w(TAG, "an error during generating OTK", exception);
							MessagingFactory.getInstance().getController().amsMessages.mFormsManager.getForm(invitationId).setFormStatus(Form.FormStatus.ERROR);
							MessagingFactory.getInstance().getController().amsMessages.setDeliveryStatusUpdateCommand(
									MessagingFactory.getInstance().getController().amsMessages.mFormsManager.getForm(invitationId), DeliveryStatus.ERROR
							);
						}
					}
			).execute();
		}
	}

	/**
	 * Send message to UMS notifying that a form was submitted to the AMS server
	 *
	 * @param invitationId The invitationId of the Form that has been submitted
	 */
	@Override
	public void sendFormSubmissionMessageCommand(final String invitationId) {
		sendFormSubmissionMessageCommand(invitationId, null);
	}

	/**
	 * Sends (or re-sends) a Form Submission Message Command.
	 * @param invitationId The invitationId of the Form that has been submitted
	 * @param existingEventId If this is a re-send, the pre-existing eventId from the first send
	 *                           attempt; otherwise, null.
	 */
	private void sendFormSubmissionMessageCommand(final String invitationId, final String existingEventId) {
		Form currentForm = Messaging.this.amsMessages.mFormsManager.getForm(invitationId);
		if (currentForm == null) {
			LPLog.INSTANCE.e(TAG, ERR_0000006F, "Failed to re-send form ID " + invitationId + " : form not found in FormsManager.");
			return;
		}
		try {
			JSONObject jo = new JSONObject();
			jo.put(SUBMISSION_ID, currentForm.getSubmissionId());
			jo.put(INVITATION_ID, currentForm.getInvitationId());
			amsMessages.mFormsManager.updateForm(currentForm.getInvitationId(), currentForm.getSubmissionId());

			Dialog dialog = amsDialogs.getDialogById(currentForm.getDialogId());
			MaskedMessage maskedMessage = getMessageValidator(dialog.getBrandId()).maskMessage(jo.toString(), false);
			if (maskedMessage == null) {
				currentForm.setFormStatus(Form.FormStatus.ERROR);
				updateMessage(currentForm.getInvitationId(), currentForm.getDialogId(), MessagingChatMessage.MessageType.AGENT_FORM , MessagingChatMessage.MessageState.ERROR);
				return;
			}
			maskedMessage.setServerMessage(jo.toString());

			jo.put(FORM_TITLE, currentForm.getFormTitle());
			maskedMessage.setDbMessage(jo.toString());

			if (!TextUtils.isEmpty(existingEventId)) {
				new ResendFormSubmissionMessageCommand(currentForm, existingEventId, maskedMessage, Messaging.this).execute();
			} else {
				new SendFormSubmissionMessageCommand(currentForm, maskedMessage, Messaging.this).execute();
			}

			updateMessage(currentForm.getInvitationId(), currentForm.getDialogId(), MessagingChatMessage.MessageType.AGENT_FORM , MessagingChatMessage.MessageState.SUBMITTED);
		} catch (JSONException e) {
			LPLog.INSTANCE.e(TAG, ERR_00000072, "JSONException while constructing JSON Object.", e);
		}
	}

	/**
	 * Re-sends a FormSubmissionMessageCommand based on the submission message's eventId and invitationId
	 * @param eventId The eventId of the original SendFormSubmissionMessageCommand
	 * @param messageJsonString The original SendFormSubmissionMessageCommand's message JSON, from
	 *                             which an invitationId can be extracted
	 */
	private void resendFormSubmissionMessageCommand(final String eventId, final String messageJsonString) {
		String invitationId = null;
		try {
			JSONObject messageObject = new JSONObject(messageJsonString);
			invitationId = messageObject.getString("invitationId");
		} catch (JSONException jsonE) {
			LPLog.INSTANCE.e(TAG, ERR_00000070, "Failed to parse message JSON while re-sending Secure Form.", jsonE);
			return;
		}
		if (TextUtils.isEmpty(invitationId)) {
			LPLog.INSTANCE.e(TAG, ERR_00000071, "invitationID was null while re-sending Secure Form.");
			return;
		}

		sendFormSubmissionMessageCommand(invitationId, eventId);
	}

	@Override
	public void updateMessage(final String invitationId, final String dialogId,
							  final MessagingChatMessage.MessageType messageType, final MessagingChatMessage.MessageState messageState) {
		final Form currentForm = Messaging.this.amsMessages.mFormsManager.getForm(invitationId);

		if (currentForm == null){
			LPLog.INSTANCE.i(TAG, "pci update message- form does not exists or closed.");
			return;
		}
		Dialog dialog = amsDialogs.getDialogById(dialogId);
		if (dialog == null || dialog.getState() == DialogState.CLOSE) {
			LPLog.INSTANCE.i(TAG, "pci update message- dialog does not exists or closed.");
			return;
		}

		ArrayList<String> l = new ArrayList<>();
		l.add(currentForm.getEventId());

		LPLog.INSTANCE.i(TAG, "pci update message- with eventID "+ currentForm.getEventId() + " to state: " +messageState);

		amsMessages.updateMessagesState(l, messageState);
	}

	@Override
	public void removeMultipleOlderImages(String brandId) {
		LPLog.INSTANCE.d(TAG, "removeMultipleOlderFiles without service");

		// Remove redundant image files
		mFileSharingManager.removeMultipleOlderFiles(brandId, mMaxNumberOfStoredImageFiles, FileExtensionTypes.getImageExtensionsAsSqlString());

		// Remove redundant voice files
		mFileSharingManager.removeMultipleOlderFiles(brandId, mMaxNumberOfStoredVoiceFiles, FileExtensionTypes.getVoiceExtensionsAsSqlString());

		// Remove redundant Document files
		mFileSharingManager.removeMultipleOlderFiles(brandId, mMaxNumberOfStoredDocumentFiles, FileExtensionTypes.getDocumentExtensionsAsSqlString());

	}

	@Override
	public void registerPusher(String brandId, String appId, String token, PushType pushType, LPAuthenticationParams authenticationParams, ICallback<Void, Exception> registrationCompletedCallback) {
		new RegisterPusherCommand(this, brandId, appId, token, pushType, authenticationParams, registrationCompletedCallback).execute();
	}

	@Override
	public void updateTokenInBackground(String brandId, LPAuthenticationParams authenticationParams) {

		LPLog.INSTANCE.d(TAG, "updateTokenInBackground: Clearing token from account");
		//mAccountsController.setToken(brandId, null);
		connect(brandId, authenticationParams, null, true);
	}

	@Override
	public void unregisterPusherOnLiteLogout(String brandId, String consumerId) {
		new UnregisterPusherCommand(this, brandId, consumerId).execute();
	}

	@Override
	public void unregisterPusher(String brandId, String appId, PushUnregisterType type, ICallback<Void, Exception> unregisteringCompletedCallback, boolean isFullLogout) {
		new UnregisterPusherCommand(this, brandId, appId, type, unregisteringCompletedCallback, isFullLogout).execute();
	}

	@Override
	public void getNumUnreadMessages(String brandId, String appId, final ICallback<Integer, Exception> callback) {
		new GetUnreadMessagesCountCommand(this, brandId, appId, null, new ICallback<Integer, Exception>() {
			@Override
			public void onSuccess(Integer value) {
				callback.onSuccess(value);
			}

			@Override
			public void onError(Exception exception) {
				callback.onError(exception);
			}
		}).execute();
	}

	@Override
	public void getUnreadMessagesCount(String brandId, String appId, LPAuthenticationParams authenticationParams, final ICallback<Integer, Exception> callback) {
		new GetUnreadMessagesCountCommand(this, brandId, appId, authenticationParams, new ICallback<Integer, Exception>() {
			@Override
			public void onSuccess(Integer value) {
				callback.onSuccess(value);
			}

			@Override
			public void onError(Exception exception) {
				callback.onError(exception);
			}
		}).execute();
	}

	@Override
	public void isPusherRegistered(String brandId, String deviceToken, String appId, LPAuthenticationParams authenticationParams, @NonNull ICallback<Boolean, Exception> callback) {
		new GetIsPusherRegisteredCommand(this, brandId, deviceToken, appId, authenticationParams, callback).execute();
	}

	/**
	 * A wrapper around ChangeChatStateCommand
	 *
	 * @param state
	 */
	@Override
	public ActionFailureReason changeChatState(String targetId, String brandId, ChatState state) {
		ActionFailureReason failedReason = getConversationActionFailedReason(targetId, brandId);
		if (failedReason != null) {
			return failedReason;
		}
		new ChangeChatStateCommand(amsDialogs, mAccountsController.getConnectionUrl(brandId), state).execute();
		return null;
	}

	public ActionFailureReason closeCurrentDialog() {
		ActionFailureReason actionFailureReason;
		Dialog activeDialog = amsDialogs.getActiveDialog();
		if (activeDialog == null) {
			LPLog.INSTANCE.e(TAG, ERR_00000073, "There's no active dialog. Aborting from closing dialog");
			actionFailureReason = ActionFailureReason.NOT_ACTIVE;
		} else {
			actionFailureReason = closeDialog(activeDialog.getBrandId());
		}

		return actionFailureReason;
	}

	/**
	 *
	 * Wraps the CloseDialogCommand to execute this operation.
	 */
	@Override
	public ActionFailureReason closeDialog(final String brandId) {
		ActionFailureReason actionFailureReason = getConversationActionFailedReason(brandId, brandId);
		if (actionFailureReason != null) {
			return actionFailureReason;
		} // Socket is open and there's an open conversation....

		final Dialog activeDialog = amsDialogs.getActiveDialog();
		boolean isDialogOpen = activeDialog != null && activeDialog.isOpen();
		if (isDialogOpen && AmsDialogs.isUmsSupportingDialogs()) {
			final CloseDialogCommand closeDialogCommand = new CloseDialogCommand(amsDialogs, activeDialog.getDialogId(), mAccountsController.getConnectionUrl(brandId));
			closeDialogCommand.setCallback(new ICallback<Integer, Exception>() {
				@Override
				public void onSuccess(Integer value) {
					// Cleanup
					closeDialogCommand.setCallback(null);
				}

				@Override
				public void onError(Exception exception) {
					if (Integer.parseInt(exception.getMessage()) == 400) {
						LPLog.INSTANCE.e(TAG, FlowTags.DIALOGS, ERR_00000074, "Failed to close dialog due to an error (with code 400), closing the whole conversation.", exception);
						new ResolveConversationCommand(amsConversations, brandId, mAccountsController.getConnectionUrl(brandId)).execute();
					}
					// Cleanup
					closeDialogCommand.setCallback(null);
				}
			});

			closeDialogCommand.execute();
		} else {
			// There's no open dialog (but the conversation is open) / The current UMS doesn't support multi dialogs
			new ResolveConversationCommand(amsConversations, brandId, mAccountsController.getConnectionUrl(brandId)).execute();
		}

		return null;
	}

	/**
	 * A wrapper around markConversationAsUrgent, it prepopulate it with TTRType.URGENT
	 */
	@Override
	public ActionFailureReason markConversationAsUrgent(String targetId, String brandId) {
		ActionFailureReason failedReason = getConversationActionFailedReason(targetId, brandId);
		if (failedReason != null) {
			return failedReason;
		}

		Dialog activeDialog = amsDialogs.getActiveDialog();
		ActionFailureReason changeTTRFailedReason = getDialogChangeTTRActionFailedReason(activeDialog);
		if (changeTTRFailedReason != null){
			return changeTTRFailedReason;
		}

		new ChangeConversationTTRCommand(amsConversations, targetId, mAccountsController.getConnectionUrl(brandId), TTRType.URGENT).execute();

		return null;
	}

    /**
     * A wrapper around markConversationAsUrgent, it prepopulate it with TTRType.NORMAL
     */
    @Override
    public ActionFailureReason markConversationAsNormal(String targetId, String brandId) {
        ActionFailureReason failedReason = getConversationActionFailedReason(targetId, brandId);
        if (failedReason != null) {
            return failedReason;
        }

		Dialog activeDialog = amsDialogs.getActiveDialog();
		ActionFailureReason changeTTRFailedReason = getDialogChangeTTRActionFailedReason(activeDialog);
        if (changeTTRFailedReason != null) {
            return changeTTRFailedReason;
        }

        new ChangeConversationTTRCommand(amsConversations, targetId, mAccountsController.getConnectionUrl(brandId), TTRType.NORMAL).execute();

        return null;
    }

	@Nullable
	private ActionFailureReason getConversationActionFailedReason(String targetId, String brandId) {
		if (!mConnectionController.isSocketReady(brandId)) {
			LPLog.INSTANCE.d(TAG, "Socket is not open");
			return ActionFailureReason.NO_NETWORK;
		}
		if (!amsConversations.isConversationActive(targetId)) {
			LPLog.INSTANCE.d(TAG, "There's no active dialog");
			return ActionFailureReason.NOT_ACTIVE;
		}

		return null;
	}

	@Nullable
	private ActionFailureReason getDialogChangeTTRActionFailedReason(Dialog activeDialog) {
		ActionFailureReason failureReason = null;

		if (activeDialog == null || !activeDialog.isOpen()) {
			failureReason = ActionFailureReason.NOT_ACTIVE;

		} else if (activeDialog.getDialogType() == DialogType.POST_SURVEY) {
			failureReason = ActionFailureReason.POST_SURVEY_IN_PROGRESS;
		}

		return failureReason;
	}

	public boolean canActiveDialogChangeTTR() {
		Dialog activeDialog = amsDialogs.getActiveDialog();
		ActionFailureReason dialogFailedReason = getDialogChangeTTRActionFailedReason(activeDialog);
		return dialogFailedReason == null;
	}

	@Override
	public void sendCSAT(String brandId, String conversationID, int mStarsNumber, int yesNoValue) {
		new SendCsatCommand(mAccountsController.getConnectionUrl(brandId), conversationID, mStarsNumber, yesNoValue).execute();
	}

	@Override
	public void sendUserProfile(String brandId, UserProfile userProfile) {
		new SendSetUserProfileCommand(this, brandId, userProfile).execute();
	}

	@Override
	public void setCallback(LivePersonCallback listener) {
		mEventsProxy.setCallback(listener);
	}

	@Override
	public void removeCallback() {
		mEventsProxy.removeCallback();
	}

	@Override
	public void checkActiveConversation(String targetId, final ICallback<Boolean, Exception> callback) {
		amsConversations.getActiveConversation(targetId).setPostQueryOnBackground(data -> {
			boolean conversationOpen = false;
			if (data != null) {
				conversationOpen = true;
			}
			callback.onSuccess(conversationOpen);
		}).execute();
	}

	@Override
	public void checkConversationIsMarkedAsUrgent(String targetId, final ICallback<Boolean, Exception> callback) {
		amsConversations.getActiveConversation(targetId).setPostQueryOnBackground(data -> {
			boolean conversationUrgent = false;
			if (data != null) {
				if (data.getConversationTTRType() == TTRType.URGENT &&
						data.isConversationOpen()) {
					conversationUrgent = true;
				}
			}
			callback.onSuccess(conversationUrgent);
		}).execute();
	}

	@Override
	public void checkAgentID(String brandId, final ICallback<AgentData, Exception> callback) {
		amsDialogs.queryActiveDialog(brandId).setPostQueryOnBackground(dialog -> {
			if (dialog != null) {

				Context context = Infra.instance.getApplicationContext();
				if (context == null) {
					callback.onSuccess(null);
					return;
				}
				boolean updateAlways = Configuration.getBoolean(R.bool.send_agent_profile_updates_when_conversation_closed);
				if (updateAlways || dialog.isOpen()) {
					String agentOriginatorId = dialog.getAssignedAgentId();
					if (TextUtils.isEmpty(agentOriginatorId)) {
						callback.onSuccess(null);
					} else {
						amsUsers.getUserById(agentOriginatorId).setPostQueryOnBackground(profile -> {
							if (profile != null) {
								AgentData agentData = new AgentData();
								agentData.mFirstName = profile.getFirstName();
								agentData.mLastName = profile.getLastName();
								agentData.mAvatarURL = profile.getAvatarUrl();
								agentData.mEmployeeId = profile.getDescription();
								agentData.mNickName = profile.getNickname();
								agentData.mAgentId = agentOriginatorId;
								callback.onSuccess(agentData);
								return;
							}
							callback.onSuccess(null);
						}).execute();
					}
				} else {
					callback.onSuccess(null);
				}
			} else {
				callback.onSuccess(null);
			}
		}).execute();
	}

	/**
	 * Clear all messages and conversations of the given targetId.
	 * This method will clear only if there is no open conversation active.
	 *
	 * @param targetId
	 * @return <code>true</code> if messages cleared, <code>false</code> if messages were not cleared (due to open conversation, or no current brand)
	 */
	@Override
	public boolean clearHistory(final String targetId) {

		// Get active conversation
		boolean conversationActive = amsConversations.isConversationActive(targetId);
		if (conversationActive) {

			LPLog.INSTANCE.w(TAG, "clearHistory: There is an open conversation. Cannot clear history");

			return false;
		}

		// Turn on flag clear history action
		MessagingFactory.getInstance().getController().setClearHistoryFlag(targetId, true);

		// Remove all messages
		amsMessages.clearMessagesOfClosedConversations(targetId).setPostQueryOnBackground(data -> LPLog.INSTANCE.d(TAG, "clearHistory: Removed " + data + " messages")).execute();

		// Remove all dialogs
		amsDialogs.clearClosedDialogs(targetId).setPostQueryOnBackground(data -> LPLog.INSTANCE.d(TAG, "clearHistory: Removed " + data + " dialogs")).execute();

		// Remove all conversations
		amsConversations.clearClosedConversations(targetId).setPostQueryOnBackground(data -> LPLog.INSTANCE.d(TAG, "clearHistory: Removed " + data + " conversations")).execute();

		return true;

	}

	@Override
	public void clearAllConversationData(final String targetId) {
		LPLog.INSTANCE.d(TAG, "clearAllConversationData");

		amsMessages.clearAllMessages(targetId).setPostQueryOnBackground(data -> {
			amsMessages.clear();
			amsDialogs.clearAllDialogs(targetId).setPostQueryOnBackground(data1 -> {
				amsDialogs.clear();
				amsConversations.clearAllConversations(targetId).setPostQueryOnBackground(data11 -> amsConversations.clear()).execute();
			}).execute();
		}).execute();
	}

	public String getOriginatorId(String targetId) {
		return amsUsers.getConsumerId(targetId);
	}

	public void onAgentDetailsChanged(final MessagingUserProfile userProfile, boolean isDialogOpen) {
		Context context = Infra.instance.getApplicationContext();
		if (context == null) {
			return;
		}
		boolean updateAlways = Configuration.getBoolean(R.bool.send_agent_profile_updates_when_conversation_closed);
		if (updateAlways || isDialogOpen) {
			//update
			// If user profile is not provided we need to send agent details null
			AgentData agentData = null;
			if (userProfile != null) {
				agentData = new AgentData();
				agentData.mFirstName = userProfile.getFirstName();
				agentData.mLastName = userProfile.getLastName();
				agentData.mAvatarURL = userProfile.getAvatarUrl();
				agentData.mEmployeeId = userProfile.getDescription();
				agentData.mNickName = userProfile.getNickname();
				agentData.mAgentId = userProfile.getOriginatorId();
			}

			mEventsProxy.onAgentDetailsChanged(agentData);
		}
	}

	/**
	 * Shut down Messaging.
	 * @param listener to shut down completion.
	 */
	@Override
	public void shutDown(ShutDownCompletionListener listener) {
		LPLog.INSTANCE.d(TAG, "Shutting down...");
		messagingShutDown(listener);
	}

	/**
	 * Shut down Messaging when upper module (MessagingUI) exists. with it's own ShutDownProcess.
	 * If Messaging is being used without upper module of UI, use {@link #shutDown(ShutDownCompletionListener)}
	 *
	 * ShutDown flow :
	 *
	 * MessagingUi shutdown -> ShutDownCompleted -> Messaging shutdown -> ShutDownCompleted -> Infra shutdown.
	 *
	 * Shut down get MessagingUI's ShutDownProcess, that runs before shutting down Messaging - since
	 * MessagingUI module is based on Messaging module.
	 *
	 * This method call Infra.shutDown with Messaging's ShutDownProcess and will call it before Infra ShutDownProcess.
	 *
	 * Infra's ShutDownProcess will run only when Messaging's ShutDownProcess completed - since
	 * Messaging module is based on Infra module.
	 *
	 * @param process shut down process of upper module (MessagingUI)
	 */
	@Override
	public void shutDown(final ShutDownProcess process) {

		//Sending Infra how to shutdown Messaging module.
		Infra.instance.shutDown(new ShutDownProcess() {

			// get Infra's ShutDownCompletionListener. when Messaging will finish shut down-
			// we will notify Infra with this listener, and it'll run Infra's shutdown process.
			@Override
			public void shutDown(final ShutDownCompletionListener listener) {

				//first, call MessagingUI shut down process.
				process.shutDown(new ShutDownCompletionListener() {

					@Override
					public void shutDownCompleted() {
						//when MessagingUI's completed- run messaging's shutdown.
						LPLog.INSTANCE.d(TAG, "Shutting down...");
						messagingShutDown(listener);
					}

					@Override
					public void shutDownFailed() {

					}

				});
			}
		});
	}

	private void messagingShutDown(final ShutDownCompletionListener listener) {
		ShutDownCompletionListener messagingShutDownListener = new ShutDownCompletionListener() {
			@Override
			public void shutDownCompleted() {
				listener.shutDownCompleted();
			}

			@Override
			public void shutDownFailed() {
				listener.shutDownFailed();
			}
		};
		mConnectionController.shutDown(messagingShutDownListener);
		amsReadController.shutDown();
		amsMessages.shutDown();
		amsDialogs.shutDown();
		amsConversations.shutDown();
	}

	@Override
	public void liteLogout(String brandId, String oldConsumerId, String newConsumerId) {
		mConnectionController.clearLastUpdateTime(brandId);
		amsMessages.shutDown();
		amsDialogs.shutDown();
		amsDialogs.clear();
		amsConversations.shutDown();
		amsConversations.clear();
		amsUsers.clear();
		QuickRepliesMessageHolder.deleteFromSharedPreferences(brandId);
		PreferenceManager.getInstance().remove(KEY_TYPED_TEXT, brandId); //Need to delete here because no access of messaging_ui and ui modules.
		unregisterPusherOnLiteLogout(brandId, oldConsumerId);
		amsMessages.removeAllMessages(brandId);
		Infra.instance.liteLogout();
		//Update consumerId in Database.
		amsUsers.updateConsumerId(brandId, newConsumerId);
	}

	/**
	 * Logout and removing all user data from messaging when upper module (MessagingUI) exists.
	 *
	 * @param context       - app Context
	 * @param initData      - brand id to logout , host app id
	 * @param forceLogOut   When true, log out will not wait for pusher successfully unregistered. When false, log out will wait for unregister pusher result.
	 *                      Potential issue: If fails to unregister pusher, consumer still can receive push notification.
	 * @param logoutProcess - logout process of upper module (MessagingUI)
	 */
	@Override
	public void logout(final Context context, final MessagingInitData initData, final boolean forceLogOut, PushUnregisterType type, final LogoutProcess logoutProcess) {
		Infra.instance.logout(context, initData, new LogoutProcess() {

			@Override
			public void initForLogout() {
				initMessaging(context, initData);
				logoutProcess.initForLogout();
			}

			/**
			 * un-registering pusher before logging out.
			 */
			@Override
			public void preLogout(final PreLogoutCompletionListener listener) {
				logoutProcess.preLogout(new PreLogoutCompletionListener() {
					@Override
					public void preLogoutCompleted() {
						if (forceLogOut) {
							listener.preLogoutCompleted();
						}
						unregisterPusher(initData.getBrandId(), initData.getAppId(), type, new ICallback<Void, Exception>() {
							@Override
							public void onSuccess(Void value) {
								if (!forceLogOut) {
									listener.preLogoutCompleted();
								}
							}

							@Override
							public void onError(Exception exception) {
								if (!forceLogOut) {
									listener.preLogoutFailed(exception);
								}
							}
						}, true);
					}

					@Override
					public void preLogoutFailed(Exception e) {
						listener.preLogoutFailed(e);
					}
				});
			}

			@Override
			public LogoutLivePersonCallBack getLogoutCallback() {
				return logoutProcess.getLogoutCallback();
			}

			@Override
			public void shutDownForLogout(final ShutDownCompletionListener listener) {
				logoutProcess.shutDownForLogout(new ShutDownCompletionListener() {
					@Override
					public void shutDownCompleted() {
						messagingShutDown(listener);
					}

					@Override
					public void shutDownFailed() {

					}
				});
			}

			@Override
			public void logout() {
				//first delete upper level than us.
				logoutProcess.logout();
				clear();
			}
		});
	}

	@Override
	public void clear() {
		amsConversations.clear();
		amsDialogs.clear();
		amsMessages.clear();
		amsUsers.clear();
		mAccountsController.clear();
		mConnectionController.clear();
		mClientProperties.clear();
		FileUtils.deleteFilesSync(Infra.instance.getApplicationContext().getFilesDir());
		mMessageValidator = null;
	}
	@Override
	public boolean isSocketOpen(String brandId) {
		return mConnectionController.isSocketOpen(brandId);
	}

	/**
	 * Check if initialized based on lower module- Infra.
	 * @return if Messaging is Initialized.
	 */
	@Override
	public boolean isInitialized() {
		return Infra.instance.isInitialized();
	}

	/**
	 * Get a comma separated string with all rowIds of messages that are currently uploading images
	 * @return
	 */
	public String getInProgressUploadMessageRowIdsString(){

		LPLog.INSTANCE.d(TAG, "getInProgressUploadMessageRowIdsString: direct call (no service)");
		return mFileSharingManager.getInProgressUploadMessageRowIdsString();

	}


	public FileSharingManager getFileSharingManager() {
		return mFileSharingManager;
	}

	public boolean isEnableStructuredContent() {
		return mEnableStructuredContent;
	}

	/**
	 * Set a pending intent action for the image foreground service notification when clicked
	 *
	 * @param imageServicePendingIntent
	 */
	public void setImageServicePendingIntent(PendingIntent imageServicePendingIntent) {
		mImageServicePendingIntent = imageServicePendingIntent;
	}

	public PendingIntent getImageServicePendingIntent() {
		return mImageServicePendingIntent;
	}

	public Notification.Builder getImageForegroundServiceUploadNotificationBuilder() {
		return mImageForegroundServiceUploadNotificationBuilder;
	}

	public Notification.Builder getImageForegroundServiceDownloadNotificationBuilder() {
		return mImageForegroundServiceDownloadNotificationBuilder;
	}

	public void setImageForegroundServiceUploadNotificationBuilder(Notification.Builder imageForegroundServiceNotificationBuilder) {
		mImageForegroundServiceUploadNotificationBuilder = imageForegroundServiceNotificationBuilder;
	}

	public void setImageForegroundServiceDownloadNotificationBuilder(Notification.Builder imageForegroundServiceNotificationBuilder) {
		mImageForegroundServiceDownloadNotificationBuilder = imageForegroundServiceNotificationBuilder;
	}

	public void sendDeliveryStatusUpdateCommand(String brandId, String dialogId, String conversationId, int sequence , DeliveryStatus deliveryStatus, DeliveryStatusUpdateInfo info){
		new DeliveryStatusUpdateCommand(mAccountsController.getConnectionUrl(brandId), brandId, dialogId, conversationId, sequence, deliveryStatus, info).execute();
	}

	public ConversationViewParams getConversationViewParams() {
		return mConversationViewParams;
	}

	public void setConversationViewParams(ConversationViewParams conversationViewParams) {
        LPLog.INSTANCE.d(TAG, "Setting conversation view params : " + conversationViewParams);
        mConversationViewParams = conversationViewParams;
    }

	public MessagingEventSubscriptionManager getMessagingEventSubscriptionManager() {
		return mMessagingEventSubscriptionManager;
	}

	public LPAudioUtils getAudioUtils() {
		return mAudioUtils;
	}

	@SuppressWarnings("BooleanMethodIsAlwaysInverted") //Currently method is better for understanding.
	public boolean isStillBusyFetching() { return isStillBusyFetching; }

	public void setStillBusyFetching(boolean stillBusyFetching) {
		isStillBusyFetching = stillBusyFetching;
	}

	/**
	 * @return the context of the application
	 */
	public Context getApplicationContext() {
		return Infra.instance.getApplicationContext();
	}

	/**
	 * @return Event manager service instance
	 */
	public EventManagerService getEventManagerService() {
		return Infra.instance.getEventManagerService();
	}

	public void setClearHistoryFlag(String brandId, boolean value) {
		PreferenceManager.getInstance().setBooleanValue(KEY_DID_CLEAR_HISTORY, brandId, value);
	}

	public boolean getClearHistoryFlag(String brandId) {
		return PreferenceManager.getInstance().getBooleanValue(KEY_DID_CLEAR_HISTORY, brandId, false);
	}

	public void removeClearHistoryFlag(String brandId) {
		PreferenceManager.getInstance().remove(KEY_DID_CLEAR_HISTORY, brandId);
	}

	public void resetDBEncryptionService() {
		Infra.instance.resetDBEncryptionService();
	}
}
