package com.liveperson.messaging.controller.connection;

import com.liveperson.infra.Infra;
import com.liveperson.infra.statemachine.interfaces.IState;
import com.liveperson.infra.statemachine.interfaces.IStateMachine;
import com.liveperson.messaging.commands.tasks.BaseAmsAccountConnectionTask;
import com.liveperson.messaging.commands.tasks.CloseConnectionTask;
import com.liveperson.messaging.commands.tasks.CollectSDKConfigurationDataTask;
import com.liveperson.messaging.commands.tasks.ConcurrentTasks;
import com.liveperson.messaging.commands.tasks.ConfigurationFetcherTask;
import com.liveperson.messaging.commands.tasks.CsdsTask;
import com.liveperson.messaging.commands.tasks.DataBaseTask;
import com.liveperson.messaging.commands.tasks.IdpTask;
import com.liveperson.messaging.commands.tasks.LptagTask;
import com.liveperson.messaging.commands.tasks.OpenSocketTask;
import com.liveperson.messaging.commands.tasks.RemoveOlderImagesTask;
import com.liveperson.messaging.commands.tasks.SiteSettingsFetcherTask;
import com.liveperson.messaging.commands.tasks.UnAuthGetEngagementTask;

import java.util.ArrayList;
import java.util.List;

/**
 * Created by nirni on 6/5/17.
 */

public class ConnectionTasksHolder {

	private final BaseAmsAccountConnectionTask mCsdsTask;
	private final BaseAmsAccountConnectionTask mLptagTask;
	private final BaseAmsAccountConnectionTask mConfigurationFetcherTask;
	private final BaseAmsAccountConnectionTask mSiteSettingsFetcherTask;
	private final BaseAmsAccountConnectionTask mIdpTask;
	private final BaseAmsAccountConnectionTask mUnAuthGetEngagementTask;
	private final BaseAmsAccountConnectionTask mRemoveOlderImagesTask;
	private final BaseAmsAccountConnectionTask mDataBaseTask;
	private final BaseAmsAccountConnectionTask mOpenSocketTask;
	private final BaseAmsAccountConnectionTask mCollectSDKConfigurationDataTask;
	private final BaseAmsAccountConnectionTask mCloseConnectionTask;

	private final BaseAmsAccountConnectionTask mConcurrentTasks;

	private final MessagingStateMachineInterface mController;

	private List<BaseAmsAccountConnectionTask> mPrimaryFullConnectingTasks;
	private List<BaseAmsAccountConnectionTask> mPrimaryShortConnectingTasks;
	private List<BaseAmsAccountConnectionTask> mSecondaryConnectingTasks;

	private List<BaseAmsAccountConnectionTask> mDisconnectingTasks;

	public ConnectionTasksHolder(IStateMachine<IState> stateMachine, MessagingStateMachineInterface messagingController, String brandId) {

		mController = messagingController;

		// Get the host version
		String hostVersion = Infra.instance.getHostVersion();

		// Connecting tasks
		mCsdsTask = new CsdsTask(Infra.instance.getApplicationContext(), mController.getAccountsController(), brandId);
		mCsdsTask.setCallback(new DefaultTaskCallbackAmsAccount(stateMachine, mCsdsTask.getClass().getSimpleName()));

		// To be executed concurrently
		mLptagTask = new LptagTask(mController.getAccountsController());

		// To be executed concurrently
		mConfigurationFetcherTask = new ConfigurationFetcherTask(mController.getAccountsController());

		// To be executed concurrently
		mSiteSettingsFetcherTask = new SiteSettingsFetcherTask(mController.getAccountsController());

		// To be executed concurrently
		mIdpTask = new IdpTask(mController.getAccountsController(), mController.getAmsUsers(), hostVersion);

		mUnAuthGetEngagementTask = new UnAuthGetEngagementTask(mController.getAccountsController());
		mUnAuthGetEngagementTask.setCallback(new DefaultTaskCallbackAmsAccount(stateMachine, mUnAuthGetEngagementTask.getClass().getSimpleName()));

		// To be executed concurrently
		mRemoveOlderImagesTask = new RemoveOlderImagesTask(brandId);


		mCollectSDKConfigurationDataTask = new CollectSDKConfigurationDataTask();
		mCollectSDKConfigurationDataTask.setCallback(new DefaultTaskCallbackAmsAccount(stateMachine, mCollectSDKConfigurationDataTask.getClass().getSimpleName()));

		// To be executed concurrently
		mDataBaseTask = new DataBaseTask(mController.getAmsUsers(), mController.getAmsConversations(), mController.getAmsDialogs());

		mOpenSocketTask = new OpenSocketTask(mController.getAccountsController(), mController.getConnectionController());
		mOpenSocketTask.setCallback(new DefaultTaskCallbackAmsAccount(stateMachine, mOpenSocketTask.getClass().getSimpleName()));

		// Disconnecting tasks
		mCloseConnectionTask = new CloseConnectionTask(mController.getAccountsController());
		mCloseConnectionTask.setCallback(new DefaultTaskCallbackAmsAccount(stateMachine, mCloseConnectionTask.getClass().getSimpleName()));

		mConcurrentTasks = new ConcurrentTasks();
		mConcurrentTasks.setCallback(new DefaultTaskCallbackAmsAccount(stateMachine, mConcurrentTasks.getClass().getSimpleName()));

		mPrimaryFullConnectingTasks = constructPrimaryFullConnectionTasks();
		mPrimaryShortConnectingTasks = constructPrimaryShortConnectionTasks();
		mSecondaryConnectingTasks = constructSecondaryConnectionTasks();

		mDisconnectingTasks = constructDisconnectingTasks();
	}

	// primary full happens on CSDS empty or CSDS data change or in connection failure
	public List<BaseAmsAccountConnectionTask> constructPrimaryFullConnectionTasks() {

		List<BaseAmsAccountConnectionTask> connectingTasks = new ArrayList<>();

		connectingTasks.add(mCsdsTask);
		connectingTasks.add(mUnAuthGetEngagementTask);
		connectingTasks.add(mConcurrentTasks);
		connectingTasks.add(mOpenSocketTask);
		//TODO need to update Loggos class before perform mCollectSDKConfigurationDataTask
//		connectingTasks.add(mCollectSDKConfigurationDataTask);

		return connectingTasks;
	}

	// primary short happens whenever primary long doesn't happen , this is different from the secondary cause those are tasks that we must do to open socket
	public List<BaseAmsAccountConnectionTask> constructPrimaryShortConnectionTasks() {

		List<BaseAmsAccountConnectionTask> connectingTasks = new ArrayList<>();

		connectingTasks.add(mUnAuthGetEngagementTask);
		connectingTasks.add(mConcurrentTasks);
		connectingTasks.add(mOpenSocketTask);

		return connectingTasks;
	}

	// after primary short
	public List<BaseAmsAccountConnectionTask> constructSecondaryConnectionTasks() {

		List<BaseAmsAccountConnectionTask> connectingTasks = new ArrayList<>();
		connectingTasks.add(mConcurrentTasks);

		connectingTasks.add(mCsdsTask);
//		TODO need to update Loggos class before perform mCollectSDKConfigurationDataTask
//		connectingTasks.add(mCollectSDKConfigurationDataTask);
		return connectingTasks;
	}

	private List<BaseAmsAccountConnectionTask> constructDisconnectingTasks() {
		List<BaseAmsAccountConnectionTask> disconnectingTasks = new ArrayList<>();
		disconnectingTasks.add(mCloseConnectionTask);
		return disconnectingTasks;
	}

	private void constructConcurrentFullConnectionTasks() {
		mConcurrentTasks.resetConcurrentTasks(); // Important

		mConcurrentTasks.addConcurrentTask(mLptagTask);
		mConcurrentTasks.addConcurrentTask(mConfigurationFetcherTask);
		mConcurrentTasks.addConcurrentTask(mSiteSettingsFetcherTask);
		mConcurrentTasks.addConcurrentTask(mDataBaseTask);
		mConcurrentTasks.addConcurrentTask(mIdpTask);
		mConcurrentTasks.addConcurrentTask(mRemoveOlderImagesTask);
	}

	private void constructPrimaryShortConnectionConcurrentTasks() {
		mConcurrentTasks.resetConcurrentTasks(); // Important

		mConcurrentTasks.addConcurrentTask(mDataBaseTask);
		mConcurrentTasks.addConcurrentTask(mIdpTask);
	}

	private void constructSecondaryConnectionConcurrentTasks() {
		mConcurrentTasks.resetConcurrentTasks(); // Important

		mConcurrentTasks.addConcurrentTask(mConfigurationFetcherTask);
		mConcurrentTasks.addConcurrentTask(mSiteSettingsFetcherTask);
		mConcurrentTasks.addConcurrentTask(mRemoveOlderImagesTask);
		mConcurrentTasks.addConcurrentTask(mLptagTask);
	}

	/**
	 * Return list of primary full connection tasks to be executed.
	 * It is important to create new list of "concurrent connection tasks" each time new tasks are
	 * executed since concurrent task list is static and needs to be cleared and re-constructed.
	 * @return List of connection tasks
	 */
	public List<BaseAmsAccountConnectionTask> getPrimaryFullConnectionTasks() {
		constructConcurrentFullConnectionTasks();
		return mPrimaryFullConnectingTasks;
	}

	/**
	 * Return list of primary short connection tasks to be executed.
	 * It is important to create new list of "concurrent connection tasks" each time new tasks are
	 * executed since concurrent task list is static and needs to be cleared and re-constructed.
	 * @return List of connection tasks
	 */
	public List<BaseAmsAccountConnectionTask> getPrimaryShortConnectionTasks() {
		constructPrimaryShortConnectionConcurrentTasks();
		return mPrimaryShortConnectingTasks;
	}

	/**
	 * Return list of secondary connection tasks to be executed.
	 * It is important to create new list of "concurrent connection tasks" each time new tasks are
	 * executed since concurrent task list is static and needs to be cleared and re-constructed.
	 * @return List of connection tasks
	 */
	public List<BaseAmsAccountConnectionTask> getSecondaryConnectionTasks() {
		constructSecondaryConnectionConcurrentTasks();
		return mSecondaryConnectingTasks;
	}

	public List<BaseAmsAccountConnectionTask> getDisconnectionTasks() {
		return mDisconnectingTasks;
	}
}
