package com.liveperson.messaging.model;

import android.os.Bundle;
import android.text.TextUtils;

import com.liveperson.infra.Infra;
import com.liveperson.infra.auth.LPAuthenticationParams;
import com.liveperson.infra.auth.LPAuthenticationType;
import com.liveperson.infra.controller.DBEncryptionHelper;
import com.liveperson.infra.log.LPLog;
import com.liveperson.infra.managers.ConsumerManager;
import com.liveperson.infra.managers.PreferenceManager;
import com.liveperson.infra.utils.EncryptionVersion;
import com.liveperson.infra.utils.LocalBroadcast;
import com.liveperson.messaging.controller.connection.ConnectionParamsCache;
import com.liveperson.messaging.controller.connection.IConnectionParamsCache;
import com.liveperson.messaging.utils.TokenUtils;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;

/**
 * Created by Ilya Gazman on 11/10/2015.
 * <p/>
 * Accounts model
 */
public class AmsAccount {

    private static final String TAG = "AmsAccount";
    private static final String KEY_CONNECTOR_ID_ENC = "account_connector_id_enc";
    public static final String BROADCAST_KEY_AUTH_COMPLETED_ACTION = "BROADCAST_KEY_AUTH_COMPLETED_ACTION";

    private PreferenceManager mPreferenceManager;
	private IConnectionParamsCache mConnectionParamsCache;

    private String mBrandId;
    private LPAuthenticationParams mLPAuthenticationParams;
    private String mAppId;
    private String mConnectorId;
    private ConsumerManager consumerManager;

    public AmsAccount(String brandId) {
        mPreferenceManager = PreferenceManager.getInstance();
        mBrandId = brandId;

        consumerManager = Infra.instance.getConsumerManager();
		 //restore ConnectorID
        String decryptedConnectorId = mPreferenceManager.getStringValue(KEY_CONNECTOR_ID_ENC, mBrandId, null);
        if (!TextUtils.isEmpty(decryptedConnectorId)){
            mConnectorId = DBEncryptionHelper.decrypt(EncryptionVersion.VERSION_1, decryptedConnectorId);
        }

		mConnectionParamsCache = new ConnectionParamsCache(mBrandId);
        if (consumerManager.getActiveConsumer() != null) {
            LPLog.INSTANCE.d(TAG, "restoring mOriginalConsumerId = " +
                    LPLog.INSTANCE.mask(consumerManager.getActiveConsumer().getOriginalConsumerId())
                    + ", mConnectorId = " + LPLog.INSTANCE.mask(mConnectorId));

            LPLog.INSTANCE.d(TAG, "restoring data: mToken = " + LPLog.INSTANCE.mask(consumerManager.getActiveConsumer().getLpToken()));
        }
	}

    public String getToken() {
        // Get from active consumer prefs
        return consumerManager.getActiveConsumer() != null ? consumerManager.getActiveConsumer().getLpToken() : "";
    }

    public void sendAuthenticationCompletedStatus() {
        Bundle connectionBundle = new Bundle();
        connectionBundle.putString(AmsConnection.BROADCAST_KEY_BRAND_ID, mBrandId);
        LocalBroadcast.sendBroadcast(BROADCAST_KEY_AUTH_COMPLETED_ACTION, connectionBundle);
    }

    public LPAuthenticationParams getLPAuthenticationParams() {
        return mLPAuthenticationParams;
    }

    public void setLPAuthenticationParams(LPAuthenticationParams lpAuthenticationParams) {
        mLPAuthenticationParams = lpAuthenticationParams;
    }

    /**
     * Getting the certificate keys from memory, or from the cached value as a fallback
     * @return return list of certificates
     */
    public List<String> getCertificatePinningKeys() {
        if (mLPAuthenticationParams != null) {
            return mLPAuthenticationParams.getCertificatePinningKeys();
        }
        // Try to get cert pinning keys from active user's cache
        if (consumerManager.getActiveConsumer() != null) {
             LPAuthenticationParams authParams = consumerManager.getActiveConsumer().getLpAuthenticationParams();
             if (authParams != null) {
                 return authParams.getCertificatePinningKeys();
             }
        }
        return new ArrayList<>();
    }


    public boolean isAuthenticated() {
       return mLPAuthenticationParams != null && mLPAuthenticationParams.isAuthenticated();
    }

    public boolean hasFinishedAuthenticating() {
        return consumerManager.isAuthenticated();
    }

    /**
     * Reset consumer's authorization state to be NOT_AUTHENTICATED.
     * Used to reset only If we are authenticating from background.
     */
    public void resetConsumerAuthState() {
        consumerManager.resetAuthState();
    }

	/**
	 * Set all domains on the account storage
	 * @param csdsUrls List of domains to cache
	 * @return true - if one of the relevant values is new, false - non is new
	 */
	public boolean setCSDSMap(HashMap<String, String> csdsUrls) {

		return mConnectionParamsCache.updateCsdsDomains(csdsUrls);
	}

	/**
	 * Check if CSDS data exist in persistent memory
	 * @return true if exist, false if not
	 */
    public boolean isCsdsDataMissing() {

		return !mConnectionParamsCache.isCsdsFilled();
    }

    public String getServiceUrl(String serviceName) {

		if (!mConnectionParamsCache.isCsdsFilled()) {
			return null;
		}

		return mConnectionParamsCache.getServiceDomain(serviceName);
	}

    public void setAppId(String appId) {
        mAppId = appId;
    }

    public String getAppId() {
        return mAppId;
    }

	public IConnectionParamsCache getConnectionParamsCache() {
		return mConnectionParamsCache;
	}

    /**
     * @return true if token exists and expired.
     */
    public boolean isHostAppJWTExpired() {
        String jwt =  mLPAuthenticationParams.getHostAppJWT();

        return TokenUtils.isJwtExpired(jwt);

    }
    /**
     * @return true if token exists and expired.
     */
    public boolean isTokenExpired() {

        String jwt = getToken();

        return TokenUtils.isJwtExpired(jwt);
    }

	/**
	 * Return whether AutoMessages is enabled
	 * @return boolean
	 */
	public boolean isAutoMessagesEnabled(){

		return mConnectionParamsCache.isAutoMessagesFeatureEnabled();
	}

    public void setConnectorId(String connectorId) {
        this.mConnectorId = connectorId;
        String encryptedConnectorId = DBEncryptionHelper.encrypt(EncryptionVersion.VERSION_1, connectorId);
        mPreferenceManager.setStringValue(KEY_CONNECTOR_ID_ENC, mBrandId, encryptedConnectorId);
    }

    public String getConnectorId() {
        return mConnectorId;
    }

    public boolean isInUnAuthMode(){
        LPAuthenticationType authType = getLPAuthenticationParams().getAuthType();
	    return authType == LPAuthenticationType.UN_AUTH;
    }
}
