package com.liveperson.monitoring

import android.content.Context
import android.os.Handler
import android.os.Looper
import android.text.TextUtils
import com.liveperson.infra.InternetConnectionService
import com.liveperson.infra.log.LPMobileLog
import com.liveperson.monitoring.cache.MonitoringParamsCache
import com.liveperson.monitoring.managers.MonitoringRequestManager
import com.liveperson.monitoring.model.LPMonitoringIdentity
import com.liveperson.monitoring.requests.GetEngagementRequest
import com.liveperson.monitoring.requests.SendSdeRequest
import com.liveperson.monitoring.sdk.callbacks.MonitoringErrorType
import com.liveperson.monitoring.sdk.callbacks.SdeCallback
import com.liveperson.sdk.MonitoringInternalInitParams
import com.liveperson.sdk.MonitoringParams
import com.liveperson.sdk.callbacks.EngagementCallback
import java.util.*

/**
 * Created by nirni on 11/21/17.
 */
open class Monitoring : IMonitoring {


    private val TAG = "Monitoring"
    private var appHandler : Handler? = null
    private var requestManager : MonitoringRequestManager? = null

    // Indicates that Monitoring was initialized
    var initialized = false

    var paramsCache : MonitoringParamsCache? = null
    var brandId : String? = null
    var applicationContext : Context? = null
    /**
     *
     */
    fun init(monitoringInternalInitParams: MonitoringInternalInitParams): Boolean {

        LPMobileLog.d(TAG, "init: Monitoring module version = " + com.liveperson.lp_monitoring_sdk.BuildConfig.VERSION_NAME)

        LPMobileLog.d(TAG, "Initializing Monitoring SDK. MonitoringInternalInitParams: $monitoringInternalInitParams")

        appHandler = Handler(Looper.getMainLooper())
        applicationContext = monitoringInternalInitParams.applicationContext
        brandId = monitoringInternalInitParams.accountId
        if (TextUtils.isEmpty(brandId)) {
            LPMobileLog.w(TAG, "BrandId is empty. Aborting")
            return false
        }
        val lBrandId = brandId
        lBrandId ?.let { paramsCache = MonitoringParamsCache(lBrandId)}
        // Fail if paramsCache is not initialized properly
        paramsCache ?: return false
        val lParamsCache = paramsCache
        paramsCache?.appInstallId = monitoringInternalInitParams.appInstallId

        lParamsCache?.let {requestManager = MonitoringRequestManager(brandId, lParamsCache)}
        initialized = true

        return true
    }

    /**
     *
     */
    override fun logout(context: Context): Boolean {

        LPMobileLog.d(TAG, "Logging out...")

        if (!initialized) {
            LPMobileLog.w(TAG, "Not initialized")
            return true
        }

        paramsCache?.clear()
        paramsCache = null
        brandId = null
        applicationContext = null

        initialized = false
        return true
    }

    /**
     *
     */
    override fun isInitialized() : Boolean {
        return initialized
    }

    /**
     *
     */
    override fun  getEngagement(context: Context, identities: List<LPMonitoringIdentity>?, monitoringParams: MonitoringParams?, callback: EngagementCallback) {

        LPMobileLog.d(TAG, "GetEngagement. LPMonitoringIdentity: ${identities?.get(0)}, SDE: $monitoringParams")

        if (!initialized) {
            LPMobileLog.w(TAG, "Not initialized")
            callback.onError(MonitoringErrorType.NOT_INITIALIZED, null)
            return
        }

        if (!InternetConnectionService.isNetworkAvailable()) {
            LPMobileLog.d(TAG, "No network. calling callback.onError")

            callback.onError(MonitoringErrorType.NO_NETWORK, null)
            return
        }

        val getEngagementRequest = GetEngagementRequest(context, identities, monitoringParams, callback)
        requestManager?.csdsDependantRequest(context, getEngagementRequest, callback)

    }

    /**
     *
     */
    override fun sendSde(context: Context, identities: List<LPMonitoringIdentity>?, monitoringParams: MonitoringParams?, callback: SdeCallback) {

        LPMobileLog.d(TAG, "sendSde. ConsumerId: ${identities}, SDE: $monitoringParams")

        if (!initialized) {
            LPMobileLog.w(TAG, "Not initialized")
            callback.onError(MonitoringErrorType.NOT_INITIALIZED, null)
            return
        }

        if (identities == null || identities.isEmpty()) {
            callback.onError(MonitoringErrorType.PARAMETER_MISSING, Exception("ConsumerId is mandatory"))
            return
        }

        if (!InternetConnectionService.isNetworkAvailable()) {
            LPMobileLog.d(TAG, "No network. calling callback.onError")
            callback.onError(MonitoringErrorType.NO_NETWORK, null)
            return
        }


        val sendSdeRequest = SendSdeRequest(context, identities, monitoringParams, callback)
        requestManager?.csdsDependantRequest(context, sendSdeRequest, callback)
    }

    /**
     * Post the given Runnable on the main thread
     */
    fun postOnMainThread(runnable: Runnable) {
        appHandler?.post(runnable)
    }
}