package com.liveperson.monitoring.managers

import android.content.Context
import com.liveperson.infra.log.LPMobileLog
import com.liveperson.monitoring.Command
import com.liveperson.monitoring.MonitoringFactory
import com.liveperson.monitoring.cache.MonitoringParamsCache
import com.liveperson.monitoring.requests.CSDSRequest
import com.liveperson.monitoring.sdk.callbacks.MonitoringErrorType
import com.liveperson.monitoring.sdk.callbacks.IMonitoringCallback
import com.liveperson.monitoring.sdk.responses.LPSdeResponse
import com.liveperson.monitoring.utils.DomainBuilder

/**
 * Created by nirni on 12/7/17.
 */
class MonitoringRequestManager(val brandId : String?, val paramsCache: MonitoringParamsCache) {

    private val TAG = "MonitoringRequestManager"

    val twentyFourHoursInMillis = 1000 * 60 *60 * 24 // TODO: 12/4/17 Get from resources

    /**
     * This sends the given request, but first checks whether the CSDS needs to be updated. If so it first request CSDS data
     * and after data receives sends the given request.
     */
    fun csdsDependantRequest(context: Context, request : Command, callback: IMonitoringCallback<out LPSdeResponse, Exception>) {

        val now = System.currentTimeMillis()
        val expireTime = paramsCache.lastCsdsUpdateTimestamp + twentyFourHoursInMillis

        LPMobileLog.d(TAG, "Current time: $now, last CSDS update time: ${paramsCache.lastCsdsUpdateTimestamp}, expire time: ${expireTime}")

        // If no CSDS data is available, or the data is old
        if (!paramsCache.isCsdsFilled() || (now > expireTime)) {

            LPMobileLog.d(TAG, "CSDS data does not exist or expired. Call CSDS")

            // Get domain for the given brandId
            val csdsDomain = DomainBuilder.getCsdsDomain(context, brandId)

            CSDSRequest(csdsDomain, brandId, object : IMonitoringCallback<HashMap<String, String>, Exception> {

                override fun onSuccess(csdsDomainMap: HashMap<String, String>?) {

                    if (csdsDomainMap != null) {
                        LPMobileLog.i(TAG, csdsDomainMap.toString())

                        // Update the CSDS data to cache
                        paramsCache.updateCsdsDomains(csdsDomainMap)

                        // Init Loggos
// TODO: 2/5/18 Check that indeed not needed
//                        brandId ?.let {initLoggos(brandId)}

                        // Request
                        LPMobileLog.d(TAG, "Calling the next request...")
                        request.execute()

                    } else {
                        LPMobileLog.e(TAG, "csdsDependantRequest: Received empty CSDS domain map")
                        MonitoringFactory.monitoring.postOnMainThread(Runnable {
                            callback.onError(MonitoringErrorType.CSDS_ERROR, null)
                        })

                    }
                }

                override fun onError(errorType: MonitoringErrorType, e: Exception?) {

                    LPMobileLog.e(TAG, e)
                    MonitoringFactory.monitoring.postOnMainThread(Runnable {
                        callback.onError(MonitoringErrorType.CSDS_ERROR, e)
                    })
                }

            }).execute()
        } else { // No update for CSDS is needed

            LPMobileLog.d(TAG, "CSDS data exist in cache. Proceed to request")
            request.execute()
        }
    }
}