package com.liveperson.monitoring.requests

import android.content.Context
import android.content.pm.PackageManager
import com.liveperson.infra.network.http.request.HttpPostRequest
import com.liveperson.infra.network.http.request.HttpRequest
import com.liveperson.lp_monitoring_sdk.R
import com.liveperson.monitoring.MonitoringFactory
import com.liveperson.monitoring.model.EngagementWithSession
import com.liveperson.monitoring.model.LPMonitoringIdentity
import com.liveperson.monitoring.requests.BaseMonitoringRequest
import com.liveperson.monitoring.sdk.callbacks.MonitoringErrorType
import com.liveperson.monitoring.sdk.responses.LPEngagementResponse
import com.liveperson.sdk.MonitoringParams
import com.liveperson.sdk.callbacks.EngagementCallback
import org.json.JSONObject
import java.lang.IllegalArgumentException


/**
 * Created by nirni on 11/22/17.
 *
 * This is the GetEngagment HTTP request. This request an engagement from the monitoring server (Shark) and return
 * it using the provided callback.
 * The request's body is built using the given MonitoringInitParams object
 */
class GetEngagementRequest(context: Context, identities: List<LPMonitoringIdentity>?, monitoringParams: MonitoringParams?, val callback: EngagementCallback) :
            BaseMonitoringRequest(context, identities, monitoringParams){

    private val TAG = "GetEngagementRequest"

    private val OS_ANDROID = "ANDROID"
    private val DEVICE_FAMILY_MOBILE = "MOBILE"

    // Client properties keys
    private val KEY_OS = "os"
    private val KEY_OS_VERSION = "osVersion"
    private val KEY_APP_VERSION = "appVersion"
    private val KEY_DEVICE_FAMILY = "deviceFamily"
    private val KEY_CLIENT_PROPERTIES = "clientProperties"

    override fun getRequest(): HttpRequest = HttpPostRequest(buildRequestUrl())

    override fun getRequestUrl(): String = context.resources.getString(R.string.get_engagement_url)

    /**
     * Handle the response
     */
    override fun handleResponse(response: String) {

        // Build the engagement from the request
        val engagementWithSession = EngagementWithSession(response)

        // Save the sessionId and/or visitorId if returned in the engagement
        engagementWithSession.let { paramsCache?.sessionId = it?.sessionId }
        engagementWithSession.let { paramsCache?.visitorId = it?.visitorId }

        try {
            engagementWithSession.engagementDetailsList!!.get(0)
                .let { paramsCache?.connectorId = it!!.connectorId }
        } catch (e: NullPointerException) {
            callErrorCallback(MonitoringErrorType.PARAMETER_MISSING,
                IllegalArgumentException("No connecterid because EngagementDetails is null"))
            return
        }

        // Post onSuccess to hostapp
        MonitoringFactory.monitoring.postOnMainThread(Runnable {

            callback.onSuccess(LPEngagementResponse(engagementWithSession))
        })
    }

    /**
     *
     */
    override fun callErrorCallback(monitoringErrorType: MonitoringErrorType,exception : Exception?) {

        MonitoringFactory.monitoring.postOnMainThread(Runnable {
            callback.onError(monitoringErrorType, exception)
        })
    }

    /**
     *
     */
    override fun buildRequestBody(identities: List<LPMonitoringIdentity>?, monitoringParams: MonitoringParams?): JSONObject {
        val getEngagementBody = super.buildRequestBody(identities, monitoringParams)
        getEngagementBody.put(KEY_CLIENT_PROPERTIES, buildClientProperties())

        return getEngagementBody
    }

    /**
     * Build the client properties data
     */
    private fun buildClientProperties(): JSONObject {

        val clientProperties = JSONObject();

        clientProperties.put(KEY_OS, OS_ANDROID)
        clientProperties.put(KEY_OS_VERSION, android.os.Build.VERSION.RELEASE)
        val version: String? = try {
            val pInfo = context.getPackageManager().getPackageInfo(context.getPackageName(), 0)
            pInfo.versionName
        } catch (e: PackageManager.NameNotFoundException) {
            "Unavailable"
        }

        clientProperties.put(KEY_APP_VERSION, version)
        clientProperties.put(KEY_DEVICE_FAMILY, DEVICE_FAMILY_MOBILE)

        return clientProperties
    }

}